package ru.yandex.infra.stage.podspecs.patcher.defaults;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.controller.util.ProtobufEnumWrappers;
import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.dto.NetworkDefaults;
import ru.yandex.infra.stage.dto.SidecarVolumeSettings;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.SidecarDiskVolumeDescription;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.infra.stage.yp.Attributes;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.Enums;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.EResolvConf;
import ru.yandex.yp.client.pods.EWorkloadTargetState;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TMutableWorkload;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TRootfsVolume;
import ru.yandex.yp.client.pods.TVolume;
import ru.yandex.yt.ytree.TAttribute;
import ru.yandex.yt.ytree.TAttributeDictionary;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.DEFAULT_BOX_SPECIFIC_TYPE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.GIGABYTE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.USED_BY_INFRA_LABEL;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addBoxIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addSidecarDiskVolumeRequest;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.allocationAccordingToDiskIsolationLabel;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.calculateSidecarDiskVolumeDescription;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.hasLabel;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.ip6Request;


// Set defaults required by lower-level infrastructure (node and pod agents)
public abstract class DefaultsPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    @VisibleForTesting
    static final String ROOT_BOX_ID = "root_box";
    @VisibleForTesting
    static final String SOX_FILTER = "[/labels/extras/safe_for_sox] = %true";
    public static final String BACKBONE_VLAN = "backbone";
    public static final String FASTBONE_VLAN = "fastbone";
    public static final String ALLOWED_SSH_KEY_SET_LABEL_KEY = "allowed_ssh_key_set";
    private static final String BOXES_SUBNET_LABEL_KEY = "id";
    private static final byte[] BOXES_SUBNET_LABEL_VALUE = new YTreeBuilder().value("boxes_subnet").build().toBinary();
    public static final TAttribute BOXES_SUBNET_LABEL =
            Attributes.binaryAttribute(BOXES_SUBNET_LABEL_KEY, BOXES_SUBNET_LABEL_VALUE);
    private static final long POD_AGENT_ALLOCATION_CAPACITY = 1 * GIGABYTE;

    private final ResourceSupplier podAgentBinarySupplier;
    private final ResourceSupplier podAgentLayerSupplier;
    private final Optional<String> podAgentAllocationId;
    private final Optional<List<String>> allSidecarDiskAllocationIds;
    private final boolean patchBoxSpecificType;
    private final long releaseGetterTimeoutSeconds;
    private final boolean placeBinaryRevisionToPodAgentMeta;

    public DefaultsPatcherV1Base(DefaultsPatcherV1Context context) {
        this(context.getDefaultPodAgentBinarySupplier(),
                context.getDefaultPodAgentLayerSupplier(),
                context.getPodAgentAllocationId(),
                context.getAllSidecarDiskAllocationIds(),
                context.isPatchBoxSpecificType(),
                context.getReleaseGetterTimeoutSeconds(),
                context.placeBinaryRevisionToPodAgentMeta());
    }

    public DefaultsPatcherV1Base(ResourceSupplier podAgentBinarySupplier,
                                 ResourceSupplier podAgentLayerSupplier,
                                 Optional<String> podAgentAllocationId,
                                 Optional<List<String>> allSidecarDiskAllocationIds,
                                 boolean patchBoxSpecificType,
                                 long releaseGetterTimeoutSeconds,
                                 boolean placeBinaryRevisionToPodAgentMeta) {
        this.podAgentBinarySupplier = podAgentBinarySupplier;
        this.podAgentLayerSupplier = podAgentLayerSupplier;
        this.podAgentAllocationId = podAgentAllocationId;
        this.allSidecarDiskAllocationIds = allSidecarDiskAllocationIds;
        this.patchBoxSpecificType = patchBoxSpecificType;
        this.releaseGetterTimeoutSeconds = releaseGetterTimeoutSeconds;
        this.placeBinaryRevisionToPodAgentMeta = placeBinaryRevisionToPodAgentMeta;
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        patchHostNameKind(podTemplateSpecBuilder.getSpecBuilder());
        patchPodSpec(podTemplateSpecBuilder, context, getPodAgentBinarySupplier(context), getPodAgentLayerSupplier(context));
        enableExtraRouters(podTemplateSpecBuilder.getSpecBuilder());
        patchAllowedSshKeySet(podTemplateSpecBuilder.getSpecBuilder(), context.getStageContext().getLabels());
    }

    private void patchAllowedSshKeySet(DataModel.TPodSpec.Builder specBuilder, Map<String, YTreeNode> labels) {
        if (!specBuilder.hasAllowedSshKeySet()) {
            final YTreeNode node = labels.get(ALLOWED_SSH_KEY_SET_LABEL_KEY);
            Enums.EPodSshKeySet allowedSet = ProtobufEnumWrappers.POD_SSH_KEY_SET.tryParseEnumValue(node);
            if (allowedSet != null) {
                specBuilder.setAllowedSshKeySet(allowedSet);
            }
        }
    }

    private void enableExtraRouters(DataModel.TPodSpec.Builder specBuilder) {
        TPodAgentSpec spec = specBuilder.getPodAgentPayload().getSpec();
        for (TBox tBox : spec.getBoxesList()) {
            if (tBox.getResolvConf().equals(EResolvConf.EResolvConf_NAT64) || tBox.getResolvConf().equals(EResolvConf.EResolvConf_NAT64_LOCAL)) {
                specBuilder.getNetworkSettingsBuilder().setExtraRoutes(true);
                break;
            }
        }
    }


    protected void patchHostNameKind(DataModel.TPodSpec.Builder specBuilder) {
        if (!specBuilder.hasHostNameKind()) {
            specBuilder.setHostNameKind(Enums.EPodHostNameKind.PHNK_PERSISTENT);
        }
    }

    protected ResourceWithMeta getPodAgentBinarySupplier(DeployUnitContext context) {
        return PatcherUtils.getResource(podAgentBinarySupplier, context.getSpec().getPodAgentResourceInfo(),
                releaseGetterTimeoutSeconds);
    }

    protected ResourceWithMeta getPodAgentLayerSupplier(DeployUnitContext context) {
        return PatcherUtils.getResource(podAgentLayerSupplier, context.getSpec().getPodAgentLayerResourceInfo(),
                releaseGetterTimeoutSeconds);
    }

    protected void patchPodSpec(TPodTemplateSpec.Builder podTemplateSpec, DeployUnitContext context,
                                ResourceWithMeta podAgentBinarySupplier, ResourceWithMeta podAgentLayerSupplier) {
        DataModel.TPodSpec.Builder podSpec = podTemplateSpec.getSpecBuilder();

        patchNodeFilterWithSox(podSpec, context.getSpec().isSoxService());

        patchNetworks(podSpec, context.getSpec().getNetworkDefaults());

        patchPodAgentMeta(podSpec.getPodAgentPayloadBuilder(), podAgentBinarySupplier, podAgentLayerSupplier,
                placeBinaryRevisionToPodAgentMeta);

        if (patchBoxSpecificType) {
            patchEmptyBoxSpecificType(podSpec.getPodAgentPayloadBuilder().getSpecBuilder());
        }

        Optional<String> initialRootAllocation = patchVirtualDisks(context.getStageContext().getStageId(),
                context.getSpec().getDetails().extractPodAgentConfig().getSidecarVolumeSettings(),
                allSidecarDiskAllocationIds,
                podTemplateSpec, podAgentAllocationId);
        patchPodAgentSpec(podSpec.getPodAgentPayloadBuilder().getSpecBuilder(), initialRootAllocation);

        patchAllowedSshKeySet(podTemplateSpec.getSpecBuilder(), context.getStageContext().getLabels());
    }

    @VisibleForTesting
    static void patchPodAgentMeta(DataModel.TPodSpec.TPodAgentPayload.Builder agentPayloadBuilder,
                                  ResourceWithMeta podAgentBinarySupplier,
                                  ResourceWithMeta podAgentLayerSupplier,
                                  boolean placeBinaryRevisionToPodAgentMeta) {
        //no all meta in spec
        if (!agentPayloadBuilder.hasMeta()) {
            DownloadableResource podAgentBinary = podAgentBinarySupplier.getResource();
            DownloadableResource podAgentBaseLayer = podAgentLayerSupplier.getResource();

            DataModel.TPodSpec.TPodAgentDeploymentMeta.Builder metaBuilder =
                    DataModel.TPodSpec.TPodAgentDeploymentMeta.newBuilder()
                            .setUrl(podAgentBinary.getUrl())
                            .setChecksum(podAgentBinary.getChecksum().toAgentFormat())
                            .addLayers(DataModel.TPodSpec.TPodAgentDeploymentMeta.TLayer.newBuilder()
                                    .setUrl(podAgentBaseLayer.getUrl())
                                    .setChecksum(podAgentBaseLayer.getChecksum().toAgentFormat()));

            if (placeBinaryRevisionToPodAgentMeta) {
                podAgentBinarySupplier.getMeta().ifPresent(m -> metaBuilder.setBinaryRevision(m.getResourceId()));
            }

            agentPayloadBuilder.setMeta(metaBuilder);
        } else {
            DataModel.TPodSpec.TPodAgentDeploymentMeta.Builder metaBuilder = agentPayloadBuilder.getMetaBuilder();
            //no pod agent binary in spec
            if (!metaBuilder.hasUrl()) {
                DownloadableResource podAgentBinary = podAgentBinarySupplier.getResource();
                metaBuilder.setUrl(podAgentBinary.getUrl()).setChecksum(podAgentBinary.getChecksum().toAgentFormat());

                if (placeBinaryRevisionToPodAgentMeta) {
                    podAgentBinarySupplier.getMeta().ifPresent(m -> metaBuilder.setBinaryRevision(m.getResourceId()));
                }
            }
            //no pod agent base layer in spec
            if (0 == metaBuilder.getLayersCount()) {
                DownloadableResource podAgentBaseLayer = podAgentLayerSupplier.getResource();
                metaBuilder.addLayers(DataModel.TPodSpec.TPodAgentDeploymentMeta.TLayer.newBuilder()
                        .setUrl(podAgentBaseLayer.getUrl())
                        .setChecksum(podAgentBaseLayer.getChecksum().toAgentFormat()));
            }
        }
    }

    @VisibleForTesting
    static void patchPodAgentSpec(TPodAgentSpec.Builder agentSpec, Optional<String> initialRootAllocation) {
        if (agentSpec.getBoxesCount() == 0) {
            TBox.Builder rootBox = TBox.newBuilder()
                    .setId(ROOT_BOX_ID)
                    .setRootfs(TRootfsVolume.newBuilder()
                            .addAllLayerRefs(agentSpec.getResources().getLayersList().stream()
                                    .map(TLayer::getId)
                                    .collect(Collectors.toList())
                            )
                            .build());

            addBoxIfAbsent(agentSpec, rootBox, ROOT_BOX_ID, initialRootAllocation);
            agentSpec.getWorkloadsBuilderList()
                    .forEach(builder -> builder.setBoxRef(ROOT_BOX_ID));
        }

        if (agentSpec.getMutableWorkloadsCount() == 0) {
            agentSpec.addAllMutableWorkloads(agentSpec.getWorkloadsList().stream().map(workload ->
                    TMutableWorkload.newBuilder()
                            .setWorkloadRef(workload.getId())
                            .setTargetState(EWorkloadTargetState.EWorkloadTarget_ACTIVE)
                            .build()
            ).collect(Collectors.toList()));
        }
    }

    protected void patchNodeFilterWithSox(DataModel.TPodSpec.Builder podSpec, boolean isSoxService) {
        patchNodeFilterWithSoxFromV4ToLast(podSpec, isSoxService);
    }

    protected static void patchNodeFilterWithSoxFromV1ToV3(DataModel.TPodSpec.Builder podSpec, boolean isSoxService) {
        if (isSoxService) {
            podSpec.setNodeFilter(SOX_FILTER);
        }
    }

    private static void patchNodeFilterWithSoxFromV4ToLast(DataModel.TPodSpec.Builder podSpec, boolean isSoxService) {
        if (!isSoxService) {
            return;
        }
        var nodeFilter = podSpec.getNodeFilter();
        if (StringUtils.isNotBlank(nodeFilter) && !nodeFilter.contains(SOX_FILTER)) {
            podSpec.setNodeFilter(String.format("%s AND %s", SOX_FILTER, nodeFilter));
        } else {
            podSpec.setNodeFilter(SOX_FILTER);
        }
    }

    @VisibleForTesting
    void patchNetworks(DataModel.TPodSpec.Builder podSpec, NetworkDefaults networkDefaults) {
        if (networkDefaults.isOverrideIp6AddressRequests()) {
            podSpec.clearIp6AddressRequests();
        }
        if (podSpec.getIp6AddressRequestsCount() == 0) {
            podSpec.addIp6AddressRequests(ip6Request(networkDefaults.getNetworkId(), BACKBONE_VLAN));
            podSpec.addIp6AddressRequests(ip6Request(networkDefaults.getNetworkId(), FASTBONE_VLAN));
        }
        if (networkDefaults.isOverrideIp6SubnetRequests()) {
            podSpec.clearIp6SubnetRequests();
        }
        if (podSpec.getIp6SubnetRequestsList().stream()
                .noneMatch(request -> hasLabel(request.getLabels(), BOXES_SUBNET_LABEL))) {
            podSpec.addIp6SubnetRequests(DataModel.TPodSpec.TIP6SubnetRequest.newBuilder()
                    .setNetworkId(networkDefaults.getNetworkId())
                    .setVlanId(BACKBONE_VLAN)
                    .setLabels(TAttributeDictionary.newBuilder()
                            .addAttributes(BOXES_SUBNET_LABEL))
                    .build());
        }
        networkDefaults.getIp4AddressPoolId().ifPresent(poolId -> podSpec.getIp6AddressRequestsBuilderList()
                .forEach(request -> request.setIp4AddressPoolId(poolId)));
        if (!networkDefaults.getVirtualServiceIds().isEmpty()) {
            podSpec.getIp6AddressRequestsBuilderList()
                    .forEach(request -> request.addAllVirtualServiceIds(networkDefaults.getVirtualServiceIds()));
        }
    }

    //returns initial root allocation if allocations used
    @VisibleForTesting
    static Optional<String> patchVirtualDisks(String stageId, Optional<SidecarVolumeSettings> sidecarVolumeSettings,
                                              Optional<List<String>> sidecarDiskAllocationIds,
                                              TPodTemplateSpec.Builder podTemplateSpec,
                                              Optional<String> podAgentAllocationId) {
        DataModel.TPodSpec.Builder podSpec = podTemplateSpec.getSpecBuilder();

        return allocationAccordingToDiskIsolationLabel(podTemplateSpec, podAgentAllocationId).map(agentAllocationId -> {
            List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> allocationsExceptPodAgent =
                    getAllAllocationsExceptPodAgent(podSpec, podAgentAllocationId);

            String initRootAllocationId = addPodAgentDiskVolumeRequest(stageId, sidecarVolumeSettings, podSpec,
                    allocationsExceptPodAgent, sidecarDiskAllocationIds.orElseThrow(), agentAllocationId);
            patchEmptyVirtualDisksRefs(podSpec.getPodAgentPayloadBuilder().getSpecBuilder(), allocationsExceptPodAgent);
            return initRootAllocationId;
        });
    }

    private static String addPodAgentDiskVolumeRequest(String stageId,
                                                       Optional<SidecarVolumeSettings> sidecarVolumeSettings,
                                                       DataModel.TPodSpec.Builder podSpec,
                                                       List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> allocationsExceptPodAgent,
                                                       List<String> sidecarDiskAllocationIds,
                                                       String podAgentAllocationId) {

        Optional<DataModel.TPodSpec.TDiskVolumeRequest.Builder> prevRootDiskVolumeRequest =
                allocationsExceptPodAgent.stream()
                        .filter(r -> hasLabel(r.getLabels(), USED_BY_INFRA_LABEL))
                        .findFirst();

        SidecarDiskVolumeDescription sidecarDiskVolumeDescription = calculateSidecarDiskVolumeDescription(stageId,
                podSpec, podAgentAllocationId, sidecarVolumeSettings, sidecarDiskAllocationIds);
        Optional<DataModel.TPodSpec.TDiskVolumeRequest.Builder> addedPodAgentDiskVolumeRequest =
                addSidecarDiskVolumeRequest(podSpec, POD_AGENT_ALLOCATION_CAPACITY, sidecarDiskVolumeDescription);

        addedPodAgentDiskVolumeRequest.ifPresent(DefaultsPatcherV1Base::addUsedByInfraLabel);
        prevRootDiskVolumeRequest.ifPresent(DefaultsPatcherV1Base::removeUsedByInfraLabel);

        return prevRootDiskVolumeRequest.orElseThrow().getId();
    }

    @VisibleForTesting
    static void patchEmptyVirtualDisksRefs(TPodAgentSpec.Builder agentSpec,
                                           List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> allocationsExceptPodAgent) {
        if (allocationsExceptPodAgent.size() == 1) {
            String defaultAllocationId = allocationsExceptPodAgent.get(0).getId();

            addAllocationRef(agentSpec.getBoxesBuilderList(),
                    TBox.Builder::getVirtualDiskIdRef,
                    (b) -> b.setVirtualDiskIdRef(defaultAllocationId));

            addAllocationRef(agentSpec.getVolumesBuilderList(),
                    TVolume.Builder::getVirtualDiskIdRef,
                    (b) -> b.setVirtualDiskIdRef(defaultAllocationId));

            addAllocationRef(agentSpec.getResourcesBuilder().getLayersBuilderList(),
                    TLayer.Builder::getVirtualDiskIdRef,
                    (b) -> b.setVirtualDiskIdRef(defaultAllocationId));

            addAllocationRef(agentSpec.getResourcesBuilder().getStaticResourcesBuilderList(),
                    TResource.Builder::getVirtualDiskIdRef,
                    (b) -> b.setVirtualDiskIdRef(defaultAllocationId));
        }
    }

    @VisibleForTesting
    static void patchEmptyBoxSpecificType(TPodAgentSpec.Builder agentSpec) {
        agentSpec.getBoxesBuilderList().stream()
                .filter(builder -> builder.getSpecificType().isEmpty())
                .forEach(builder -> builder.setSpecificType(DEFAULT_BOX_SPECIFIC_TYPE));
    }

    private static List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> getAllAllocationsExceptPodAgent(DataModel.TPodSpec.Builder podSpec, Optional<String> podAgentAllocationId) {
        List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> allocations = podSpec.getDiskVolumeRequestsBuilderList();
        return podAgentAllocationId.map(agentAllocationId -> allocations.stream()
                .filter(a -> !a.getId().equals(agentAllocationId))
                .collect(Collectors.toList())
        ).orElse(allocations);
    }

    private static void removeUsedByInfraLabel(DataModel.TPodSpec.TDiskVolumeRequest.Builder request) {
        TAttributeDictionary.Builder labelsBuilder = request.getLabelsBuilder();
        List<TAttribute> labels =
                labelsBuilder.getAttributesList().stream().filter(l -> !l.equals(USED_BY_INFRA_LABEL)).collect(Collectors.toList());
        labelsBuilder.clearAttributes();
        labels.forEach(labelsBuilder::addAttributes);
    }

    private static void addUsedByInfraLabel(DataModel.TPodSpec.TDiskVolumeRequest.Builder request) {
        TAttributeDictionary.Builder labelsBuilder = request.getLabelsBuilder();
        if (!labelsBuilder.getAttributesList().contains(USED_BY_INFRA_LABEL)) {
            labelsBuilder.addAttributes(USED_BY_INFRA_LABEL);
        }
    }

    private static <ElementBuilder> void addAllocationRef(List<ElementBuilder> elementsBuilderList,
                                                          Function<ElementBuilder, String> virtualDiskIdGetter,
                                                          Consumer<ElementBuilder> virtualDiskIdSetter) {
        elementsBuilderList.stream()
                .filter(b -> virtualDiskIdGetter.apply(b).isEmpty())
                .forEach(virtualDiskIdSetter);
    }
}
