package ru.yandex.infra.stage.podspecs.patcher.docker;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.StreamEx;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TEnvVar;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;

import static java.util.function.Function.identity;

abstract class DockerPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {

    @VisibleForTesting
    static final String DOCKER_LAYER_ID_PREFIX = "docker-layer";

    public DockerPatcherV1Base(DockerPatcherV1Context context) {

    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        patchPodSpec(podTemplateSpecBuilder.getSpecBuilder(), context);
    }

    private void patchPodSpec(DataModel.TPodSpec.Builder builder, DeployUnitContext context) {
        addDockerInfo(builder.getPodAgentPayloadBuilder().getSpecBuilder(), context.getSpec().getImagesForBoxes(),
                context.getDockerImagesContents());
    }

    protected void addDockerInfo(TPodAgentSpec.Builder agentSpec, Map<String, DockerImageDescription> imagesForBoxes,
                                 Map<DockerImageDescription, DockerImageContents> dockerImagesContents) {
        Map<TBox.Builder, DockerImageDescription> boxToImage =
                agentSpec.getBoxesBuilderList().stream().filter(b -> imagesForBoxes.containsKey(b.getId())).collect(Collectors.toMap(identity(), b -> imagesForBoxes.get(b.getId())));

        TResourceGang.Builder resourcesBuilder = agentSpec.getResourcesBuilder();
        boxToImage.forEach((box, description) -> {
            List<DownloadableResource> imageLayers = dockerImagesContents.get(description).getLayers();
            List<String> currentLayersId = new ArrayList<>();
            for (int i = 0; i < imageLayers.size(); ++i) {
                DownloadableResource current = imageLayers.get(i);
                String currentId = generateLayerId(description, box.getVirtualDiskIdRef(), i, current);
                currentLayersId.add(currentId);

                TLayer.Builder layer =
                        TLayer.newBuilder().setId(currentId).setChecksum(current.getChecksum().toAgentFormat()).setUrl(current.getUrl());

                PodSpecUtils.addLayerIfAbsent(resourcesBuilder, currentId, layer,
                        Optional.of(box.getVirtualDiskIdRef()));
            }
            currentLayersId = StreamEx.ofReversed(currentLayersId).distinct().toList();
            box.getRootfsBuilder().addAllLayerRefs(currentLayersId);
            appendEnvFromImage(box.getEnvList(), box::addEnv, dockerImagesContents.get(description).getEnvironment());
        });

        agentSpec.getWorkloadsBuilderList().forEach(workload -> {
            String boxId = workload.getBoxRef();
            DockerImageDescription description = imagesForBoxes.get(boxId);
            if (description == null) {
                return;
            }
            DockerImageContents contents = dockerImagesContents.get(description);
            TUtilityContainer.Builder start = workload.getStartBuilder();
            if (start.getCommandLine().isEmpty()) {
                List<String> fullCommand = new ArrayList<>();
                fullCommand.addAll(contents.getEntryPoint());
                fullCommand.addAll(contents.getCommand());
                start.setCommandLine(fullCommand.stream().map(arg -> String.format("'%s'",
                        StringUtils.escapeUnaryQuotes(arg))).collect(Collectors.joining(" ")));
            }
            if (start.getUser().isEmpty()) {
                contents.getUser().ifPresent(start::setUser);
            }
            if (start.getGroup().isEmpty()) {
                contents.getGroup().ifPresent(start::setGroup);
            }
            if (start.getCwd().isEmpty()) {
                contents.getWorkingDir().ifPresent(start::setCwd);
            }
            appendWorkloadEnvFromImage(workload, contents.getEnvironment());
        });
    }

    protected abstract String generateLayerId(DockerImageDescription description, String allocationId, int layerNum,
                                              DownloadableResource resource);

    protected void appendWorkloadEnvFromImage(TWorkload.Builder workload, Map<String, String> envVarsFromDockerImage) {
        appendEnvFromImage(workload.getEnvList(), workload::addEnv, envVarsFromDockerImage);
    }

    @VisibleForTesting
    static String toLayerId(DockerImageDescription description, String allocationId, int layerNum) {
        return String.format("%s-%s-%s-%d", DOCKER_LAYER_ID_PREFIX, allocationId, description.getName(), layerNum);
    }

    private static void appendEnvFromImage(List<TEnvVar> list, Consumer<TEnvVar> appender,
                                           Map<String, String> imageEnvironment) {
        Set<String> usedVariables = list.stream().map(TEnvVar::getName).collect(Collectors.toSet());
        imageEnvironment.forEach((key, value) -> {
            if (!usedVariables.contains(key)) {
                appender.accept(PodSpecUtils.literalEnvVar(key, value));
            }
        });
    }
}
