package ru.yandex.infra.stage.podspecs.patcher.docker;

import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.yp.client.pods.TPodAgentSpec;

public class DockerPatcherV2 extends DockerPatcherV1Base {
    public final static String DEPLOY_DOCKER_HASH_ENV_NAME = "DEPLOY_DOCKER_HASH";
    public final static String DEPLOY_DOCKER_IMAGE_ENV_NAME = "DEPLOY_DOCKER_IMAGE";

    private final static Logger LOG = LoggerFactory.getLogger(DockerPatcherV2.class);

    public DockerPatcherV2(DockerPatcherV1Context context) {
        super(context);
    }

    @Override
    protected void addDockerInfo(TPodAgentSpec.Builder agentSpec,
                                 Map<String, DockerImageDescription> imagesForBoxes,
                                 Map<DockerImageDescription, DockerImageContents> dockerImagesContents) {
        super.addDockerInfo(agentSpec, imagesForBoxes, dockerImagesContents);
        patchDockerEnvVars(agentSpec, imagesForBoxes, dockerImagesContents);
    }

    protected void patchDockerEnvVars(TPodAgentSpec.Builder agentSpec,
                                      Map<String, DockerImageDescription> imagesForBoxes,
                                      Map<DockerImageDescription, DockerImageContents> dockerImagesContents) {

        agentSpec.getBoxesBuilderList()
                 .stream()
                 .filter(boxBuilder -> imagesForBoxes.containsKey(boxBuilder.getId()))
                 .forEach(boxBuilder -> {
                     DockerImageDescription description = imagesForBoxes.get(boxBuilder.getId());
                     LOG.debug("Patching '{}' with docker image: {}", boxBuilder.getId(), description);

                     String dockerImage = String.format("%s/%s:%s", description.getRegistryHost(), description.getName(), description.getTag());
                     boxBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_DOCKER_IMAGE_ENV_NAME, dockerImage));

                     DockerImageContents contents = dockerImagesContents.get(description);
                     Optional<String> digest = contents.getDigest();

                     digest.ifPresentOrElse(hash -> {
                         boxBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_DOCKER_HASH_ENV_NAME, hash));
                         LOG.debug("Added {}={} into box env vars. Box: '{}', docker image: {}",
                                 DEPLOY_DOCKER_HASH_ENV_NAME,
                                 hash,
                                 boxBuilder.getId(),
                                 description);
                     }, () -> LOG.warn("TDockerImageContents.digest is empty. Can't add {} into box environment variables. Box: '{}', docker: {}",
                                         DEPLOY_DOCKER_HASH_ENV_NAME,
                                         boxBuilder.getId(),
                                         description));
                 });
    }

    @Override
    protected String generateLayerId(DockerImageDescription description,
                                     String allocationId,
                                     int layerNum,
                                     DownloadableResource resource){
        return toLayerIdV2(allocationId, resource);
    }

    //Generating layerId based only on disk-volumeId & layer digest
    @VisibleForTesting
    static String toLayerIdV2(String allocationId, DownloadableResource resource) {
        return String.format("%s-%s-%s", DOCKER_LAYER_ID_PREFIX, allocationId, resource.getChecksum().toAgentFormat());
    }
}
