package ru.yandex.infra.stage.podspecs.patcher.dynamic_resource;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TRootfsVolume;
import ru.yandex.yp.client.pods.TTimeLimit;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addCpuMemResources;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addLayerIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addResources;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.buildMutableWorkload;

abstract class DynamicResourcePatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    private final ResourceSupplier druDefaultLayerSupplier;
    private final long releaseGetterTimeoutSeconds;
    private static final String DRU_LAYER = "dru-layer";
    private static final String DRU_CMD =
            "/dru --statefile /tmp/dru.state --infosfile /tmp/dru.infos --logfile /tmp/dru.log";
    private static final String PYTHONUNBUFFERED = "PYTHONUNBUFFERED";

    // DRU default TL configuration
    private static final long INITIAL_DELAY_MS = 1000;
    private static final long MAX_EXECUTION_TIME_MS = 10000000;
    private static final long MAX_RESTART_PERIOD_MS = 10000;
    private static final long MIN_RESTART_PERIOD_MS = 5000;
    private static final long RESTART_PERIOD_BACK_OFF = 1;
    private static final long RESTART_PERIOD_SCALE_MS = 100;

    public DynamicResourcePatcherV1Base(DynamicResourcePatcherV1Context context) {
        this(
                context.getDruDefaultLayerSupplier(),
                context.getDruLayerSupplier(),
                context.getReleaseGetterTimeoutSeconds()
        );
    }

    public DynamicResourcePatcherV1Base(ResourceSupplier druDefaultLayerSupplier,
                                        ResourceSupplier druLayerSupplier,
                                        long releaseGetterTimeoutSeconds) {
        this.druDefaultLayerSupplier = druDefaultLayerSupplier;
        this.releaseGetterTimeoutSeconds = releaseGetterTimeoutSeconds;
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {

        DataModel.TPodSpec.Builder podSpec = podTemplateSpecBuilder.getSpecBuilder();
        TPodAgentSpec.Builder agentSpec = podSpec.getPodAgentPayloadBuilder().getSpecBuilder();

        Set<String> boxToPatch = context.getStageContext().getDynamicResources().values().stream()
                .filter(drs -> context.getDeployUnitId().equals(drs.getDeployUnitRef()))
                .flatMap(drs -> drs.getDynamicResource().getDeployGroupsList().stream()
                        .map(dg -> dg.getStorageOptions().getBoxRef())
                )
                .collect(Collectors.toSet());

        agentSpec.getBoxesBuilderList()
                .stream()
                .filter(box -> boxToPatch.contains(box.getId()))
                .forEach(box -> {
                    box.setBindSkynet(true);
                    addDRULayer(box, agentSpec.getResourcesBuilder(), getDRULayerResource(context));
                    patchBoxAndPodResources(podSpec, box);

                    String workloadId = String.format("%s_%s__dru", box.getId(), box.getVirtualDiskIdRef());
                    TWorkload workload = TWorkload.newBuilder()
                            .setId(workloadId)
                            .setBoxRef(box.getId())
                            .setStart(configureDRUStartCommand())
                            .addEnv(PodSpecUtils.literalEnvVar(PYTHONUNBUFFERED, "1"))
                            .build();

                    agentSpec
                            .addWorkloads(workload)
                            .addMutableWorkloads(buildMutableWorkload(workloadId));
                });
    }

    protected abstract AllComputeResources getAdditionalBoxResources();

    protected void patchBoxAndPodResources(DataModel.TPodSpec.Builder podSpecBuilder, TBox.Builder boxBuilder) {
        var resources = getAdditionalBoxResources();
        if (resources != null) {
            addCpuMemResources(podSpecBuilder, resources);
            addResources(boxBuilder, resources, true);
        }
    }

    private ResourceWithMeta getDRULayerResource(DeployUnitContext context) {
        return PatcherUtils.getResource(druDefaultLayerSupplier,
                context.getSpec().getDruLayerResourceInfo(),
                releaseGetterTimeoutSeconds
        );
    }

    private void addDRULayer(TBox.Builder box, TResourceGang.Builder resourceBuilder,
                             ResourceWithMeta druLayerResource) {
        String layerId = String.format("%s_%s", DRU_LAYER, box.getVirtualDiskIdRef());
        TLayer.Builder druLayer = PodSpecUtils.layer(layerId, Optional.empty(), druLayerResource);
        addLayerIfAbsent(resourceBuilder, layerId, druLayer, Optional.of(box.getVirtualDiskIdRef()));
        TRootfsVolume.Builder layerRefs = box.getRootfsBuilder().addLayerRefs(layerId);
        box.setRootfs(layerRefs);
    }

    private TUtilityContainer configureDRUStartCommand() {
        return TUtilityContainer.newBuilder()
                .setCommandLine(DRU_CMD)
                .setTimeLimit(TTimeLimit.newBuilder()
                        .setInitialDelayMs(INITIAL_DELAY_MS)
                        .setMaxExecutionTimeMs(MAX_EXECUTION_TIME_MS)
                        .setMaxRestartPeriodMs(MAX_RESTART_PERIOD_MS)
                        .setMinRestartPeriodMs(MIN_RESTART_PERIOD_MS)
                        .setRestartPeriodBackOff(RESTART_PERIOD_BACK_OFF)
                        .setRestartPeriodScaleMs(RESTART_PERIOD_SCALE_MS))
                .build();
    }
}
