package ru.yandex.infra.stage.podspecs.patcher.juggler;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.controller.util.ResourceUtils;
import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.BoxJugglerConfig;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TMonitoringSubagentEndpoint;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.THttpGet;
import ru.yandex.yp.client.pods.TLivenessCheck;
import ru.yandex.yp.client.pods.TMountedStaticResource;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.MEGABYTE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addCpuMemResources;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addResources;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addStaticResourceIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.buildMutableWorkload;

abstract class JugglerPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    protected static final String JUGGLER_DIR_ROOT = "/juggler";
    protected static final String JUGGLER_DIR_BIN = "bin";
    protected static final String JUGGLER_DIR_TAR_CHECKS = "tar-checks";
    protected static final String TEMPLATE_START_COMMAND = String.format("%s/%s/juggler-scheduler --instance-port %s " +
                    "--base-dir %s " +
                    "--config-dir configs --checks-dir checks --varlib-dir state --run-dir state --logs-dir logs " +
                    "--no-console-log",
            JUGGLER_DIR_ROOT, JUGGLER_DIR_BIN, "%d", JUGGLER_DIR_ROOT);

    @VisibleForTesting
    static final String JUGGLER_DEFAULT_BINARY_ID = "juggler-binary-default";

    @VisibleForTesting
    static final AllComputeResources JUGGLER_RESOURCE_PER_BOX_V1 =
            new AllComputeResources(130, 256 * MEGABYTE, 512 * MEGABYTE, 0);

    private final ResourceSupplier jugglerBinaryDefaultSupplier;
    protected final Optional<List<String>> allSidecarAllocationIds;

    public JugglerPatcherV1Base(JugglerPatcherV1Context context) {
        this.jugglerBinaryDefaultSupplier = context.getJugglerBinaryDefaultSupplier();
        this.allSidecarAllocationIds = context.getAllSidecarDiskAllocationIds();
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        if (context.getSpec().getBoxJugglerConfigs().isEmpty()) {
            return;
        }

        patchPodSpec(podTemplateSpecBuilder.getSpecBuilder(), context.getSpec().getBoxJugglerConfigs(),
                context.getResolvedSbr());
    }

    protected void patchPodSpec(DataModel.TPodSpec.Builder builder,
                                Map<String, BoxJugglerConfig> boxJugglerConfigs,
                                Map<String, String> resolvedSbr) {
        //1. in a case of disabled disk allocations diskIdToBoxJugglerConfigs has one disk key with value "" - empty
        // string
        //2. in a case of enabled disk allocations and one user disk in spec diskIdToBoxJugglerConfigs has one disk
        // key with filled value
        //3. in a case of enabled disk allocations and multiple user disks in spec diskIdToBoxJugglerConfigs has
        // multiple disk keys with filled values
        //4. multiple user disks with disabled disk isolation - impossible
        Map<String, Map<String, BoxJugglerConfig>> diskIdToBoxJugglerConfigs =
                addJugglerConfig(builder, boxJugglerConfigs, resolvedSbr);

        AllComputeResources jugglerResourcesToAdd = getAdditionalBoxResources().multiply(boxJugglerConfigs.size());
        addCpuMemResources(builder, jugglerResourcesToAdd);

        long allAdditionalDiskCapacity = jugglerResourcesToAdd.getDiskCapacity();

        patchDiskQuota(builder, allAdditionalDiskCapacity, diskIdToBoxJugglerConfigs);

        builder.getHostInfraBuilder().getMonitoringBuilder().addAllJugglerSubagents(boxJugglerConfigs.values()
                .stream().map(config -> TMonitoringSubagentEndpoint.newBuilder()
                        .setPort(config.getPortOrDefault())
                        .build()).collect(Collectors.toList()));
    }

    private Map<String, Map<String, BoxJugglerConfig>> addJugglerConfig(DataModel.TPodSpec.Builder podSpec,
                                                                        Map<String, BoxJugglerConfig> boxJugglerConfigs,
                                                                        Map<String, String> resolvedSbr) {

        var agentSpec = podSpec.getPodAgentPayload().getSpec();
        Map<String, Map<String, BoxJugglerConfig>> allocationIdToBoxJugglerConfigs = agentSpec.getBoxesList().stream()
                .filter(b -> boxJugglerConfigs.containsKey(b.getId()))
                .collect(Collectors.groupingBy(TBox::getVirtualDiskIdRef, Collectors.toMap(TBox::getId,
                        b -> boxJugglerConfigs.get(b.getId()))));

        allocationIdToBoxJugglerConfigs.forEach((allocationId, boxJugglerConfig) ->
                addJugglerConfigForAllocation(podSpec, allocationId, boxJugglerConfig, resolvedSbr));
        return allocationIdToBoxJugglerConfigs;
    }

    private void addJugglerConfigForAllocation(DataModel.TPodSpec.Builder podSpec,
                                               String allocationId,
                                               Map<String, BoxJugglerConfig> boxJugglerConfigs,
                                               Map<String, String> resolvedSbr) {
        Map<DownloadableResource, String> idsByResource = new HashMap<>();
        int index = 1;
        for (DownloadableResource resource : boxJugglerConfigs.values().stream()
                .flatMap(value -> value.getArchivedChecks().stream())
                .collect(Collectors.toSet())) {
            idsByResource.put(resource, jugglerCheckId(allocationId, index));
            ++index;
        }

        index = 1;
        boolean needDefault = false;
        for (Optional<DownloadableResource> resource :
                boxJugglerConfigs.values().stream().map(BoxJugglerConfig::getJugglerBinary).collect(Collectors.toSet())) {
            if (resource.isPresent()) {
                idsByResource.put(resource.get(), jugglerBinaryId(allocationId, index));
                ++index;
            } else {
                needDefault = true;
            }
        }

        var agentSpec = podSpec.getPodAgentPayloadBuilder().getSpecBuilder();

        idsByResource.forEach((resource, id) -> addStaticResourceIfAbsent(agentSpec.getResourcesBuilder(),
                PodSpecUtils.staticResource(id, resolvedSbr.getOrDefault(resource.getUrl(), resource.getUrl()),
                        resource.getChecksum().toAgentFormat()),
                id,
                Optional.of(allocationId)));

        if (needDefault) {
            String defaultBinaryId = jugglerDefaultBinaryId(allocationId);
            addStaticResourceIfAbsent(agentSpec.getResourcesBuilder(), PodSpecUtils.staticResource(defaultBinaryId,
                    Optional.empty(), jugglerBinaryDefaultSupplier.get()), defaultBinaryId, Optional.of(allocationId));
        }
        configureJugglerInBoxes(allocationId, boxJugglerConfigs, idsByResource, agentSpec);
    }

    private void configureJugglerInBoxes(String allocationId, Map<String, BoxJugglerConfig> boxJugglerConfigs,
                                         Map<DownloadableResource, String> idsByResource,
                                         TPodAgentSpec.Builder agentSpec) {
        agentSpec.getBoxesBuilderList().stream()
                .filter(b -> b.getVirtualDiskIdRef().equals(allocationId))
                .forEach(box -> {
                    BoxJugglerConfig config = boxJugglerConfigs.get(box.getId());
                    if (config == null) {
                        return;
                    }
                    box.addStaticResources(TMountedStaticResource.newBuilder()
                            .setMountPoint(String.format("%s/%s", JUGGLER_DIR_ROOT, JUGGLER_DIR_BIN))
                            .setResourceRef(config.getJugglerBinary().map(idsByResource::get).orElse(jugglerDefaultBinaryId(allocationId))));
                    config.getArchivedChecks()
                            .forEach(resource -> box.addStaticResources(TMountedStaticResource.newBuilder()
                                    .setMountPoint(archiveCheckMountPoint(idsByResource.get(resource)))
                                    .setResourceRef(idsByResource.get(resource))));
                    addJugglerInit(box);
                    TWorkload workload = createJugglerWorkload(box, config);
                    agentSpec.addWorkloads(workload);
                    agentSpec.addMutableWorkloads(buildMutableWorkload(workload.getId()));

                    addResources(box, getAdditionalBoxResources(), patchBoxesWithoutConstraint());
                });
    }

    protected abstract boolean patchBoxesWithoutConstraint();

    protected abstract TUtilityContainer createStartContainer(BoxJugglerConfig config);

    protected abstract String archiveCheckMountPoint(String resourceId);

    protected abstract String getInitScript();

    protected abstract void patchDiskQuota(DataModel.TPodSpec.Builder spec, long allAdditionalDiskCapacity,
                                           Map<String, Map<String, BoxJugglerConfig>> diskIdToBoxJugglerConfigs);

    protected abstract AllComputeResources getAdditionalBoxResources();

    private void addJugglerInit(TBox.Builder box) {
        box.addInit(TUtilityContainer.newBuilder()
                .setCommandLine(String.format("/bin/bash -c '%s'", getInitScript())));
    }

    private TWorkload createJugglerWorkload(TBox.Builder box, BoxJugglerConfig config) {
        return TWorkload.newBuilder()
                .setId(jugglerWorkloadId(box.getId()))
                .setBoxRef(box.getId())
                .setStart(createStartContainer(config))
                .setLivenessCheck(TLivenessCheck.newBuilder()
                        .setHttpGet(THttpGet.newBuilder()
                                .setPort(config.getPortOrDefault())
                                .setPath("/stats")
                                .setAny(true)))
                .build();
    }

    @VisibleForTesting
    static String jugglerDefaultBinaryId(String allocationId) {
        return String.format("%s-%s", JUGGLER_DEFAULT_BINARY_ID, allocationId);
    }

    @VisibleForTesting
    static String jugglerCheckId(String allocationId, int index) {
        return String.format("juggler-checks-%s-%d", allocationId, index);
    }

    @VisibleForTesting
    static String jugglerBinaryId(String allocationId, int index) {
        return String.format("juggler-binary-%s-%d", allocationId, index);
    }

    @VisibleForTesting
    static String jugglerWorkloadId(String boxId) {
        return String.format("%s-juggler-workload", boxId);
    }

    @VisibleForTesting
    static String createJugglerInitScript(String scriptName) {
        return StringUtils.escapeUnaryQuotes(String.format(
                ResourceUtils.readResource(scriptName), JUGGLER_DIR_ROOT, JUGGLER_DIR_BIN, JUGGLER_DIR_TAR_CHECKS));
    }
}
