package ru.yandex.infra.stage.podspecs.patcher.juggler;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.BoxJugglerConfig;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.Enums;
import ru.yandex.yp.client.pods.TUtilityContainer;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.patchDiskVolumeRequest;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.startContainer;

public class JugglerPatcherV2 extends JugglerPatcherV1Base {
    private static final String INIT_SCRIPT = createJugglerInitScript("juggler_init_template_v2.sh");

    public JugglerPatcherV2(JugglerPatcherV1Context context) {
        super(context);
    }

    @Override
    protected AllComputeResources getAdditionalBoxResources() {
        return JugglerPatcherV1Base.JUGGLER_RESOURCE_PER_BOX_V1;
    }

    @Override
    protected TUtilityContainer createStartContainer(BoxJugglerConfig config) {
        return startContainer(String.format(TEMPLATE_START_COMMAND, config.getPortOrDefault()),
                              getAdditionalBoxResources());
    }

    @Override
    protected void patchPodSpec(DataModel.TPodSpec.Builder builder,
                                Map<String, BoxJugglerConfig> boxJugglerConfigs,
                                Map<String, String> resolvedSbr) {
        super.patchPodSpec(builder, boxJugglerConfigs, resolvedSbr);

        patchHostNameKind(builder);
    }

    private void patchHostNameKind(DataModel.TPodSpec.Builder specBuilder) {
        if (!specBuilder.hasHostNameKind()) {
            specBuilder.setHostNameKind(Enums.EPodHostNameKind.PHNK_PERSISTENT);
        }
    }

    @Override
    protected String archiveCheckMountPoint(String resourceId) {
        return String.format("%s/%s/%s", JUGGLER_DIR_ROOT, JUGGLER_DIR_TAR_CHECKS, resourceId);
    }

    @Override
    protected String getInitScript() {
        return INIT_SCRIPT;
    }

    @Override
    protected void patchDiskQuota(DataModel.TPodSpec.Builder spec, long allAdditionalDiskCapacity, Map<String, Map<String, BoxJugglerConfig>> diskIdToBoxJugglerConfigs) {
        List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> userDiskVolumeRequests = getUserDiskVolumeRequests(spec, allSidecarAllocationIds);
        if (userDiskVolumeRequests.size() == 1) {
            //one user disk in spec;
            //we can't use the same logic here same as for multiple disks, because in a case of disabled disk isolation
            //map diskIdToBoxJugglerConfigs will have key with empty string;
            //TODO remove if else logic after enabling disk isolation for everyone.
            patchDiskVolumeRequest(userDiskVolumeRequests.get(0), allAdditionalDiskCapacity);
        } else {
            //multiple user disks in spec
            diskIdToBoxJugglerConfigs.forEach((userDiskId, configsPerDisk) -> {
                int numConfigsPerDisk = configsPerDisk.size();
                userDiskVolumeRequests.stream()
                        .filter(d -> d.getId().equals(userDiskId))
                        .findAny()
                        .ifPresent(d -> patchDiskVolumeRequest(d, numConfigsPerDisk*getAdditionalBoxResources().getDiskCapacity()));
            });
        }
    }

    @Override
    protected boolean patchBoxesWithoutConstraint() {
        return true;
    }

    private static List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> getUserDiskVolumeRequests(DataModel.TPodSpec.Builder builder, Optional<List<String>> allSidecarAllocationIds) {
        List<DataModel.TPodSpec.TDiskVolumeRequest.Builder> allDiskRequests = builder.getDiskVolumeRequestsBuilderList();
        return allSidecarAllocationIds.map(sidecarDiskIds -> allDiskRequests.stream()
                .filter(r -> !sidecarDiskIds.contains(r.getId()))
                .collect(Collectors.toList())).orElse(allDiskRequests);
    }
}
