package ru.yandex.infra.stage.podspecs.patcher.logbroker;

import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;

import ru.yandex.infra.stage.dto.AllComputeResources;

class LogbrokerBoxResourcesConfig {

    private final Map<String, AllComputeResources> whiteListResources;

    LogbrokerBoxResourcesConfig(Map<String, AllComputeResources> whiteListResources) {
        this.whiteListResources = ImmutableMap.copyOf(whiteListResources);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        LogbrokerBoxResourcesConfig that = (LogbrokerBoxResourcesConfig) o;
        return Objects.equal(whiteListResources, that.whiteListResources);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                whiteListResources
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("whiteListResources", whiteListResources)
                .toString();
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {

        private Map<String, AllComputeResources> whiteListResources;

        Builder(LogbrokerBoxResourcesConfig config) {
            this.whiteListResources = Map.copyOf(config.whiteListResources);
        }

        @VisibleForTesting
        Builder withWhiteListResources(Map<String, AllComputeResources> whiteListResources) {
            this.whiteListResources = whiteListResources;
            return this;
        }

        LogbrokerBoxResourcesConfig build() {
            return new LogbrokerBoxResourcesConfig(
                    whiteListResources
            );
        }
    }

    /**
     *
     * @param fullDeployUnitId - if deploy unit id in white list - customized resources
     * @param patcherDefaultResources -  in other cases - default resources from patcher
     */
    @VisibleForTesting
    AllComputeResources getDefaultBoxResources(String fullDeployUnitId,
                                               AllComputeResources patcherDefaultResources) {
        var customResources = whiteListResources.get(fullDeployUnitId);

        return Optional.ofNullable(customResources)
                .orElse(patcherDefaultResources);
    }

    @VisibleForTesting
    static AllComputeResources getMergedResources(AllComputeResources defaultBoxResources,
                                                  AllComputeResources requestedResources) {
        long actualVcpuLimit = Math.max(
                defaultBoxResources.getVcpuLimit(),
                requestedResources.getVcpuLimit()
        );

        return defaultBoxResources.toBuilder()
                .withVcpuGuarantee(requestedResources.getVcpuGuarantee())
                .withVcpuLimit(actualVcpuLimit)
                .build();
    }

    public AllComputeResources getActualBoxResources(String fullDeployUnitId,
                                                     AllComputeResources patcherDefaultResources,
                                                     Optional<AllComputeResources> requestedResourcesOptional) {
        var defaultResources = getDefaultBoxResources(fullDeployUnitId, patcherDefaultResources);

        return requestedResourcesOptional
                .map(requestedResources -> getMergedResources(defaultResources, requestedResources))
                .orElse(defaultResources);
    }
}
