package ru.yandex.infra.stage.podspecs.patcher.logbroker;

import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.LogbrokerConfig;
import ru.yandex.infra.stage.dto.LogbrokerDestroyPolicy;
import ru.yandex.infra.stage.dto.LogbrokerTopicConfig;
import ru.yandex.infra.stage.dto.PushAgentConfig;
import ru.yandex.infra.stage.dto.SandboxResourceInfo;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.SidecarDiskVolumeDescription;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfig;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigFactory;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.ETransmitSystemLogs;
import ru.yandex.yp.client.pods.EVolumeMountMode;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TComputeResources;
import ru.yandex.yp.client.pods.TDestroyPolicy;
import ru.yandex.yp.client.pods.TEnvVar;
import ru.yandex.yp.client.pods.THttpGet;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TLivenessCheck;
import ru.yandex.yp.client.pods.TMountedStaticResource;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TReadinessCheck;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TRootfsVolume;
import ru.yandex.yp.client.pods.TStopPolicy;
import ru.yandex.yp.client.pods.TTimeLimit;
import ru.yandex.yp.client.pods.TTransmitSystemLogsPolicy;
import ru.yandex.yp.client.pods.TUlimitSoft;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;
import ru.yandex.yp.client.pods.TWorkloadOrBuilder;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.MEGABYTE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.SYSTEM_BOX_SPECIFIC_TYPE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addBoxIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addLayerIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addStaticResourceIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addVolumeIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.allocationAccordingToDiskIsolationLabel;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.buildMutableWorkload;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.startContainer;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.staticResource;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.DEPLOY_LOGNAME_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.DEPLOY_LOGS_ENDPOINT_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.DEPLOY_LOGS_SECRET_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.ERROR_BOOSTER_HTTP_HOST;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.ERROR_BOOSTER_HTTP_PORT;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.ERROR_BOOSTER_SENTRY_DSN;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.ERROR_BOOSTER_SENTRY_DSN_HARDCODE;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.ERROR_BOOSTER_SYSLOG_HOST;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.ERROR_BOOSTER_SYSLOG_PORT;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOCALHOST;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_AGENT_LOGS_VOLUME_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_AGENT_PORTO_LOGS_VOLUME_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_AGENT_STATE_VOLUME_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_AGENT_TVM_SECRET_ENV_NAME;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_AGENT_WORKLOAD_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_BASE_LAYER_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_DEFAULT_STATIC_SECRET;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_MONITOR_WORKLOAD_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_THREAD_LIMIT;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.LOGBROKER_TOOLS_LAYER_ID;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.useLogbrokerTools;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigV2.SYSLOG_HTTP_PORT;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigV2.SYSLOG_INPUT_PORT;
import static ru.yandex.infra.stage.util.JsonUtils.jsonToYaml;
import static ru.yandex.yp.client.pods.EContainerULimitType.EContainerULimit_CORE;

abstract class LogbrokerPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    @VisibleForTesting
    static final String LOGBROKER_AGENT_STAGE_ID_ENV_NAME = "DEPLOY_STAGE";
    @VisibleForTesting
    static final String LOGBROKER_AGENT_DEPLOY_UNIT_ENV_NAME = "DEPLOY_DEPLOY_UNIT";
    @VisibleForTesting
    static final String LOGBROKER_AGENT_PROJECT_ID_ENV_NAME = "DEPLOY_PROJECT";
    private static final String LOGBROKER_AGENT_STATIC_SECRET_ENV_NAME = "STATIC_SECRET";
    @VisibleForTesting
    static final String LOGBROKER_AGENT_CONF_RESOURCE_ID = "push_agent_config";
    @VisibleForTesting
    static final String LOGBROKER_UNIFIED_AGENT_CONF_RESOURCE_ID = "unified_agent_config";
    private static final String LOGBROKER_AGENT_LOGS_MOUNT_POINT = "/unified-agent/logs";
    private static final String LOGBROKER_AGENT_PORTO_LOGS_MOUNT_POINT = "/porto_log_files";
    private static final String LOGBROKER_AGENT_STATE_MOUNT_POINT = "/push-agent/logs";
    @VisibleForTesting
    static final int LOGBROKER_AGENT_STDERR_STDOUT_LIMIT_BYTES = 100 * 1024 * 1024;

    @VisibleForTesting
    static final String LOGNAME_LABEL_KEY = "default_logname";
    @VisibleForTesting
    static final String LOGS_ENDPOINT_LABEL_KEY = "logs_endpoint";

    @VisibleForTesting
    static final String UNIFIED_AGENT_VERSION_LABEL_KEY = "unified_agent_version";

    @VisibleForTesting
    static final String DELIVERY_ENABLED_KEY = "delivery_enabled";
    @VisibleForTesting
    static final boolean DELIVERY_ENABLED_VALUE = true;
    @VisibleForTesting
    static final String DELIVERY_TVM_DST_CLIENT_ID_KEY = "delivery_tvm_dst_client_id";
    @VisibleForTesting
    static final String LOGS_KEY = "logs";

    @VisibleForTesting
    static final String UNIFIED_AGENT_NEW_PROTOCOL_SUPPORTED_KEY = "ua_v2_protocol_supported";
    @VisibleForTesting
    static final boolean UNIFIED_AGENT_NEW_PROTOCOL_SUPPORTED_VALUE = true;

    @VisibleForTesting
    static final String LOGBROKER_USE_NEW_PROTOCOL_FLAG_PATH = "LogsTransmitter.UseNewProtocol";
    @VisibleForTesting
    static final String LOGBROKER_USE_NEW_PROTOCOL_FLAG = "true";

    @VisibleForTesting
    static final String LOGBROKER_USE_STRINGS_BATCHING_FLAG_PATH = "LogsTransmitter" +
            ".UseStringsBatchingWithNewProtocol";
    @VisibleForTesting
    static final String LOGBROKER_USE_STRINGS_BATCHING_FLAG = "true";

    private static final String LOGBROKER_PORTO_LOG_FILES_DIR_PATH = "LogsTransmitter.PortoLogFilesDir";
    @VisibleForTesting
    static final String LOGBROKER_LOGNAME_PATH = "LogsTransmitter.LogName";
    @VisibleForTesting
    static final String LOGBROKER_DEPLOY_UNIT_PATH = "LogsTransmitter.DeployUnit";
    @VisibleForTesting
    static final String LOGBROKER_PROJECT_ID_PATH = "LogsTransmitter.ProjectId";
    private static final String LOGBROKER_PUSH_AGENT_CONF_MOUNT_POINT = "static_data";
    private static final String LOGBROKER_UNIFIED_AGENT_CONF_MOUNT_POINT = "static_data_unified_agent_conf";
    private static final int LOGBROKER_UNIFIED_AGENT_HTTP_HOOKS_PORT = 12502;
    private static final int LOGBROKER_MONITOR_HTTP_HOOKS_PORT = 12503;
    private static final String PORTO_LOG_FILE_EXTENSION = "portolog";
    @VisibleForTesting
    static final String STDOUT_LOG_TYPE = "stdout";
    @VisibleForTesting
    static final String STDERR_LOG_TYPE = "stderr";

    @VisibleForTesting
    static final String PUSH_AGENT_CONFIG_COPY_CMD =
            "bash -c 'rm push-agent/internal/config.json; cp static_data/raw_file push-agent/internal/config" +
                    ".json'";

    @VisibleForTesting
    static final String UNIFIED_AGENT_CONFIG_COPY_CMD =
            "bash -c 'rm push-agent/internal/unified_agent.yaml; cp static_data_unified_agent_conf/raw_file " +
                    "push-agent/internal/unified_agent" +
                    ".yaml'";

    protected static final String LOGBROKER_PATCHER_THROTTLING_LIMITS_KEY_FROM_V1_TO_V2 = "limits_from_patcher_v1_to_v2";
    private static final String LOGBROKER_PATCHER_THROTTLING_LIMITS_KEY_FROM_V3_TO_LAST = "limits_from_patcher_v3_to_last";

    @VisibleForTesting
    static final long LOGBROKER_TOOLS_HDD_CAPACITY = 3 * 1024 * MEGABYTE;

    protected static final AllComputeResources LOGBROKER_BOX_COMPUTING_RESOURCES_FOR_V1 =
            new AllComputeResources(
                    400,
                    512 * MEGABYTE,
                    LOGBROKER_TOOLS_HDD_CAPACITY,
                    LOGBROKER_THREAD_LIMIT
            );

    protected static final AllComputeResources LOGBROKER_BOX_COMPUTING_RESOURCES_FROM_V2_TO_LAST =
            LOGBROKER_BOX_COMPUTING_RESOURCES_FOR_V1.toBuilder()
                    .withAnonymousMemoryLimit(448 * MEGABYTE)
                    .build();

    private final ResourceSupplier logbrokerAgentLayerSupplier;

    private final Optional<String> diskVolumeAllocationId;
    private final Optional<List<String>> allSidecarDiskAllocationIds;
    private final boolean patchBoxSpecificType;

    private final UnifiedAgentConfigFactory unifiedAgentConfigFactory;
    private final LogbrokerBoxResourcesConfig boxResourcesConfig;

    private final long releaseGetterTimeoutSeconds;

    LogbrokerPatcherV1Base(LogbrokerPatcherV1Context context) {
        this(context.getLogbrokerAgentLayerSupplier(),
                context.getDiskVolumeAllocationId(),
                context.getAllSidecarDiskAllocationIds(),
                context.isPatchBoxSpecificType(),
                context.getUnifiedAgentConfigFactory(),
                context.getBoxResourcesConfig(),
                context.getReleaseGetterTimeoutSeconds());
    }

    private LogbrokerPatcherV1Base(ResourceSupplier logbrokerAgentLayerSupplier,
                                   Optional<String> diskVolumeAllocationId,
                                   Optional<List<String>> allSidecarDiskAllocationIds,
                                   boolean patchBoxSpecificType,
                                   UnifiedAgentConfigFactory unifiedAgentConfigFactory,
                                   LogbrokerBoxResourcesConfig boxResourcesConfig,
                                   long releaseGetterTimeoutSeconds) {
        this.logbrokerAgentLayerSupplier = logbrokerAgentLayerSupplier;
        this.diskVolumeAllocationId = diskVolumeAllocationId;
        this.allSidecarDiskAllocationIds = allSidecarDiskAllocationIds;
        this.patchBoxSpecificType = patchBoxSpecificType;
        this.unifiedAgentConfigFactory = unifiedAgentConfigFactory;
        this.boxResourcesConfig = boxResourcesConfig;
        this.releaseGetterTimeoutSeconds = releaseGetterTimeoutSeconds;
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {

        LogbrokerConfig logbrokerConfig = context.getSpec().getLogbrokerConfig();
        if (useLogbrokerTools(podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder(), logbrokerConfig)) {

            Optional<String> logbrokerAllocation = allocationAccordingToDiskIsolationLabel(podTemplateSpecBuilder,
                    diskVolumeAllocationId);
            Optional<String> portoLogsVolumeDiskAllocation = allocationAccordingToDiskIsolationLabel(podTemplateSpecBuilder,
                    getPortoLogsVolumeDiskAllocation(podTemplateSpecBuilder.getSpecBuilder(), allSidecarDiskAllocationIds, logbrokerConfig));

            patchPodSpec(logbrokerAllocation, portoLogsVolumeDiskAllocation, podTemplateSpecBuilder.getSpecBuilder(), context);
            final ResourceWithMeta toolsLayerResource =
                    getLogbrokerToolsLayerResource(context.getSpec().getLogbrokerConfig(),
                            context.getSpec().getLogbrokerToolsResourceInfo());
            logbrokerLabels(labelsBuilder, context, toolsLayerResource);
        } else {
            addDefaultLogbrokerStaticSecretToWorkloads(podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder());
        }
        if (errorBoosterEnvironmentsExport()){
            addErrorBoosterEnv(podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder());
        }
        if (errorBoosterHttpInputEnvironmentsExport()){
            addErrorBoosterHttpInputEnv(podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder());
        }
        if (autoEnableSystemLogsWhenUserLogsEnabled()) {
            enableSystemLogsWhenUserLogsEnabled(podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder());
        }
    }

    private void patchPodSpec(Optional<String> logbrokerAllocation,
                              Optional<String> portoLogsVolumeDiskAllocation,
                              DataModel.TPodSpec.Builder builder,
                              DeployUnitContext context) {

        var logbrokerTopicConfig = context.getLogbrokerTopicConfig().orElseThrow();

        addLogbrokerEnvironment(
                logbrokerAllocation, portoLogsVolumeDiskAllocation, builder, context,
                LOGBROKER_DEFAULT_STATIC_SECRET, logbrokerTopicConfig);

        DataModel.TPodSpec.TPodAgentDeploymentMeta.Builder metaBuilder = builder.getPodAgentPayloadBuilder()
                .getMetaBuilder();

        metaBuilder.putConfiguration(LOGBROKER_LOGNAME_PATH, context.getStageContext().getStageId())
                .putConfiguration(LOGBROKER_DEPLOY_UNIT_PATH, context.getDeployUnitId())
                .putConfiguration(LOGBROKER_PROJECT_ID_PATH, context.getStageContext().getProjectId())
                .putConfiguration(LOGBROKER_PORTO_LOG_FILES_DIR_PATH, LOGBROKER_AGENT_PORTO_LOGS_MOUNT_POINT);

        metaBuilder.putConfiguration(LOGBROKER_USE_NEW_PROTOCOL_FLAG_PATH, LOGBROKER_USE_NEW_PROTOCOL_FLAG);
        metaBuilder.putConfiguration(LOGBROKER_USE_STRINGS_BATCHING_FLAG_PATH, LOGBROKER_USE_STRINGS_BATCHING_FLAG);
    }

    private static void logbrokerLabels(YTreeBuilder labelsBuilder, DeployUnitContext context,
                                        ResourceWithMeta toolsLayerResource) {
        String stageId = context.getStageContext().getStageId();
        var logbrokerTopicConfig = context.getLogbrokerTopicConfig().orElseThrow();

        var deliveryTvmDstClientId = String.valueOf(logbrokerTopicConfig.getTopicDescription().getTvmClientId());

        YTreeBuilder nodeBuilder = new YTreeBuilder().beginMap()
                .key(DELIVERY_ENABLED_KEY).value(DELIVERY_ENABLED_VALUE)
                .key(DELIVERY_TVM_DST_CLIENT_ID_KEY).value(deliveryTvmDstClientId)
                .key(LOGNAME_LABEL_KEY).value(stageId)
                .key(LOGS_ENDPOINT_LABEL_KEY).value(logEndpoint());

        nodeBuilder.key(UNIFIED_AGENT_NEW_PROTOCOL_SUPPORTED_KEY).value(UNIFIED_AGENT_NEW_PROTOCOL_SUPPORTED_VALUE);

        labelsBuilder.key(LOGS_KEY).value(nodeBuilder.endMap().build());

        // TODO think about move to UnifiedAgentConfigFactory
        toolsLayerResource.getMeta()
                .map(m -> m.getAttributes().get(UNIFIED_AGENT_VERSION_LABEL_KEY))
                .ifPresent(version ->
                        labelsBuilder.key(UNIFIED_AGENT_VERSION_LABEL_KEY).value(version)
                );
    }

    @VisibleForTesting
    static Path portoLogFilePathInBox(String workloadId, String logType) {
        return Path.of(LOGBROKER_AGENT_PORTO_LOGS_MOUNT_POINT,
                String.format("%s_%s.%s", workloadId, logType, PORTO_LOG_FILE_EXTENSION));
    }

    @VisibleForTesting
    static Path unifiedAgentLogFilePathInBox(String logType) {
        return Path.of(LOGBROKER_AGENT_LOGS_MOUNT_POINT,
                String.format("%s_%s.%s", LOGBROKER_AGENT_WORKLOAD_ID, logType, PORTO_LOG_FILE_EXTENSION));
    }

    private ResourceWithMeta getLogbrokerToolsLayerResource(LogbrokerConfig config,
                                                            Optional<SandboxResourceInfo> logbrokerToolsSidecar) {
        return PatcherUtils.getResource(logbrokerAgentLayerSupplier,
                logbrokerToolsSidecar,
                releaseGetterTimeoutSeconds
        );
    }

    protected String getPatcherThrottlingLimitsKey() {
        return LOGBROKER_PATCHER_THROTTLING_LIMITS_KEY_FROM_V3_TO_LAST;
    }

    private void addLogbrokerEnvironment(Optional<String> logbrokerAllocation,
                                         Optional<String> portoLogsVolumeDiskAllocation,
                                         DataModel.TPodSpec.Builder podSpecBuilder,
                                         DeployUnitContext context,
                                         String logbrokerStaticSecret,
                                         LogbrokerTopicConfig logbrokerTopicConfig) {
        var logbrokerToolsSidecar = context.getSpec().getLogbrokerToolsResourceInfo();
        var stageId = context.getStageContext().getStageId();
        var deployUnitId = context.getDeployUnitId();
        var fullDeployUnitId = context.getFullDeployUnitId();
        var projectId = context.getStageContext().getProjectId();
        var logbrokerConfig = context.getSpec().getLogbrokerConfig();
        var podAdditionalResourcesRequest = logbrokerConfig.getPodAdditionalResourcesRequest();
        var clusters = context.getSpec().getDetails().extractClusters();

        TPodAgentSpec.Builder agentSpec = podSpecBuilder.getPodAgentPayloadBuilder().getSpecBuilder();
        TResourceGang.Builder resourcesBuilder = agentSpec.getResourcesBuilder();

        ResourceWithMeta logbrokerToolsLayerResource = getLogbrokerToolsLayerResource(logbrokerConfig, logbrokerToolsSidecar);

        TResource.Builder conf = staticResource(LOGBROKER_AGENT_CONF_RESOURCE_ID,
                String.format("raw:%s", new PushAgentConfig(stageId, logbrokerStaticSecret).toJsonString()),
                "EMPTY:");
        addStaticResourceIfAbsent(resourcesBuilder, conf, LOGBROKER_AGENT_CONF_RESOURCE_ID, logbrokerAllocation);

        var unifiedAgentConfig = unifiedAgentConfigFactory.build(
                resourcesBuilder.getLayersList(),
                logbrokerToolsLayerResource.getMeta(),
                logbrokerToolsSidecar,
                logbrokerConfig.getTopicRequest(),
                logbrokerAgentLayerSupplier,
                stageId,
                deployUnitId,
                fullDeployUnitId,
                clusters,
                logbrokerTopicConfig,
                getPatcherThrottlingLimitsKey(),
                logbrokerStaticSecret
        );

        TResource.Builder unifiedAgentConf = staticResource(LOGBROKER_UNIFIED_AGENT_CONF_RESOURCE_ID,
                String.format("raw:%s", jsonToYaml(unifiedAgentConfig.toJsonString())),
                "EMPTY:");
        addStaticResourceIfAbsent(resourcesBuilder, unifiedAgentConf, LOGBROKER_UNIFIED_AGENT_CONF_RESOURCE_ID, logbrokerAllocation);

        TLayer.Builder baseLayer = TLayer.newBuilder()
                .setId(LOGBROKER_BASE_LAYER_ID)
                // https://sandbox.yandex-team.ru/resource/904130395/view
                .setUrl("rbtorrent:edd2795d2d7674eae43c4ad0de3dad563be11f94")
                .setChecksum("MD5:234a033db29f2e4afae189851c0ee63d");

        addLayerIfAbsent(resourcesBuilder, LOGBROKER_BASE_LAYER_ID, baseLayer, logbrokerAllocation);

        addLayerIfAbsent(resourcesBuilder,
                LOGBROKER_TOOLS_LAYER_ID, PodSpecUtils.layer(LOGBROKER_TOOLS_LAYER_ID, logbrokerConfig.getLogbrokerAgentLayer(), logbrokerToolsLayerResource),
                logbrokerAllocation);

        agentSpec.setResources(resourcesBuilder);

        addVolumeIfAbsent(agentSpec, LOGBROKER_AGENT_LOGS_VOLUME_ID, logbrokerAllocation);
        addVolumeIfAbsent(agentSpec, LOGBROKER_AGENT_PORTO_LOGS_VOLUME_ID, portoLogsVolumeDiskAllocation);
        addVolumeIfAbsent(agentSpec, LOGBROKER_AGENT_STATE_VOLUME_ID, logbrokerAllocation);

        TBox.Builder logbrokerBox = TBox.newBuilder()
                .setId(getLogbrokerBoxId())
                .setRootfs(TRootfsVolume.newBuilder()
                        .addAllLayerRefs(ImmutableList.of(LOGBROKER_BASE_LAYER_ID, LOGBROKER_TOOLS_LAYER_ID))
                        .build())
                .addStaticResources(TMountedStaticResource.newBuilder()
                        .setResourceRef(LOGBROKER_AGENT_CONF_RESOURCE_ID)
                        .setMountPoint(LOGBROKER_PUSH_AGENT_CONF_MOUNT_POINT)
                        .build());

        logbrokerBox.addStaticResources(TMountedStaticResource.newBuilder()
                .setResourceRef(LOGBROKER_UNIFIED_AGENT_CONF_RESOURCE_ID)
                .setMountPoint(LOGBROKER_UNIFIED_AGENT_CONF_MOUNT_POINT)
                .build());

        AllComputeResources logbrokerBoxComputeResources = boxResourcesConfig.getActualBoxResources(
                fullDeployUnitId,
                getPatcherDefaultBoxResources(),
                podAdditionalResourcesRequest
        );

        logbrokerBox.setComputeResources(logbrokerBoxComputeResources.toProto())
                .addAllVolumes(ImmutableList.of(
                        TMountedVolume.newBuilder()
                                .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                                .setMountPoint(LOGBROKER_AGENT_LOGS_MOUNT_POINT)
                                .setVolumeRef(LOGBROKER_AGENT_LOGS_VOLUME_ID)
                                .build(),
                        TMountedVolume.newBuilder()
                                .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                                .setMountPoint(LOGBROKER_AGENT_STATE_MOUNT_POINT)
                                .setVolumeRef(LOGBROKER_AGENT_STATE_VOLUME_ID)
                                .build()));
        if (patchBoxSpecificType) {
            logbrokerBox.setSpecificType(SYSTEM_BOX_SPECIFIC_TYPE);
        }

        addBoxIfAbsent(agentSpec, logbrokerBox, getLogbrokerBoxId(), logbrokerAllocation);

        //detect workloads with logs transmitting
        agentSpec.getWorkloadsBuilderList().stream()
                .filter(TWorkload.Builder::getTransmitLogs)
                .forEach(workloadBuilder -> {
                    TUtilityContainer.Builder startContainerBuilder = workloadBuilder.getStartBuilder();
                    startContainerBuilder
                            .setStdoutFile(portoLogFilePathInBox(workloadBuilder.getId(), STDOUT_LOG_TYPE).toString());
                    startContainerBuilder
                            .setStderrFile(portoLogFilePathInBox(workloadBuilder.getId(), STDERR_LOG_TYPE).toString());
                    workloadBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_LOGNAME_ENV_NAME, stageId));
                    workloadBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_LOGS_ENDPOINT_ENV_NAME, logEndpoint()));
                    workloadBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_LOGS_SECRET_ENV_NAME, logbrokerStaticSecret));
                    workloadBuilder.build();
                });

        //detect boxes with workloads with logs transmitting
        Set<String> boxIdsWithLogs = agentSpec.getWorkloadsList().stream()
                .filter(TWorkload::getTransmitLogs)
                .map(TWorkload::getBoxRef)
                .filter(boxId -> !boxId.isEmpty())
                .collect(Collectors.toSet());

        agentSpec.getBoxesBuilderList().stream()
                .filter(boxBuilder -> boxIdsWithLogs.contains(boxBuilder.getId()))
                .forEach(boxBuilder -> boxBuilder.addVolumes(TMountedVolume.newBuilder()
                        .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                        .setMountPoint(LOGBROKER_AGENT_PORTO_LOGS_MOUNT_POINT)
                        .setVolumeRef(LOGBROKER_AGENT_PORTO_LOGS_VOLUME_ID)
                        .build()
                ));


        addUnifiedAgentSpecificWorkloads(
                podSpecBuilder, stageId, deployUnitId, projectId,
                logbrokerStaticSecret,
                logbrokerConfig.getDestroyPolicy(),
                logbrokerTopicConfig
        );

        var podAdditionalCpuMemoryResources = agentSpec.getBoxesList().stream()
                .filter(box -> getLogbrokerBoxId().equals(box.getId()))
                .findFirst()
                .map(TBox::getComputeResources)
                .orElse(TComputeResources.newBuilder().build());

        var defaultPodAdditionalResources = new AllComputeResources(
                podAdditionalCpuMemoryResources,
                logbrokerBoxComputeResources.getDiskCapacity(),
                logbrokerBoxComputeResources.getThreadLimit()
        );

        var actualPodAdditionalResources = podAdditionalResourcesRequest
                .orElse(defaultPodAdditionalResources);

        var podAdditionalResources = defaultPodAdditionalResources.toBuilder()
                .withVcpuGuarantee(actualPodAdditionalResources.getVcpuGuarantee())
                .withVcpuLimit(actualPodAdditionalResources.getVcpuLimit())
                .build();

        Optional<SidecarDiskVolumeDescription> sidecarDiskVolumeDescription = logbrokerAllocation.map(diskAllocationId ->
                PodSpecUtils.calculateSidecarDiskVolumeDescription(
                        stageId, podSpecBuilder, diskAllocationId, logbrokerConfig.getSidecarVolumeSettings(),
                        allSidecarDiskAllocationIds.orElseThrow()
                )
        );

        PodSpecUtils.addResources(podSpecBuilder, podAdditionalResources, sidecarDiskVolumeDescription);
    }


    private void addUnifiedAgentSpecificWorkloads(DataModel.TPodSpec.Builder podSpec,
                                                  String stageId,
                                                  String deployUnitId, String projectId,
                                                  String logbrokerStaticSecret,
                                                  LogbrokerDestroyPolicy unifiedAgentDestroyPolicy,
                                                  LogbrokerTopicConfig logbrokerTopicConfig) {
        String pushMonitorStartCmd = "/push-agent/hooks/monitor-start";

        TWorkload monitorWorkload = TWorkload.newBuilder()
                .setId(LOGBROKER_MONITOR_WORKLOAD_ID)
                .setBoxRef(getLogbrokerBoxId())
                .setStart(startContainer(pushMonitorStartCmd))
                .setStopPolicy(TStopPolicy.newBuilder()
                        .setMaxTries(LogbrokerPatcherUtils.DEFAULT_UNIFIED_AGENT_POLICY_MAX_TRIES)
                        .setContainer(TUtilityContainer.newBuilder()
                                .setCommandLine("/push-agent/hooks/monitor-stop")
                                .setTimeLimit(logbrokerWorkloadHooksTimeLimit())
                                .build())
                        .build())
                .setReadinessCheck(TReadinessCheck.newBuilder()
                        .setHttpGet(THttpGet.newBuilder()
                                .setPort(LOGBROKER_MONITOR_HTTP_HOOKS_PORT)
                                .setPath("/ready")
                                .setTimeLimit(logbrokerWorkloadHooksTimeLimit())
                                .setExpectedAnswer("OK")
                                .build())
                        .build())
                .setLivenessCheck(TLivenessCheck.newBuilder()
                        .setHttpGet(THttpGet.newBuilder()
                                .setPort(LOGBROKER_MONITOR_HTTP_HOOKS_PORT)
                                .setPath("/alive")
                                .setTimeLimit(logbrokerWorkloadLivenessHooksTimeLimit())
                                .setExpectedAnswer("OK")
                                .build())
                        .build())
                .build();

        String pushAgentStartCmd = "/push-agent/hooks/agent-start";
        String pushAgentCoreCmd = "cp --sparse=always /dev/stdin /coredumps/${CORE_EXE_NAME}.core";
        TWorkload.Builder pushAgentWorkload = TWorkload.newBuilder()
                .setId(LOGBROKER_AGENT_WORKLOAD_ID)
                .setBoxRef(getLogbrokerBoxId())
                .addUlimitSoft(TUlimitSoft.newBuilder()
                        .setName(EContainerULimit_CORE)
                        .setValue(768 * MEGABYTE)
                        .build())
                .addInit(TUtilityContainer.newBuilder()
                        .setCommandLine(PUSH_AGENT_CONFIG_COPY_CMD)
                        .setTimeLimit(logbrokerWorkloadHooksTimeLimit())
                        .build());

        pushAgentWorkload.addInit(TUtilityContainer.newBuilder()
                .setCommandLine(UNIFIED_AGENT_CONFIG_COPY_CMD)
                .setTimeLimit(logbrokerWorkloadHooksTimeLimit())
                .build());

        pushAgentWorkload
                .setStopPolicy(TStopPolicy.newBuilder()
                        .setMaxTries(LogbrokerPatcherUtils.DEFAULT_UNIFIED_AGENT_POLICY_MAX_TRIES)
                        .setContainer(TUtilityContainer.newBuilder()
                                .setCommandLine("/push-agent/hooks/agent-stop")
                                .setTimeLimit(logbrokerWorkloadHooksTimeLimit())
                                .build()))
                .setDestroyPolicy(TDestroyPolicy.newBuilder()
                        .setMaxTries(unifiedAgentDestroyPolicy.getMaxTries())
                        .setContainer(TUtilityContainer.newBuilder()
                                .setCommandLine("/push-agent/hooks/agent-destroy")
                                .setTimeLimit(logbrokerWorkloadHooksTimeLimit(unifiedAgentDestroyPolicy.getRestartPeriodMs()))
                                .build()))
                .setReadinessCheck(TReadinessCheck.newBuilder()
                        .setHttpGet(THttpGet.newBuilder()
                                .setPort(LOGBROKER_UNIFIED_AGENT_HTTP_HOOKS_PORT)
                                .setPath("/ready")
                                .setTimeLimit(logbrokerWorkloadHooksTimeLimit())
                                .setExpectedAnswer("OK")
                                .build())
                        .build())
                .setLivenessCheck(TLivenessCheck.newBuilder()
                        .setHttpGet(THttpGet.newBuilder()
                                .setPort(LOGBROKER_UNIFIED_AGENT_HTTP_HOOKS_PORT)
                                .setPath("/status")
                                .setTimeLimit(logbrokerWorkloadLivenessHooksTimeLimit())
                                .setAny(true)
                                .build())
                        .build())
                .addEnv(logbrokerTopicConfig.isCommunalTopic() ?
                        PodSpecUtils.literalEnvVar(LOGBROKER_AGENT_TVM_SECRET_ENV_NAME,
                                logbrokerTopicConfig.getCommunalTopicTvmToken().orElseThrow()) :
                        PodSpecUtils.secretEnvVar(LOGBROKER_AGENT_TVM_SECRET_ENV_NAME,
                                logbrokerTopicConfig.getCustomTopicSecretSelector().orElseThrow()))
                .addEnv(PodSpecUtils.literalEnvVar(LOGBROKER_AGENT_STAGE_ID_ENV_NAME, stageId))
                .addEnv(PodSpecUtils.literalEnvVar(LOGBROKER_AGENT_DEPLOY_UNIT_ENV_NAME, deployUnitId))
                .addEnv(PodSpecUtils.literalEnvVar(LOGBROKER_AGENT_PROJECT_ID_ENV_NAME, projectId))
                .addEnv(PodSpecUtils.literalEnvVar(LOGBROKER_AGENT_STATIC_SECRET_ENV_NAME, logbrokerStaticSecret))
                .setStart(
                        startContainer(pushAgentStartCmd,
                                pushAgentCoreCmd,
                                LOGBROKER_AGENT_STDERR_STDOUT_LIMIT_BYTES,
                                Optional.of(unifiedAgentLogFilePathInBox(STDOUT_LOG_TYPE)),
                                Optional.of(unifiedAgentLogFilePathInBox(STDERR_LOG_TYPE))))
                .build();

        addWorkloadIfNotExists(podSpec, pushAgentWorkload.build(), LOGBROKER_AGENT_WORKLOAD_ID);
        addWorkloadIfNotExists(podSpec, monitorWorkload, LOGBROKER_MONITOR_WORKLOAD_ID);
    }

    private static void addDefaultLogbrokerStaticSecretToWorkloads(TPodAgentSpec.Builder agentSpec) {
        agentSpec.getWorkloadsBuilderList()
                .forEach(workloadBuilder -> {
                    if (workloadBuilder.getEnvList().stream().noneMatch(e -> e.getName().equals(DEPLOY_LOGS_SECRET_ENV_NAME))) {
                        workloadBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_LOGS_SECRET_ENV_NAME, LOGBROKER_DEFAULT_STATIC_SECRET));
                    }
                });
    }

    private static void addErrorBoosterEnv(TPodAgentSpec.Builder agentSpec) {
        ImmutableList<TEnvVar> errorBoosterEnvs = ImmutableList.of(
                PodSpecUtils.literalEnvVar(ERROR_BOOSTER_SYSLOG_HOST, LOCALHOST),
                PodSpecUtils.literalEnvVar(ERROR_BOOSTER_SYSLOG_PORT, String.valueOf(SYSLOG_INPUT_PORT)),
                PodSpecUtils.literalEnvVar(ERROR_BOOSTER_SENTRY_DSN, ERROR_BOOSTER_SENTRY_DSN_HARDCODE));

        agentSpec.getWorkloadsBuilderList().forEach(workloadBuilder -> workloadBuilder.addAllEnv(errorBoosterEnvs));
        agentSpec.getBoxesBuilderList().forEach(boxBuilder -> boxBuilder.addAllEnv(errorBoosterEnvs));
    }

    private static void addErrorBoosterHttpInputEnv(TPodAgentSpec.Builder agentSpec) {
        ImmutableList<TEnvVar> errorBoosterEnvs = ImmutableList.of(
                PodSpecUtils.literalEnvVar(ERROR_BOOSTER_HTTP_HOST, LOCALHOST),
                PodSpecUtils.literalEnvVar(ERROR_BOOSTER_HTTP_PORT, String.valueOf(SYSLOG_HTTP_PORT)));

        agentSpec.getWorkloadsBuilderList().forEach(workloadBuilder -> workloadBuilder.addAllEnv(errorBoosterEnvs));
        agentSpec.getBoxesBuilderList().forEach(boxBuilder -> boxBuilder.addAllEnv(errorBoosterEnvs));
    }

    private static void enableSystemLogsWhenUserLogsEnabled(TPodAgentSpec.Builder agentSpec) {
        if (agentSpec.getWorkloadsOrBuilderList().stream().anyMatch(TWorkloadOrBuilder::getTransmitLogs) &&
            agentSpec.getTransmitSystemLogsPolicy().getTransmitSystemLogs() == ETransmitSystemLogs.ETransmitSystemLogsPolicy_NONE ) {
            agentSpec.setTransmitSystemLogsPolicy(TTransmitSystemLogsPolicy.newBuilder()
                    .setTransmitSystemLogs(ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED)
                    .build()
            );
        }
    }

    private static TTimeLimit logbrokerWorkloadHooksTimeLimit() {
        return logbrokerWorkloadHooksTimeLimit(LogbrokerPatcherUtils.DEFAULT_UNIFIED_AGENT_POLICY_RESTART_PERIOD_MS);
    }

    private static TTimeLimit logbrokerWorkloadHooksTimeLimit(long restartPeriodMs) {
        return TTimeLimit.newBuilder()
                .setMinRestartPeriodMs(restartPeriodMs)
                .setMaxRestartPeriodMs(restartPeriodMs)
                .setMaxExecutionTimeMs(15000)
                .build();
    }

    private static TTimeLimit logbrokerWorkloadLivenessHooksTimeLimit() {
        return TTimeLimit.newBuilder()
                .setMinRestartPeriodMs(30000)
                .setMaxRestartPeriodMs(30000)
                .setMaxExecutionTimeMs(15000)
                .setInitialDelayMs(150000)
                .build();
    }

    private static void addWorkloadIfNotExists(DataModel.TPodSpec.Builder podSpec, TWorkload workload, String workloadId) {
        var agentSpec = podSpec.getPodAgentPayloadBuilder().getSpecBuilder();

        if (agentSpec.getWorkloadsList().stream().noneMatch(w -> w.getId().equals(workloadId))) {
            agentSpec.addWorkloads(workload);
            agentSpec.addMutableWorkloads(buildMutableWorkload(workloadId));
        }
    }

    @VisibleForTesting
    static String logEndpoint() {
        return UnifiedAgentConfig.GRPC_ENDPOINT_URI;
    }

    @VisibleForTesting
    static Optional<String> getPortoLogsVolumeDiskAllocation(DataModel.TPodSpec.Builder podSpec,
                                                             Optional<List<String>> allSidecarAllocations,
                                                             LogbrokerConfig logbrokerConfig) {
        return allSidecarAllocations.map(sidecarAllocations -> {
            List<String> userDiskIds = getUserDiskIds(podSpec, sidecarAllocations);

            int numUserDisks = userDiskIds.size();
            if (numUserDisks == 1) {
                return userDiskIds.get(0);
            } else {
                return logbrokerConfig.getLogsVirtualDiskIdRef().orElseThrow();
            }
        });
    }

    private static List<String> getUserDiskIds(DataModel.TPodSpec.Builder podSpec, List<String> allSidecarAllocations) {
        return podSpec.getDiskVolumeRequestsBuilderList().stream()
                .map(DataModel.TPodSpec.TDiskVolumeRequest.Builder::getId)
                .filter(id -> !allSidecarAllocations.contains(id))
                .collect(Collectors.toList());
    }

    protected AllComputeResources getPatcherDefaultBoxResources() {
        return LOGBROKER_BOX_COMPUTING_RESOURCES_FROM_V2_TO_LAST;
    }

    protected String getLogbrokerBoxId() {
        return LogbrokerPatcherUtils.LOGBROKER_BOX_ID;
    }

    protected boolean errorBoosterEnvironmentsExport(){
        return true;
    }

    protected boolean errorBoosterHttpInputEnvironmentsExport(){
        return true;
    }

    protected boolean autoEnableSystemLogsWhenUserLogsEnabled() { return true; }

}
