package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import ru.yandex.infra.stage.dto.LogbrokerCustomTopicRequest;
import ru.yandex.infra.stage.dto.LogbrokerTopicRequest;
import ru.yandex.infra.stage.podspecs.SandboxResourceMeta;
import ru.yandex.infra.stage.podspecs.SandboxResourceMetaAttributesUtils;

import static ru.yandex.infra.stage.podspecs.SandboxResourceMetaAttributesUtils.FLAG_CALCULATOR;

public class ErrorBoosterIsEnabledPredicateImpl implements ErrorBoosterIsEnabledPredicate {

    private final long firstAffectedUnifiedAgentVersion;
    private final Optional<String> enabledInReleaseFlagName;
    private final SandboxResourceMetaAttributesUtils.FlagCalculator flagCalculator;

    ErrorBoosterIsEnabledPredicateImpl(long firstAffectedUnifiedAgentVersion,
                                       Optional<String> enabledInReleaseFlagName) {
        this(firstAffectedUnifiedAgentVersion,
            enabledInReleaseFlagName,
            FLAG_CALCULATOR);
    }

    @VisibleForTesting
    ErrorBoosterIsEnabledPredicateImpl(long firstAffectedUnifiedAgentVersion,
                                       Optional<String> enabledInReleaseFlagName,
                                       SandboxResourceMetaAttributesUtils.FlagCalculator flagCalculator) {
        this.firstAffectedUnifiedAgentVersion = firstAffectedUnifiedAgentVersion;
        this.enabledInReleaseFlagName = enabledInReleaseFlagName;
        this.flagCalculator = flagCalculator;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ErrorBoosterIsEnabledPredicateImpl that = (ErrorBoosterIsEnabledPredicateImpl) o;
        return firstAffectedUnifiedAgentVersion == that.firstAffectedUnifiedAgentVersion
                && Objects.equal(enabledInReleaseFlagName, that.enabledInReleaseFlagName)
                && Objects.equal(flagCalculator, that.flagCalculator);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                firstAffectedUnifiedAgentVersion,
                enabledInReleaseFlagName,
                flagCalculator
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("firstAffectedUnifiedAgentVersion", firstAffectedUnifiedAgentVersion)
                .add("enabledInReleaseFlagName", enabledInReleaseFlagName)
                .add("flagCalculator", flagCalculator)
                .toString();
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {
        private final long firstAffectedUnifiedAgentVersion;
        private Optional<String> enabledInReleaseFlagName;
        private SandboxResourceMetaAttributesUtils.FlagCalculator flagCalculator;

        Builder(ErrorBoosterIsEnabledPredicateImpl config) {
            this.firstAffectedUnifiedAgentVersion = config.firstAffectedUnifiedAgentVersion;
            this.enabledInReleaseFlagName = config.enabledInReleaseFlagName;
            this.flagCalculator = config.flagCalculator;
        }

        Builder withEnabledInReleaseFlagName(Optional<String> enabledInReleaseFlagName) {
            this.enabledInReleaseFlagName = enabledInReleaseFlagName;
            return this;
        }

        Builder withFlagCalculator(SandboxResourceMetaAttributesUtils.FlagCalculator flagCalculator) {
            this.flagCalculator = flagCalculator;
            return this;
        }

        ErrorBoosterIsEnabledPredicate build() {
            return new ErrorBoosterIsEnabledPredicateImpl(
                    firstAffectedUnifiedAgentVersion,
                    enabledInReleaseFlagName,
                    flagCalculator
            );
        }
    }

    public boolean test(Optional<Long> unifiedAgentVersionOptional,
                        Optional<SandboxResourceMeta> unifiedAgentSandboxMeta,
                        LogbrokerTopicRequest topicRequest) {
        boolean unifiedAgentVersionAffected = unifiedAgentVersionOptional
                .map(version -> version >= firstAffectedUnifiedAgentVersion)
                .orElse(false);

        if (!unifiedAgentVersionAffected) return false;

        boolean enabledInReleaseByFlag = enabledInReleaseFlagName.map(flagName ->
                        flagCalculator.getFlagValue(unifiedAgentSandboxMeta, flagName)
        ).orElse(true);

        if (!enabledInReleaseByFlag) return false;

        boolean isCustomTopic = (topicRequest instanceof LogbrokerCustomTopicRequest);
        if (isCustomTopic) return false;

        return true;
    }
}
