package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;

import ru.yandex.infra.stage.dto.LogbrokerCustomTopicRequest;
import ru.yandex.infra.stage.dto.LogbrokerTopicRequest;

class ThrottlingGenerationConfig {

    private static final String UNLIMITED_STAGE_RATE_LIMIT = "1tb";

    private static final int UNLIMITED_STAGE_MESSAGES_RATE_LIMIT = 100500;

    @VisibleForTesting
    static final ThrottlingLimits UNLIMITED_LIMITS = new ThrottlingLimits(
            UNLIMITED_STAGE_RATE_LIMIT,
            UNLIMITED_STAGE_MESSAGES_RATE_LIMIT
    );

    private final Map<String, ThrottlingLimits> whiteListLimits;
    private final long firstAffectedUnifiedAgentVersion;
    private final Map<String, ThrottlingLimits> patcherLimits;

    ThrottlingGenerationConfig(Map<String, ThrottlingLimits> whiteListLimits,
                                       long firstAffectedUnifiedAgentVersion,
                                       Map<String, ThrottlingLimits> patcherLimits) {
        this.whiteListLimits = ImmutableMap.copyOf(whiteListLimits);
        this.firstAffectedUnifiedAgentVersion = firstAffectedUnifiedAgentVersion;
        this.patcherLimits = ImmutableMap.copyOf(patcherLimits);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ThrottlingGenerationConfig that = (ThrottlingGenerationConfig) o;
        return firstAffectedUnifiedAgentVersion == that.firstAffectedUnifiedAgentVersion
                && Objects.equal(whiteListLimits, that.whiteListLimits)
                && Objects.equal(patcherLimits, that.patcherLimits);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                whiteListLimits,
                firstAffectedUnifiedAgentVersion,
                patcherLimits
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("whiteListLimits", whiteListLimits)
                .add("firstAffectedUnifiedAgentVersion", firstAffectedUnifiedAgentVersion)
                .add("patcherLimits", patcherLimits)
                .toString();
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {

        private Map<String, ThrottlingLimits> whiteListLimits;
        private final long firstAffectedUnifiedAgentVersion;
        private Map<String, ThrottlingLimits> patcherLimits;

        Builder(ThrottlingGenerationConfig config) {
            this.whiteListLimits = Map.copyOf(config.whiteListLimits);
            this.firstAffectedUnifiedAgentVersion = config.firstAffectedUnifiedAgentVersion;
            this.patcherLimits = Map.copyOf(config.patcherLimits);
        }

        Builder withDeployUnitLimits(Map<String, ThrottlingLimits> stageLimits) {
            this.whiteListLimits = stageLimits;
            return this;
        }

        Builder withPatcherLimits(Map<String, ThrottlingLimits> patcherLimits) {
            this.patcherLimits = patcherLimits;
            return this;
        }

        ThrottlingGenerationConfig build() {
            return new ThrottlingGenerationConfig(
                    whiteListLimits,
                    firstAffectedUnifiedAgentVersion,
                    patcherLimits
            );
        }
    }

    /**
     *
     * @param topicRequest - if custom request - ALWAYS default
     * @param unifiedAgentVersion - if old UA - ALWAYS default
     * @param fullDeployUnitId - if deploy unit id in white list - customized limits
     * @param patcherLimitsKey - if key in patcherLimits - use limits for patcher
     */
    @VisibleForTesting
    ThrottlingLimits getDeployUnitLimits(LogbrokerTopicRequest topicRequest,
                                         Optional<Long> unifiedAgentVersion,
                                         String fullDeployUnitId,
                                         String patcherLimitsKey) {
        if (topicRequest instanceof LogbrokerCustomTopicRequest) {
            return UNLIMITED_LIMITS;
        }

        if (unifiedAgentVersion.isEmpty() ||
            unifiedAgentVersion.filter(version -> version < firstAffectedUnifiedAgentVersion).isPresent()) {
            return UNLIMITED_LIMITS;
        }

        var customLimits = whiteListLimits.get(fullDeployUnitId);
        if (null != customLimits) {
            return customLimits;
        }

        return Optional.ofNullable(patcherLimits.get(patcherLimitsKey))
                .orElse(UNLIMITED_LIMITS);
    }
}
