package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import com.google.common.annotations.VisibleForTesting;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import one.util.streamex.EntryStream;

import ru.yandex.infra.stage.ConfigUtils;
import ru.yandex.misc.lang.StringUtils;

import static java.util.Collections.emptyMap;
import static ru.yandex.infra.controller.util.StringUtils.ID_SEPARATOR;
import static ru.yandex.infra.stage.ConfigUtils.extractMap;

class ThrottlingGenerationConfigParser {

    @VisibleForTesting
    static final String THROTTLING_WHITE_LIST_FILE_CONFIG_PATH = "white_list_file";
    private static final String THROTTLING_WHITE_LIST_LIMITS_CONFIG_PATH = "white_list_limits";

    @VisibleForTesting
    static final String THROTTLING_FIRST_AFFECTED_UNIFIED_AGENT_VERSION_CONFIG_PATH =
            "first_affected_unified_agent_version";

    @VisibleForTesting
    static final String THROTTLING_PATCHER_LIMITS_CONFIG_PATH = "patcher_limits";

    private static final String THROTTLING_LIMITS_RATE_CONFIG_PATH = "rate";
    private static final String THROTTLING_LIMITS_MESSAGES_RATE_CONFIG_PATH = "messages_rate";

    private static ThrottlingLimits parseThrottlingLimits(Config config) {
        return new ThrottlingLimits(
                config.getString(THROTTLING_LIMITS_RATE_CONFIG_PATH),
                config.getInt(THROTTLING_LIMITS_MESSAGES_RATE_CONFIG_PATH)
        );
    }

    private static Map<String, ThrottlingLimits> parseLimits(Config config) {
        return extractMap(config, Config::getConfig, ThrottlingGenerationConfigParser::parseThrottlingLimits);
    }

    private static Map<String, ThrottlingLimits> parseDeployUnitLimits(Config config) {
        var stageToDeployUnitLimitConfigs = extractMap(
                config.getConfig(THROTTLING_WHITE_LIST_LIMITS_CONFIG_PATH),
                Config::getConfig
        );

        Map<String, ThrottlingLimits> deployUnitLimits = new HashMap<>();
        EntryStream.of(stageToDeployUnitLimitConfigs)
                .forKeyValue((stageId, stageConfig) -> {
                    var stageDeployUnitLimits = parseLimits(stageConfig);
                    EntryStream.of(stageDeployUnitLimits)
                            .mapKeys(deployUnitId -> stageId + ID_SEPARATOR + deployUnitId)
                            .forKeyValue(deployUnitLimits::put);
                });

        return deployUnitLimits;
    }

    static ThrottlingGenerationConfig parseConfig(Config config) {
        var whiteListFilePath = ConfigUtils.getOptional(
                config, THROTTLING_WHITE_LIST_FILE_CONFIG_PATH,
                ConfigUtils.STRING_FROM_CONFIG
        ).filter(StringUtils::isNotEmpty);

        /*
        By default 'while list file' is processed as path to external file (similar to files with tokens)

        But for testing purpose 'white list file' could store path to resource file
        (separate file, not application_defaults)

        If both cases are possible to parse and use as correct config - external file would be prioritized
         */

        var whiteListConfigFromExternalFile = whiteListFilePath
                .map(File::new)
                .map(ConfigFactory::parseFile)
                .filter(fileConfig -> !fileConfig.isEmpty());

        var whiteListConfigFromResourceFile = whiteListFilePath
                .map(ConfigFactory::load)
                .filter(fileConfig -> !fileConfig.isEmpty());

        var whiteListConfig = whiteListConfigFromExternalFile.isPresent()
                ? whiteListConfigFromExternalFile
                : whiteListConfigFromResourceFile;

        var duLimits = whiteListConfig
                .map(ThrottlingGenerationConfigParser::parseDeployUnitLimits)
                .orElse(emptyMap());

        var firstAffectedUnifiedAgentVersion = config.getLong(
                THROTTLING_FIRST_AFFECTED_UNIFIED_AGENT_VERSION_CONFIG_PATH
        );

        var patcherLimits = parseLimits(
                config.getConfig(THROTTLING_PATCHER_LIMITS_CONFIG_PATH)
        );

        return new ThrottlingGenerationConfig(duLimits, firstAffectedUnifiedAgentVersion, patcherLimits);
    }
}
