package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.util.Collection;
import java.util.Optional;
import java.util.Set;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;

import ru.yandex.infra.stage.dto.LogbrokerTopicConfig;
import ru.yandex.infra.stage.dto.LogbrokerTopicRequest;
import ru.yandex.infra.stage.dto.SandboxResourceInfo;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.SandboxResourceMeta;
import ru.yandex.yp.client.pods.TLayer;

public class UnifiedAgentConfigFactory {

    private final ThrottlingGenerationConfig throttlingGenerationConfig;
    private final ErrorBoosterIsEnabledPredicate errorBoosterIsEnabledPredicate;
    private final DataRetentionGenerationConfig dataRetentionGenerationConfig;
    private final BuildTaskIdCalculator buildTaskIdCalculator;
    private final PodSpecUtils.SandboxResourceIdCalculator releaseVersionCalculator;
    private final ThrottlingPolicyCalculator throttlingPolicyCalculator;
    private final VersionCalculator versionCalculator;
    private final VersionFactory versionFactory;

    UnifiedAgentConfigFactory(
            ThrottlingGenerationConfig throttlingGenerationConfig,
            ErrorBoosterIsEnabledPredicate errorBoosterIsEnabledPredicate,
            VersionCalculator versionCalculator,
            DataRetentionGenerationConfig dataRetentionGenerationConfig
    ) {
        this(throttlingGenerationConfig,
            errorBoosterIsEnabledPredicate,
            dataRetentionGenerationConfig,
            BuildTaskIdCalculatorImpl.INSTANCE,
            PodSpecUtils.SANDBOX_RESOURCE_ID_CALCULATOR,
            ThrottlingPolicyCalculatorImpl.INSTANCE,
            versionCalculator,
            VersionFactoryImpl.INSTANCE
        );
    }

    @VisibleForTesting
    UnifiedAgentConfigFactory(
            ThrottlingGenerationConfig throttlingGenerationConfig,
            ErrorBoosterIsEnabledPredicate errorBoosterIsEnabledPredicate,
            DataRetentionGenerationConfig dataRetentionGenerationConfig,
            BuildTaskIdCalculator buildTaskIdCalculator,
            PodSpecUtils.SandboxResourceIdCalculator releaseVersionCalculator,
            ThrottlingPolicyCalculator throttlingPolicyCalculator,
            VersionCalculator versionCalculator,
            VersionFactory versionFactory
    ) {
        this.throttlingGenerationConfig = throttlingGenerationConfig;
        this.errorBoosterIsEnabledPredicate = errorBoosterIsEnabledPredicate;
        this.dataRetentionGenerationConfig = dataRetentionGenerationConfig;
        this.buildTaskIdCalculator = buildTaskIdCalculator;
        this.releaseVersionCalculator = releaseVersionCalculator;
        this.throttlingPolicyCalculator = throttlingPolicyCalculator;
        this.versionCalculator = versionCalculator;
        this.versionFactory = versionFactory;
    }

    public UnifiedAgentConfig build(Collection<TLayer> podAgentPayloadLayers,
                                    Optional<SandboxResourceMeta> unifiedAgentSandboxMeta,
                                    Optional<SandboxResourceInfo> unifiedAgentSandboxInfo,
                                    LogbrokerTopicRequest topicRequest,
                                    ResourceSupplier unifiedAgentResourceSupplier,
                                    String stageId,
                                    String deployUnitId,
                                    String fullDeployUnitId,
                                    Set<String> clusters,
                                    LogbrokerTopicConfig logbrokerTopicConfig,
                                    String patcherThrottlingLimitsKey,
                                    String logbrokerStaticSecret) {
        long buildTaskId = buildTaskIdCalculator.calculate(podAgentPayloadLayers, unifiedAgentSandboxMeta);
        var throttlingPolicy = throttlingPolicyCalculator.calculate(
                unifiedAgentSandboxMeta
        );

        var releaseVersionOptional = releaseVersionCalculator.calculate(
                unifiedAgentSandboxInfo, unifiedAgentResourceSupplier
        );

        var throttlingLimits = throttlingGenerationConfig.getDeployUnitLimits(
                topicRequest,
                releaseVersionOptional,
                fullDeployUnitId,
                patcherThrottlingLimitsKey
        );

        var dataRetentionLimits = dataRetentionGenerationConfig.getDataRetentionLimits();

        var topicDescription = logbrokerTopicConfig.getTopicDescription();

        var errorBoosterEnabled = errorBoosterIsEnabledPredicate.test(
                releaseVersionOptional,
                unifiedAgentSandboxMeta,
                topicRequest
        );

        var version = versionCalculator.calculate(
                errorBoosterEnabled,
                releaseVersionOptional
        );

        return versionFactory.build(
                version,
                buildTaskId,
                logbrokerStaticSecret,
                stageId,
                deployUnitId,
                throttlingPolicy,
                throttlingLimits,
                dataRetentionLimits,
                clusters,
                topicDescription
        );
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UnifiedAgentConfigFactory that = (UnifiedAgentConfigFactory) o;
        return Objects.equal(throttlingGenerationConfig, that.throttlingGenerationConfig)
                && Objects.equal(dataRetentionGenerationConfig, that.dataRetentionGenerationConfig)
                && Objects.equal(buildTaskIdCalculator, that.buildTaskIdCalculator)
                && Objects.equal(releaseVersionCalculator, that.releaseVersionCalculator)
                && Objects.equal(throttlingPolicyCalculator, that.throttlingPolicyCalculator)
                && Objects.equal(versionFactory, that.versionFactory);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                throttlingGenerationConfig,
                dataRetentionGenerationConfig,
                buildTaskIdCalculator,
                releaseVersionCalculator,
                throttlingPolicyCalculator,
                versionFactory
        );
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {

        private ThrottlingGenerationConfig throttlingGenerationConfig;
        private ErrorBoosterIsEnabledPredicate errorBoosterIsEnabledPredicate;
        private DataRetentionGenerationConfig dataRetentionGenerationConfig;
        private BuildTaskIdCalculator buildTaskIdCalculator;
        private PodSpecUtils.SandboxResourceIdCalculator releaseVersionCalculator;
        private ThrottlingPolicyCalculator throttlingPolicyCalculator;
        private VersionCalculator versionCalculator;
        private VersionFactory versionFactory;

        Builder(UnifiedAgentConfigFactory unifiedAgentConfigFactory) {
            this.throttlingGenerationConfig = unifiedAgentConfigFactory.throttlingGenerationConfig;
            this.errorBoosterIsEnabledPredicate = unifiedAgentConfigFactory.errorBoosterIsEnabledPredicate;
            this.dataRetentionGenerationConfig = unifiedAgentConfigFactory.dataRetentionGenerationConfig;
            this.buildTaskIdCalculator = unifiedAgentConfigFactory.buildTaskIdCalculator;
            this.releaseVersionCalculator = unifiedAgentConfigFactory.releaseVersionCalculator;
            this.throttlingPolicyCalculator = unifiedAgentConfigFactory.throttlingPolicyCalculator;
            this.versionCalculator = unifiedAgentConfigFactory.versionCalculator;
            this.versionFactory = unifiedAgentConfigFactory.versionFactory;
        }

        Builder withThrottlingGenerationConfig(ThrottlingGenerationConfig throttlingGenerationConfig) {
            this.throttlingGenerationConfig = throttlingGenerationConfig;
            return this;
        }

        Builder withErrorBoosterIsEnabledPredicate(ErrorBoosterIsEnabledPredicate errorBoosterIsEnabledPredicate) {
            this.errorBoosterIsEnabledPredicate = errorBoosterIsEnabledPredicate;
            return this;
        }

        Builder withDataRetentionGenerationConfig(DataRetentionGenerationConfig dataRetentionGenerationConfig) {
            this.dataRetentionGenerationConfig = dataRetentionGenerationConfig;
            return this;
        }

        Builder withBuildTaskIdCalculator(BuildTaskIdCalculator buildTaskIdCalculator) {
            this.buildTaskIdCalculator = buildTaskIdCalculator;
            return this;
        }

        Builder withReleaseVersionCalculator(PodSpecUtils.SandboxResourceIdCalculator releaseVersionCalculator) {
            this.releaseVersionCalculator = releaseVersionCalculator;
            return this;
        }

        Builder withThrottlingPolicyCalculator(ThrottlingPolicyCalculator throttlingPolicyCalculator) {
            this.throttlingPolicyCalculator = throttlingPolicyCalculator;
            return this;
        }

        Builder withVersionCalculator(VersionCalculator versionCalculator) {
            this.versionCalculator = versionCalculator;
            return this;
        }

        Builder withVersionFactory(VersionFactory versionFactory) {
            this.versionFactory = versionFactory;
            return this;
        }

        UnifiedAgentConfigFactory build() {
            return new UnifiedAgentConfigFactory(
                    throttlingGenerationConfig,
                    errorBoosterIsEnabledPredicate,
                    dataRetentionGenerationConfig,
                    buildTaskIdCalculator,
                    releaseVersionCalculator,
                    throttlingPolicyCalculator,
                    versionCalculator,
                    versionFactory
            );
        }
    }
}
