package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import ru.yandex.infra.stage.dto.LogbrokerTopicDescription;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;

/*
First version of unified agent config generator

Examples (test/resources/logbroker/unified_agent):
- unified_agent
- unified_agent_with_yd_throttling_after_compression
- unified_agent_with_yd_throttling_before_compression

Contains only main output (communal / custom logs topic)

Throttling is optional:
- depends on ThrottlingPolicy
- "after/before compression" is about order of filters - important for old UA versions
 */
class UnifiedAgentConfigV1 implements UnifiedAgentConfig {
    private final static int LOGBROKER_MONITORING_PORT = LogbrokerPatcherUtils.LOGBROKER_AGENT_MONITORING_PORT;
    private final static int LOGBROKER_STATUS_PORT = 12502;
    private final static String TVM_SECRET = LogbrokerPatcherUtils.LOGBROKER_AGENT_TVM_SECRET_ENV_NAME;
    private final static List<Integer> SOLOMON_TVM_CLIENT_IDS = LogbrokerPatcherUtils.LOGBROKER_AGENT_SOLOMON_TVM_CLIENT_IDS;
    private final static String BUILD_TASK_ID = "BuildTaskId";
    private final static String INFLIGHT_LIMIT = "30mb";
    private final static int INPUT_CHANNEL_INFLIGHT_MESSAGES_LIMIT = 100000;

    private final static int STORAGE_INFLIGHT_MESSAGES_LIMIT = 30000;
    private final static String INPUT_CHANNEL_PLUGIN = "grpc";
    private final static String ROUTES_PLUGIN = "assign";
    private final static String COMBINE_BACKTRACES_FILTER_PLUGIN = "combine_backtraces";
    private final static String COMPRESSION_FILTER_PLUGIN = "compress";
    private final static String COMPRESSION_CODEC_TYPE = "gzip";
    private final static int COMPRESSION_QUALITY = 6;
    private final static String YD_THROTTLE_FILTER_PLUGIN = "yd_throttle";
    private final static String COMBINE_BACKTRACES_FLUSH_PERIOD = "300ms";
    private final static String STDOUT_LOGGGER_NAME = "stdout";
    private final static String STDERR_LOGGGER_NAME = "stderr";
    private final static String YD_FORMAT_FILTER_PLUGIN = "yd_format";
    private final static String YD_FORMAT_VERSION = "v2";
    private final static String BATCH_FILTER_PLUGIN = "batch";
    private final static String BATCH_FILTER_DELIMETER = "\n";
    private final static String BATCH_FILTER_FLUSH_PERIOD = "300ms";
    private final static String BATCH_FILTER_LIMIT =  "100kb";
    private final static String STORAGE_DIRECTORY = "/push-agent/logs";
    private final static String STORAGE_MAX_PARTITION_SIZE = "1gb";
    private final static String STORAGE_MAX_SEGMENT_SIZE = "50mb";
    private final static String STORAGE_SESSION_META_MISMATCH_ACTION = "ignore";
    private final static String STORAGE_UNACKNOWLEDGED_EVICTION_LOG_PRIORITY = "WARNING";
    private final static String STORAGE_NAME = "main";
    private final static String STORAGE_PLUGIN = "fs";
    private final static String OUT_CHANNEL_PLUGIN = "logbroker";
    private final static String OUT_CHANNEL_FORMAT_ID = "output-logbroker-yd-format-v2";
    private final static String OUT_CHANNEL_ENDPOINT = LogbrokerPatcherUtils.LOGBROKER_AGENT_OUT_CHANNEL_ENDPOINT;
    private final static String DISK_CACHE_DIRECTORY_1 = "/push-agent/internal/tvm-cache/monitoring";
    private final static String DISK_CACHE_DIRECTORY_2 = "/push-agent/internal/tvm-cache/logbroker-yd-format-v2";
    private final static String DEPLOY_PROJECT_LABEL_ENV_NAME = "DEPLOY_PROJECT";
    private final static String DEPLOY_STAGE_LABEL_ENV_NAME = "DEPLOY_STAGE";
    private final static String DEPLOY_DEPLOY_UNIT_LABEL_ENV_NAME = "DEPLOY_DEPLOY_UNIT";
    private final static String DEPLOY_PROJECT_META_VALUE = "{$env('DEPLOY_PROJECT')}";
    private final static String DEPLOY_STAGE_META_VALUE = "{$env('DEPLOY_STAGE')}";
    private final static String DEPLOY_DEPLOY_UNIT_META_VALUE = "{$env('DEPLOY_DEPLOY_UNIT')}";
    private final static String DEPLOY_POD_META_VALUE = "{$env('DEPLOY_POD_ID')}";
    private final static String DEPLOY_POD_TRANSIENT_FQDN_META_VALUE = "{$env('DEPLOY_POD_TRANSIENT_FQDN')}";
    private final static String DEPLOY_POD_PERSISTENT_FQDN_META_VALUE = "{$env('DEPLOY_POD_PERSISTENT_FQDN')}";
    private final static String DEPLOY_NODE_FQDN_META_VALUE = "{$env('DEPLOY_NODE_FQDN')}";
    private final static String RESOLVE_ENDPOINTS_URL = "http://sd.yandex.net:8080/resolve_endpoints/json";
    private final static String YD_THROTTLE_POLL_PERIOD = "1h";
    private final static String YD_THROTTLE_POLL_TIME_RANDOM_SHIFT = "15m";
    private final static String YD_THROTTLE_CACHE_FILENAME = "/push-agent/internal/unified_agent_pods_count";
    private final static String YD_THROTTLE_CLIENT_NAME_PREFIX = "deploy_logs_stage_discovery";
    private final static int NUM_OF_THREADS = 3;
    private final static String FLOW_GRAPH_MEMORY_QUOTA = "150mb";

    private final long buildTaskId;
    private final String staticSecret;
    private final String stageId;
    private final String deployUnitId;
    private final ThrottlingPolicy throttlingPolicy;
    private final ThrottlingLimits throttlingLimits;
    private final Set<String> clusters;
    private final LogbrokerTopicDescription topicDescription;

    /**
     Constant values should be constants of that class
     and not variables of constructor
     Unified agent config is very complex and it is safer to have tests
     that are nearest to prod config as much as possible
     */
    @VisibleForTesting
    UnifiedAgentConfigV1(
            long buildTaskId,
            String staticSecret,
            String stageId,
            String deployUnitId,
            ThrottlingPolicy throttlingPolicy,
            ThrottlingLimits throttlingLimits,
            Set<String> clusters,
            LogbrokerTopicDescription topicDescription) {
        this.buildTaskId = buildTaskId;
        this.staticSecret = staticSecret;
        this.stageId = stageId;
        this.deployUnitId = deployUnitId;
        this.throttlingPolicy = throttlingPolicy;
        this.throttlingLimits = throttlingLimits;
        this.clusters = clusters;
        this.topicDescription = topicDescription;
    }

    @VisibleForTesting
    UnifiedAgentConfigV1 withThrottlingPolicy(ThrottlingPolicy throttlingPolicy) {
        return new UnifiedAgentConfigV1(
                buildTaskId,
                staticSecret,
                stageId,
                deployUnitId,
                throttlingPolicy,
                throttlingLimits,
                clusters,
                topicDescription
        );
    }

    @VisibleForTesting
    UnifiedAgentConfigV1 withThrottlingLimits(ThrottlingLimits throttlingLimits) {
        return new UnifiedAgentConfigV1(
                buildTaskId,
                staticSecret,
                stageId,
                deployUnitId,
                throttlingPolicy,
                throttlingLimits,
                clusters,
                topicDescription
        );
    }

    @VisibleForTesting
    public UnifiedAgentConfigV1 withBuildTaskId(long buildTaskId) {
        return new UnifiedAgentConfigV1(
                buildTaskId,
                staticSecret,
                stageId,
                deployUnitId,
                throttlingPolicy,
                throttlingLimits,
                clusters,
                topicDescription
        );
    }

    private JsonNode tvmNode(ObjectMapper mapper) {
        ObjectNode tvmNode = mapper.createObjectNode();
        tvmNode.put("client_id", topicDescription.getTvmClientId());
        ArrayNode solomonClientIdsNode = mapper.createArrayNode();
        SOLOMON_TVM_CLIENT_IDS.forEach(solomonClientIdsNode::add);
        tvmNode.set("allowed_clients", solomonClientIdsNode);
        tvmNode.put("disk_cache_directory", DISK_CACHE_DIRECTORY_1);
        return tvmNode;
    }

    private static JsonNode staticLabelsNode(ObjectMapper mapper) {
        ArrayNode staticLabelsNode = mapper.createArrayNode();
        staticLabelsNode.add(labelNode(mapper, "Project", DEPLOY_PROJECT_LABEL_ENV_NAME));
        staticLabelsNode.add(labelNode(mapper, "Stage", DEPLOY_STAGE_LABEL_ENV_NAME));
        staticLabelsNode.add(labelNode(mapper, "DeployUnit", DEPLOY_DEPLOY_UNIT_LABEL_ENV_NAME));
        return staticLabelsNode;
    }

    private JsonNode staticSensorsNode(ObjectMapper mapper) {
        ArrayNode staticSensorsNode = mapper.createArrayNode();
        ObjectNode buildTaskIdNode = mapper.createObjectNode();
        buildTaskIdNode.put("name", BUILD_TASK_ID);
        buildTaskIdNode.put("value", buildTaskId);
        staticSensorsNode.add(buildTaskIdNode);
        return staticSensorsNode;
    }

    private JsonNode monitoringNode(ObjectMapper mapper) {
        ObjectNode monitoringNode = mapper.createObjectNode();
        monitoringNode.put("port", LOGBROKER_MONITORING_PORT);
        monitoringNode.set("tvm", tvmNode(mapper));
        monitoringNode.set("static_labels", staticLabelsNode(mapper));
        monitoringNode.set("static_sensors", staticSensorsNode(mapper));
        return monitoringNode;
    }

    private static JsonNode systemNode(ObjectMapper mapper) {
        ObjectNode systemNode = mapper.createObjectNode();
        systemNode.put("lock_executable_in_memory", true);
        return systemNode;
    }

    private static JsonNode flowGraphNode(ObjectMapper mapper) {
        ObjectNode flowGraphNode = mapper.createObjectNode();
        ObjectNode memoryQuotaNode = mapper.createObjectNode();
        memoryQuotaNode.put("limit", FLOW_GRAPH_MEMORY_QUOTA);

        flowGraphNode.set("memory_quota", memoryQuotaNode);

        return flowGraphNode;
    }

    private static JsonNode threadPoolNode(ObjectMapper mapper) {
        ObjectNode threadPoolNode = mapper.createObjectNode();
        threadPoolNode.put("threads", NUM_OF_THREADS);
        return threadPoolNode;
    }

    private static JsonNode statusNode(ObjectMapper mapper) {
        ObjectNode statusNode = mapper.createObjectNode();
        statusNode.put("port", LOGBROKER_STATUS_PORT);
        return statusNode;
    }

    private JsonNode inputRouteNode(ObjectMapper mapper) {
        ObjectNode inputNode = mapper.createObjectNode();
        inputNode.put("plugin", INPUT_CHANNEL_PLUGIN);

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("uri", GRPC_ENDPOINT_URI);
        configNode.put("shared_secret_key", staticSecret);
        inputNode.set("config", configNode);

        ObjectNode flowControlNode = mapper.createObjectNode();
        ObjectNode inflightNode = mapper.createObjectNode();
        inflightNode.put("limit", INFLIGHT_LIMIT);
        inflightNode.put("limit_messages", INPUT_CHANNEL_INFLIGHT_MESSAGES_LIMIT);

        flowControlNode.set("inflight", inflightNode);
        inputNode.set("flow_control", flowControlNode);
        return inputNode;
    }

    private static JsonNode sessionMetaNode(ObjectMapper mapper) {
        ArrayNode sessionMetaNode = mapper.createArrayNode();
        sessionMetaNode.add(keyValueNode(mapper, "deploy_project", DEPLOY_PROJECT_META_VALUE));
        sessionMetaNode.add(keyValueNode(mapper, "deploy_stage", DEPLOY_STAGE_META_VALUE));
        sessionMetaNode.add(keyValueNode(mapper, "deploy_unit", DEPLOY_DEPLOY_UNIT_META_VALUE));
        sessionMetaNode.add(keyValueNode(mapper, "deploy_pod", DEPLOY_POD_META_VALUE));
        sessionMetaNode.add(keyValueNode(mapper, "deploy_pod_transient_fqdn", DEPLOY_POD_TRANSIENT_FQDN_META_VALUE));
        sessionMetaNode.add(keyValueNode(mapper, "deploy_pod_persistent_fqdn", DEPLOY_POD_PERSISTENT_FQDN_META_VALUE));
        sessionMetaNode.add(keyValueNode(mapper, "deploy_node_fqdn", DEPLOY_NODE_FQDN_META_VALUE));
        return sessionMetaNode;
    }

    private JsonNode routesNode(ObjectMapper mapper) {
        ArrayNode routesNode = mapper.createArrayNode();

        ObjectNode routeNode = mapper.createObjectNode();
        routeNode.set("input", inputRouteNode(mapper));

        ObjectNode channelNode = mapper.createObjectNode();

        ArrayNode pipesNode = mapper.createArrayNode();

        ObjectNode pipeNode = mapper.createObjectNode();
        ObjectNode filterNode = mapper.createObjectNode();
        filterNode.put("plugin", ROUTES_PLUGIN);

        ObjectNode pipeConfigNode = mapper.createObjectNode();

        pipeConfigNode.set("session", sessionMetaNode(mapper));
        filterNode.set("config", pipeConfigNode);
        pipeNode.set("filter", filterNode);

        pipesNode.add(storageRefNode(mapper));

        pipesNode.add(pipeNode);
        pipesNode.add(combinedBacktracesFilterNode(mapper));

        if (throttlingPolicy == ThrottlingPolicy.THROTTLING_BEFORE_COMPRESSION) {
            pipesNode.add(ydThrottlingFilterNode(mapper));
        }

        pipesNode.add(ydFormatFilterNode(mapper));
        pipesNode.add(batchFilterNode(mapper));

        if (isThrottlingEnabled()) {
            pipesNode.add(compressFilterNode(mapper));
        }

        if (throttlingPolicy == ThrottlingPolicy.THROTTLING_AFTER_COMPRESSION) {
            pipesNode.add(ydThrottlingFilterNode(mapper));
        }


        channelNode.set("pipe", pipesNode);
        channelNode.set("output", outChannelDescriptionNode(mapper));
        routeNode.set("channel", channelNode);
        routesNode.add(routeNode);
        return routesNode;
    }

    private static JsonNode storageRefNode(ObjectMapper mapper) {
        ObjectNode storageRefNode = mapper.createObjectNode();
        storageRefNode.put("name", "main");

        ObjectNode flowControlNode = mapper.createObjectNode();

        ObjectNode inflightNode = mapper.createObjectNode();
        inflightNode.put("limit", INFLIGHT_LIMIT);
        inflightNode.put("limit_messages", STORAGE_INFLIGHT_MESSAGES_LIMIT);

        flowControlNode.set("inflight", inflightNode);
        storageRefNode.set("flow_control", flowControlNode);
        return mapper.createObjectNode().set("storage_ref", storageRefNode);
    }

    private JsonNode ydThrottlingFilterNode(ObjectMapper mapper) {
        ObjectNode filterNode = mapper.createObjectNode();
        filterNode.put("plugin", YD_THROTTLE_FILTER_PLUGIN);

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("stage_rate_limit", throttlingLimits.getMaxRate());

        if (throttlingPolicy == ThrottlingPolicy.THROTTLING_BEFORE_COMPRESSION) {
            configNode.put("stage_messages_rate_limit", throttlingLimits.getMaxMessagesRate());
        }

        ArrayNode clusterNameNode = mapper.createArrayNode();
        clusters.forEach(clusterNameNode::add);

        ObjectNode rateAdjustmentNode = mapper.createObjectNode();
        rateAdjustmentNode.put("resolve_endpoints_url", RESOLVE_ENDPOINTS_URL);
        rateAdjustmentNode.put("poll_period", YD_THROTTLE_POLL_PERIOD);
        rateAdjustmentNode.put("poll_time_random_shift", YD_THROTTLE_POLL_TIME_RANDOM_SHIFT);
        rateAdjustmentNode.set("cluster_names", clusterNameNode);
        rateAdjustmentNode.put("endpoint_set_id", String.format("%s.%s", stageId, deployUnitId));
        rateAdjustmentNode.put("client_name", String.format("%s_%s", YD_THROTTLE_CLIENT_NAME_PREFIX, stageId));
        rateAdjustmentNode.put("cache_file_name", YD_THROTTLE_CACHE_FILENAME);
        configNode.set("rate_adjustment", rateAdjustmentNode);

        filterNode.set("config", configNode);

        return mapper.createObjectNode().set("filter", filterNode);
    }

    private static JsonNode combinedBacktracesFilterNode(ObjectMapper mapper) {
        ObjectNode filterNode = mapper.createObjectNode();
        filterNode.put("plugin", COMBINE_BACKTRACES_FILTER_PLUGIN);

        ObjectNode enableIfNode = mapper.createObjectNode();
        ObjectNode sessionNode = mapper.createObjectNode();

        ArrayNode loggerNamesNode = mapper.createArrayNode();
        loggerNamesNode.add(STDOUT_LOGGGER_NAME);
        loggerNamesNode.add(STDERR_LOGGGER_NAME);

        sessionNode.set("deploy_logger_name", loggerNamesNode);
        enableIfNode.set("session", sessionNode);
        filterNode.set("enable_if", enableIfNode);

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("flush_period", COMBINE_BACKTRACES_FLUSH_PERIOD);
        filterNode.set("config", configNode);
        return mapper.createObjectNode().set("filter", filterNode);
    }

    private static JsonNode ydFormatFilterNode(ObjectMapper mapper) {
        ObjectNode filterNode = mapper.createObjectNode();
        filterNode.put("plugin", YD_FORMAT_FILTER_PLUGIN);
        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("format_version", YD_FORMAT_VERSION);
        filterNode.set("config", configNode);
        return mapper.createObjectNode().set("filter", filterNode);
    }

    private static JsonNode batchFilterNode(ObjectMapper mapper) {
        ObjectNode filterNode = mapper.createObjectNode();
        filterNode.put("plugin", BATCH_FILTER_PLUGIN);

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("delimiter", BATCH_FILTER_DELIMETER);
        configNode.put("flush_period", BATCH_FILTER_FLUSH_PERIOD);

        ObjectNode limitNode = mapper.createObjectNode();
        limitNode.put("bytes", BATCH_FILTER_LIMIT);

        configNode.set("limit", limitNode);
        filterNode.set("config", configNode);
        return mapper.createObjectNode().set("filter", filterNode);
    }

    private static JsonNode compressFilterNode(ObjectMapper mapper) {
        ObjectNode filterNode = mapper.createObjectNode();
        filterNode.put("plugin", COMPRESSION_FILTER_PLUGIN);

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("codec", COMPRESSION_CODEC_TYPE);
        configNode.put("compression_quality", COMPRESSION_QUALITY);

        filterNode.set("config", configNode);
        return mapper.createObjectNode().set("filter", filterNode);
    }

    private JsonNode outChannelDescriptionNode(ObjectMapper mapper) {
        ObjectNode descriptionNode = mapper.createObjectNode();
        descriptionNode.put("plugin", OUT_CHANNEL_PLUGIN);
        descriptionNode.put("id", OUT_CHANNEL_FORMAT_ID);

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("endpoint", OUT_CHANNEL_ENDPOINT);
        configNode.put("topic", topicDescription.getName());

        ObjectNode tvmNode = mapper.createObjectNode();

        ObjectNode secretNode = mapper.createObjectNode();
        secretNode.put("env", TVM_SECRET);

        tvmNode.put("client_id", topicDescription.getTvmClientId());
        tvmNode.set("secret", secretNode);
        tvmNode.put("disk_cache_directory", DISK_CACHE_DIRECTORY_2);
        configNode.set("tvm", tvmNode);
        descriptionNode.set("config", configNode);
        return descriptionNode;
    }

    private JsonNode storagesNode(ObjectMapper mapper) {
        ArrayNode storagesNode = mapper.createArrayNode();
        ObjectNode storageNode = mapper.createObjectNode();

        ObjectNode configNode = mapper.createObjectNode();
        configNode.put("directory", STORAGE_DIRECTORY);
        configNode.put("max_partition_size", STORAGE_MAX_PARTITION_SIZE);
        configNode.put("max_segment_size", STORAGE_MAX_SEGMENT_SIZE);

        if (!isThrottlingEnabled()) {
            configNode.put("session_meta_mismatch_action", STORAGE_SESSION_META_MISMATCH_ACTION);
        }

        if (isThrottlingEnabled()) {
            configNode.put("unacknowledged_eviction_log_priority", STORAGE_UNACKNOWLEDGED_EVICTION_LOG_PRIORITY);
        }
        storageNode.put("name", STORAGE_NAME);
        storageNode.put("plugin", STORAGE_PLUGIN);
        storageNode.set("config", configNode);
        storagesNode.add(storageNode);
        return storagesNode;
    }

    private static ObjectNode labelNode(ObjectMapper mapper, String name, String value) {
        ObjectNode labelNode = mapper.createObjectNode();
        labelNode.put("name", name);
        labelNode.put("env", value);
        return labelNode;
    }

    private static ObjectNode keyValueNode(ObjectMapper mapper, String key, String value) {
        ObjectNode node = mapper.createObjectNode();
        node.put(key, value);
        return node;
    }

    @Override
    public String toJsonString() {
        ObjectMapper mapper = new ObjectMapper();

        ObjectNode confNode = mapper.createObjectNode();
        confNode.set("monitoring", monitoringNode(mapper));
        confNode.set("status", statusNode(mapper));
        confNode.set("routes", routesNode(mapper));
        confNode.set("storages", storagesNode(mapper));
        confNode.set("system", systemNode(mapper));

        if (isThrottlingEnabled()) {
            confNode.set("main_thread_pool", threadPoolNode(mapper));
        }

        if (throttlingPolicy == ThrottlingPolicy.THROTTLING_BEFORE_COMPRESSION) {
            confNode.set("flow_graph", flowGraphNode(mapper));
        }

        try {
            return mapper.writeValueAsString(confNode);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private boolean isThrottlingEnabled() {
        return throttlingPolicy != ThrottlingPolicy.DISABLED;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UnifiedAgentConfigV1 that = (UnifiedAgentConfigV1) o;
        return buildTaskId == that.buildTaskId
                && Objects.equal(staticSecret, that.staticSecret)
                && Objects.equal(stageId, that.stageId)
                && Objects.equal(deployUnitId, that.deployUnitId)
                && throttlingPolicy == that.throttlingPolicy
                && Objects.equal(throttlingLimits, that.throttlingLimits)
                && Objects.equal(clusters, that.clusters)
                && Objects.equal(topicDescription, that.topicDescription);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                buildTaskId,
                staticSecret,
                stageId,
                deployUnitId,
                throttlingPolicy,
                throttlingLimits,
                clusters,
                topicDescription
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("buildTaskId", buildTaskId)
                .add("staticSecret", staticSecret)
                .add("stageId", stageId)
                .add("deployUnitId", deployUnitId)
                .add("throttlingPolicy", throttlingPolicy)
                .add("throttlingLimits", throttlingLimits)
                .add("clusters", clusters)
                .add("topicDescription", topicDescription)
                .toString();
    }
}
