package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;

import ru.yandex.infra.stage.dto.LogbrokerTopicDescription;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;

import static ru.yandex.bolts.collection.Tuple2.tuple;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigV2.BooleanField.field;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigV2.LongField.field;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigV2.NodeField.field;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config.UnifiedAgentConfigV2.StringField.field;

/*
Second version of unified agent config generator

Examples (test/resources/logbroker/unified_agent):
- unified_agent_with_errorbooster

Contains three outputs:
- logs topic (communal / custom)
- http / syslog errorbooster topics

Main channel is separated from route/channel for config part

Throttling is permanent (not depends on any flags)

ErrorBooster topic
- is only communal now
- depends on UA release version / build meta flag
- is not affected by throttling
 */
public class UnifiedAgentConfigV2 implements UnifiedAgentConfig {
    private final static int LOGBROKER_MONITORING_PORT = LogbrokerPatcherUtils.LOGBROKER_AGENT_MONITORING_PORT;
    private final static int LOGBROKER_STATUS_PORT = 12502;
    private final static String TVM_SECRET = LogbrokerPatcherUtils.LOGBROKER_AGENT_TVM_SECRET_ENV_NAME;
    private final static List<Integer> SOLOMON_TVM_CLIENT_IDS = LogbrokerPatcherUtils.LOGBROKER_AGENT_SOLOMON_TVM_CLIENT_IDS;
    private final static String BUILD_TASK_ID = "BuildTaskId";
    private final static long HTTP_INPUT_PORT = 12520;
    private final static String HTTP_ROUTE_PATH = "/api/111/store/";
    public final static long SYSLOG_INPUT_PORT = 12521;
    public final static long SYSLOG_HTTP_PORT = 12522;
    private final static String SYSLOG_ADDRESS = "127.0.0.1:" + SYSLOG_INPUT_PORT;
    private final static String SYSLOG_HTTP_ROUTE_PATH = "/errorbooster";
    private final static String SYSLOG_FORMAT = "permissive";
    private final static String INFLIGHT_LIMIT = "30mb";
    private final static int INPUT_CHANNEL_INFLIGHT_MESSAGES_LIMIT = 100000;

    private final static int STORAGE_INFLIGHT_MESSAGES_LIMIT = 30000;
    private final static String GRPC_PLUGIN = "grpc";
    private final static String HTTP_PLUGIN = "http";
    private final static String SYSLOG_PLUGIN = "syslog";
    private final static String ASSIGN_PLUGIN = "assign";
    private final static String COMBINE_BACKTRACES_FILTER_PLUGIN = "combine_backtraces";
    private final static String COMPRESSION_FILTER_PLUGIN = "compress";
    private static final String BATCH_AND_COMPRESS_PIPE_NAME = "batch-and-compress";

    private enum CompressionCodecType {
        GZIP("gzip"), ZSTD("zstd");

        private final String name;

        CompressionCodecType(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    private final static int COMPRESSION_QUALITY = 6;
    private final static String YD_THROTTLE_FILTER_PLUGIN = "yd_throttle";
    private final static String COMBINE_BACKTRACES_FLUSH_PERIOD_MS = "300ms";
    private final static String STDOUT_LOGGGER_NAME = "stdout";
    private final static String STDERR_LOGGGER_NAME = "stderr";
    private final static String YD_FORMAT_FILTER_PLUGIN = "yd_format";
    private final static String YD_FORMAT_VERSION = "v2";
    private final static String BATCH_FILTER_PLUGIN = "batch";
    private final static String YD_BATCH_FILTER_DELIMETER = "\n";
    private final static String BATCH_FILTER_FLUSH_PERIOD_MS = "300ms";
    private final static String BATCH_FILTER_LIMIT_BYTES =  "100kb";
    private final static String STORAGE_DIRECTORY = "/push-agent/logs";
    private final static String STORAGE_MAX_PARTITION_SIZE = "1gb";
    private final static String STORAGE_MAX_SEGMENT_SIZE = "50mb";
    private final static String STORAGE_UNACKNOWLEDGED_EVICTION_LOG_PRIORITY = "WARNING";
    private final static String MAIN_STORAGE_NAME = "main";
    private final static String MAIN_STORAGE_PLUGIN = "fs";
    private final static String OUT_CHANNEL_PLUGIN = "logbroker";
    private final static String YD_OUT_CHANNEL_FORMAT_ID = "output-logbroker-yd-format-v2";
    private final static String OUT_CHANNEL_ENDPOINT = LogbrokerPatcherUtils.LOGBROKER_AGENT_OUT_CHANNEL_ENDPOINT;
    private final static String DISK_CACHE_DIRECTORY_1 = "/push-agent/internal/tvm-cache/monitoring";
    private final static String DISK_CACHE_DIRECTORY_2 = "/push-agent/internal/tvm-cache/logbroker-yd-format-v2";
    private final static String DEPLOY_PROJECT_LABEL_ENV_NAME = "DEPLOY_PROJECT";
    private final static String DEPLOY_STAGE_LABEL_ENV_NAME = "DEPLOY_STAGE";
    private final static String DEPLOY_DEPLOY_UNIT_LABEL_ENV_NAME = "DEPLOY_DEPLOY_UNIT";
    private final static String DEPLOY_PROJECT_META_VALUE = "{$env('DEPLOY_PROJECT')}";
    private final static String DEPLOY_STAGE_META_VALUE = "{$env('DEPLOY_STAGE')}";
    private final static String DEPLOY_DEPLOY_UNIT_META_VALUE = "{$env('DEPLOY_DEPLOY_UNIT')}";
    private final static String DEPLOY_POD_META_VALUE = "{$env('DEPLOY_POD_ID')}";
    private final static String DEPLOY_POD_TRANSIENT_FQDN_META_VALUE = "{$env('DEPLOY_POD_TRANSIENT_FQDN')}";
    private final static String DEPLOY_POD_PERSISTENT_FQDN_META_VALUE = "{$env('DEPLOY_POD_PERSISTENT_FQDN')}";
    private final static String DEPLOY_NODE_FQDN_META_VALUE = "{$env('DEPLOY_NODE_FQDN')}";
    private final static String RESOLVE_ENDPOINTS_URL = "http://sd.yandex.net:8080/resolve_endpoints/json";
    private final static String YD_THROTTLE_POLL_PERIOD = "1h";
    private final static String YD_THROTTLE_POLL_TIME_RANDOM_SHIFT = "15m";
    private final static String YD_THROTTLE_CACHE_FILENAME = "/push-agent/internal/unified_agent_pods_count";
    private final static String YD_THROTTLE_CLIENT_NAME_PREFIX = "deploy_logs_stage_discovery";
    private final static int NUM_OF_THREADS = 3;
    private final static String FLOW_GRAPH_MEMORY_QUOTA_MB = "150mb";
    private final static String TVM_CLIENT_SERVICE_TYPE = "tvm_client";
    private final static String TVM_CLIENT_SERVICE_NAME = "yd-tvm";
    private final static String FILTER_FIELD_NAME = "filter";
    private final static String PLUGIN_FIELD_NAME = "plugin";
    private final static String CHANNEL_FIELD_NAME = "channel";
    private final static String PIPE_FIELD_NAME = "pipe";
    private final static String MAIN_CHANNEL_NAME = "main-channel";
    private final static String ERROR_BOOSTER_LOG_TYPE_NAME = "errorbooster";
    private final static String CONFIG_FIELD_NAME = "config";
    private final static String OUTPUT_FIELD_NAME = "output";
    private final static String ERROR_BOOSTER_OUTPUT_PLUGIN_ID_FORMAT = "output-logbroker-errorbooster-%s";
    private final static String ERROR_BOOSTER_TOPIC_FORMAT = "error-booster/deploy/%s-errors";
    private static final String ERROR_BOOSTER_UNIVERSAL_TOPIC_TYPE = "universal";
    private static final String ERROR_BOOSTER_SENTRY_TOPIC_TYPE = "sentry";
    private static final String GRPC_LOG_TYPE_NAME = "grpc";

    private final static String SENTRY_LOG_TYPE_NAME = "sentry";
    private final static String INPUT_FIELD = "input";


    private final static String ENV_FIELD = "env";
    private final static String NAME_FIELD = "name";
    private final static String PORT_FIELD = "port";
    private final static String SESSION_FIELD_NAME = "session";
    private final static String FORMAT_FIELD = "format";
    private final static String LOG_TYPE_FIELD = "log_type";
    private final static String LOG_TYPE_DEF_FIELD = "log_type_def";
    private final static String CHANNEL_REF_FIELD = "channel_ref";
    private final static boolean LOCK_EXECUTABLE_IN_MEMORY = true;
    private final static String FLUSH_PERIOD_FIELD_NAME = "flush_period";
    private final static String FLOW_CONTROL_FIELD_NAME = "flow_control";

    private final static List<String> TVM_DESTINATIONS = List.of("logbroker");

    private final long buildTaskId;
    private final String staticSecret;
    private final String stageId;
    private final String deployUnitId;
    private final ThrottlingLimits throttlingLimits;
    private final Optional<DataRetentionLimits> dataRetentionLimitsOptional;
    private final Set<String> clusters;
    private final LogbrokerTopicDescription topicDescription;

    /**
     Constant values should be constants of that class
     and not variables of constructor
     Unified agent config is very complex and it is safer to have tests
     that are nearest to prod config as much as possible
     */
    @VisibleForTesting
    UnifiedAgentConfigV2(
            long buildTaskId,
            String staticSecret,
            String stageId,
            String deployUnitId,
            ThrottlingLimits throttlingLimits,
            Optional<DataRetentionLimits> dataRetentionLimitsOptional,
            Set<String> clusters,
            LogbrokerTopicDescription topicDescription) {
        this.buildTaskId = buildTaskId;
        this.staticSecret = staticSecret;
        this.stageId = stageId;
        this.deployUnitId = deployUnitId;
        this.throttlingLimits = throttlingLimits;
        this.dataRetentionLimitsOptional = dataRetentionLimitsOptional;
        this.clusters = clusters;
        this.topicDescription = topicDescription;
    }

    private static ArrayNode arrayNode(ObjectMapper mapper,
                                       Iterable<JsonNode> nodes) {
        ArrayNode arrayNode = mapper.createArrayNode();
        nodes.forEach(arrayNode::add);
        return arrayNode;
    }

    @FunctionalInterface
    interface Field {
        void injectTo(ObjectNode parentNode);
    }

    static final class OptionalField implements Field {

        static <T> OptionalField of(
                String fieldName,
                Optional<T> valueOptional,
                BiFunction<String, T, Field> fieldFactory
        ) {
            return new OptionalField(
                    valueOptional.map(
                            value -> fieldFactory.apply(fieldName, value)
                    )
            );
        }

        private final Optional<Field> fieldOptional;

        private OptionalField(Optional<Field> fieldOptional) {
            this.fieldOptional = fieldOptional;
        }

        @Override
        public void injectTo(ObjectNode parentNode) {
            fieldOptional.ifPresent(
                    field -> field.injectTo(parentNode)
            );
        }
    }

    static abstract class FieldImpl implements Field {

        private final String name;

        FieldImpl(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    static final class StringField extends FieldImpl {

        static StringField field(String name, String value) {
            return new StringField(name, value);
        }

        private final String value;

        private StringField(String name, String value) {
            super(name);
            this.value = value;
        }

        @Override
        public void injectTo(ObjectNode parentNode) {
            parentNode.put(getName(), value);
        }
    }

    static final class LongField extends FieldImpl {

        static LongField field(String name, long value) {
            return new LongField(name, value);
        }

        private final long value;

        private LongField(String name, long value) {
            super(name);
            this.value = value;
        }

        @Override
        public void injectTo(ObjectNode parentNode) {
            parentNode.put(getName(), value);
        }
    }

    static final class BooleanField extends FieldImpl {

        static BooleanField field(String name, boolean value) {
            return new BooleanField(name, value);
        }

        private final boolean value;

        private BooleanField(String name, boolean value) {
            super(name);
            this.value = value;
        }

        @Override
        public void injectTo(ObjectNode parentNode) {
            parentNode.put(getName(), value);
        }
    }

    static final class NodeField extends FieldImpl {

        static NodeField field(String name,
                               JsonNode node) {
            return new NodeField(name, node);
        }

        private final JsonNode node;

        private NodeField(String name, JsonNode node) {
            super(name);
            this.node = node;
        }

        @Override
        public void injectTo(ObjectNode parentNode) {
            parentNode.set(getName(), node);
        }
    }

    private static void setFields(ObjectNode parentNode,
                                  Iterable<Field> fields) {
        fields.forEach(field -> field.injectTo(parentNode));
    }

    private static JsonNode nodeWithFields(ObjectMapper mapper,
                                           Iterable<Field> fields) {
        ObjectNode parentNode = mapper.createObjectNode();
        setFields(parentNode, fields);
        return parentNode;
    }

    private static JsonNode nodeWithField(ObjectMapper mapper, Field field) {
        return nodeWithFields(
                mapper,
                List.of(field)
        );
    }

    private static JsonNode wrapperNode(ObjectMapper mapper, String wrapperName, JsonNode innerNode) {
        return nodeWithField(mapper, field(wrapperName, innerNode));
    }

    private JsonNode tvmNode(ObjectMapper mapper) {
        ArrayNode solomonClientIdsNode = mapper.createArrayNode();
        SOLOMON_TVM_CLIENT_IDS.forEach(solomonClientIdsNode::add);

        return nodeWithFields(
                mapper,
                List.of(
                        field("client_id", topicDescription.getTvmClientId()),
                        field("allowed_clients", solomonClientIdsNode),
                        field("disk_cache_directory", DISK_CACHE_DIRECTORY_1)
                )
        );
    }

    private static StringField nameField(String name) {
        return field(NAME_FIELD, name);
    }

    private static JsonNode labelNode(ObjectMapper mapper, String name, String envValue) {
        return nodeWithFields(
                mapper,
                List.of(
                        nameField(name),
                        field(ENV_FIELD, envValue)
                )
        );
    }

    private static JsonNode staticLabelsNode(ObjectMapper mapper) {
        var labelNodes = List.of(
                tuple("Project", DEPLOY_PROJECT_LABEL_ENV_NAME),
                tuple("Stage", DEPLOY_STAGE_LABEL_ENV_NAME),
                tuple("DeployUnit", DEPLOY_DEPLOY_UNIT_LABEL_ENV_NAME)
        ).stream().map(
                nameEnvTuple -> labelNode(mapper, nameEnvTuple.get1(), nameEnvTuple.get2())
        ).collect(Collectors.toList());

        return arrayNode(mapper, labelNodes);
    }

    private JsonNode buildTaskStaticSensorNode(ObjectMapper mapper){
        return nodeWithFields(
                mapper,
                List.of(
                        nameField(BUILD_TASK_ID),
                        field("value", buildTaskId)
                )
        );
    }

    private JsonNode staticSensorsNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        buildTaskStaticSensorNode(mapper)
                )
        );
    }

    private JsonNode monitoringNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field(PORT_FIELD, LOGBROKER_MONITORING_PORT),
                        field("tvm", tvmNode(mapper)),
                        field("static_labels", staticLabelsNode(mapper)),
                        field("static_sensors", staticSensorsNode(mapper))
                )
        );
    }

    private static JsonNode statusNode(ObjectMapper mapper) {
        return nodeWithField(mapper, field(PORT_FIELD, LOGBROKER_STATUS_PORT));
    }

    private static JsonNode tvmClientServiceFetchNode(ObjectMapper mapper) {
        var secretNode = nodeWithField(mapper, field("env", TVM_SECRET));

        var destinationsNode = mapper.createArrayNode();
        TVM_DESTINATIONS.forEach(destinationsNode::add);

        return nodeWithFields(
                mapper,
                List.of(
                        field("secret", secretNode),
                        field("destinations", destinationsNode)
                )
        );
    }

    private JsonNode tvmClientServiceConfigNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field("client_id", topicDescription.getTvmClientId()),
                        field("disk_cache_directory", DISK_CACHE_DIRECTORY_2),
                        field("fetch", tvmClientServiceFetchNode(mapper))
                )
        );
    }

    private JsonNode tvmClientServiceNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field("type", TVM_CLIENT_SERVICE_TYPE),
                        nameField(TVM_CLIENT_SERVICE_NAME),
                        field(CONFIG_FIELD_NAME, tvmClientServiceConfigNode(mapper))
                )
        );
    }

    private JsonNode servicesNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        tvmClientServiceNode(mapper)
                )
        );
    }

    private static JsonNode ydBatchFilterNode(ObjectMapper mapper) {
        return nodeWithPluginAndConfig(
                mapper,
                BATCH_FILTER_PLUGIN,
                batchFilterConfigNode(
                        mapper,
                        field("delimiter", YD_BATCH_FILTER_DELIMETER)
                )
        );
    }

    private static JsonNode compressGzipFilterNode(ObjectMapper mapper) {
        return nodeWithPluginAndConfig(
                mapper,
                COMPRESSION_FILTER_PLUGIN,
                compressGzipConfigNode(mapper)
        );
    }

    private static JsonNode batchAndCompressFiltersNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        filterWrappedNode(mapper, ydBatchFilterNode(mapper)),
                        filterWrappedNode(mapper, compressGzipFilterNode(mapper))
                )
        );
    }

    private static NodeField pipeNodeField(JsonNode pipesNode) {
        return field(PIPE_FIELD_NAME, pipesNode);
    }

    private static JsonNode batchAndCompressPipeNode(ObjectMapper mapper) {
        var pipeNode = batchAndCompressFiltersNode(mapper);

        return nodeWithFields(
                mapper,
                List.of(
                        nameField(BATCH_AND_COMPRESS_PIPE_NAME),
                        pipeNodeField(pipeNode)
                )
        );
    }

    private static JsonNode pipesNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        batchAndCompressPipeNode(mapper)
                )
        );
    }

    private static JsonNode storageRefNode(ObjectMapper mapper) {
        var inflightNode = nodeWithFields(
                mapper,
                List.of(
                        field("limit", INFLIGHT_LIMIT),
                        field("limit_messages", STORAGE_INFLIGHT_MESSAGES_LIMIT)
                )
        );

        return nodeWithFields(
                mapper,
                List.of(
                        nameField("main"),
                        field(FLOW_CONTROL_FIELD_NAME, wrapperNode(mapper, "inflight", inflightNode))
                )
        );
    }

    private static JsonNode filterWrappedNode(ObjectMapper mapper,
                                              JsonNode filterNode) {
        return wrapperNode(mapper, FILTER_FIELD_NAME, filterNode);
    }

    private static JsonNode wrappedSessionNode(ObjectMapper mapper, Collection<StringField> envVariableFields) {
        var envVariableNodes = envVariableFields.stream()
                .map(stringField -> nodeWithField(mapper, stringField))
                .collect(Collectors.toList());

        JsonNode sessionNode = arrayNode(
                mapper, envVariableNodes
        );

        return wrapperNode(mapper, SESSION_FIELD_NAME, sessionNode);
    }

    private static JsonNode assignFilterConfigNode(ObjectMapper mapper) {
        return wrappedSessionNode(
                mapper,
                List.of(
                        field("deploy_project", DEPLOY_PROJECT_META_VALUE),
                        field("deploy_stage", DEPLOY_STAGE_META_VALUE),
                        field("deploy_unit", DEPLOY_DEPLOY_UNIT_META_VALUE),
                        field("deploy_pod", DEPLOY_POD_META_VALUE),
                        field("deploy_pod_transient_fqdn", DEPLOY_POD_TRANSIENT_FQDN_META_VALUE),
                        field("deploy_pod_persistent_fqdn", DEPLOY_POD_PERSISTENT_FQDN_META_VALUE),
                        field("deploy_node_fqdn", DEPLOY_NODE_FQDN_META_VALUE)
                )
        );
    }

    private static StringField pluginField(String pluginName) {
        return field(PLUGIN_FIELD_NAME, pluginName);
    }

    private static JsonNode nodeWithPluginAndConfig(ObjectMapper mapper,
                                                    String pluginName,
                                                    JsonNode configNode) {
        return nodeWithFields(
                mapper,
                List.of(
                        pluginField(pluginName),
                        field(CONFIG_FIELD_NAME, configNode)
                )
        );
    }

    private static JsonNode ydFormatAssignFilterNode(ObjectMapper mapper) {
        return nodeWithPluginAndConfig(
                mapper,
                ASSIGN_PLUGIN,
                assignFilterConfigNode(mapper)
        );
    }

    private static JsonNode combineBacktracesEnableIfNode(ObjectMapper mapper) {
        ArrayNode loggerNamesNode = mapper.createArrayNode();
        loggerNamesNode.add(STDOUT_LOGGGER_NAME);
        loggerNamesNode.add(STDERR_LOGGGER_NAME);

        var sessionNode = wrapperNode(mapper, "deploy_logger_name", loggerNamesNode);
        return wrapperNode(mapper, SESSION_FIELD_NAME, sessionNode);
    }

    private static JsonNode combineBacktracesConfigNode(ObjectMapper mapper) {
        return nodeWithField(mapper, field(FLUSH_PERIOD_FIELD_NAME, COMBINE_BACKTRACES_FLUSH_PERIOD_MS));
    }

    private static JsonNode combineBacktracesFilterNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        pluginField(COMBINE_BACKTRACES_FILTER_PLUGIN),
                        field("enable_if", combineBacktracesEnableIfNode(mapper)),
                        field(CONFIG_FIELD_NAME, combineBacktracesConfigNode(mapper))
                )
        );
    }

    private JsonNode ydThrottlingRateAdjustmentNode(ObjectMapper mapper) {
        ArrayNode clusterNameNode = mapper.createArrayNode();
        clusters.forEach(clusterNameNode::add);

        return nodeWithFields(
                mapper,
                List.of(
                        field("resolve_endpoints_url", RESOLVE_ENDPOINTS_URL),
                        field("poll_period", YD_THROTTLE_POLL_PERIOD),
                        field("poll_time_random_shift", YD_THROTTLE_POLL_TIME_RANDOM_SHIFT),
                        field("cluster_names", clusterNameNode),
                        field("endpoint_set_id", String.format("%s.%s", stageId, deployUnitId)),
                        field("client_name", String.format("%s_%s", YD_THROTTLE_CLIENT_NAME_PREFIX, stageId)),
                        field("cache_file_name", YD_THROTTLE_CACHE_FILENAME)
                )
        );
    }

    private JsonNode ydThrottlingConfigNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field("stage_rate_limit", throttlingLimits.getMaxRate()),
                        field("stage_messages_rate_limit", throttlingLimits.getMaxMessagesRate()),
                        field("rate_adjustment", ydThrottlingRateAdjustmentNode(mapper))
                )
        );
    }

    private JsonNode ydThrottlingFilterNode(ObjectMapper mapper) {
        return nodeWithPluginAndConfig(
                mapper,
                YD_THROTTLE_FILTER_PLUGIN,
                ydThrottlingConfigNode(mapper)
        );
    }

    private static JsonNode assignSessionLogTypeFilterNode(ObjectMapper mapper,
                                                           String logTypeFieldName,
                                                           String logTypeValue) {
        var configNode = wrappedSessionNode(
                mapper,
                List.of(
                        field(logTypeFieldName, logTypeValue)
                )
        );

        return nodeWithPluginAndConfig(mapper, ASSIGN_PLUGIN, configNode);
    }

    private static ArrayNode mainChannelPipesNode(ObjectMapper mapper) {
        var logTypeValue = String.format("{%s|%s}", LOG_TYPE_FIELD, GRPC_LOG_TYPE_NAME);
        var assignFilterNode = assignSessionLogTypeFilterNode(mapper, LOG_TYPE_DEF_FIELD, logTypeValue);

        return arrayNode(
                mapper,
                List.of(
                        wrapperNode(mapper, "storage_ref", storageRefNode(mapper)),
                        filterWrappedNode(mapper, combineBacktracesFilterNode(mapper)),
                        filterWrappedNode(mapper, assignFilterNode)
                )
        );
    }

    private static JsonNode batchFilterConfigNode(ObjectMapper mapper,
                                                  Field delimiterField) {
        var limitNode = nodeWithField(mapper, field("bytes", BATCH_FILTER_LIMIT_BYTES));

        return nodeWithFields(
                mapper,
                List.of(
                        delimiterField,
                        field(FLUSH_PERIOD_FIELD_NAME, BATCH_FILTER_FLUSH_PERIOD_MS),
                        field("limit", limitNode)
                )
        );
    }

    private static JsonNode compressGzipConfigNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field("codec", CompressionCodecType.GZIP.getName()),
                        field("compression_quality", COMPRESSION_QUALITY)
                )
        );
    }

    private static JsonNode batchAndCompressPipeRefNode(ObjectMapper mapper) {
        var pipeRefNode = nodeWithField(mapper, nameField(BATCH_AND_COMPRESS_PIPE_NAME));
        return wrapperNode(mapper, "pipe_ref", pipeRefNode);
    }

    private static ArrayNode errorBoosterFiltersNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        batchAndCompressPipeRefNode(mapper)
                )
        );
    }

    private static JsonNode outputConfigNode(ObjectMapper mapper, String topicName) {
        var tvmRefNode = nodeWithField(mapper, field("name", TVM_CLIENT_SERVICE_NAME));

        return nodeWithFields(
                mapper,
                List.of(
                        field("endpoint", OUT_CHANNEL_ENDPOINT),
                        field("topic", topicName),
                        field("tvm_ref", tvmRefNode)
                )
        );
    }

    private static JsonNode outputNode(ObjectMapper mapper, String formatId, String topicName) {
        return nodeWithFields(
                mapper,
                List.of(
                        pluginField(OUT_CHANNEL_PLUGIN),
                        field("id", formatId),
                        field(CONFIG_FIELD_NAME, outputConfigNode(mapper, topicName))
                )
        );
    }

    private static NodeField outputNodeField(ObjectMapper mapper, String formatId, String topicName) {
        return field(OUTPUT_FIELD_NAME, outputNode(mapper, formatId, topicName));
    }

    private static JsonNode errorBoosterChannelFieldNode(ObjectMapper mapper, String topicType) {
        return nodeWithFields(
                mapper,
                List.of(
                        pipeNodeField(errorBoosterFiltersNode(mapper)),
                        outputNodeField(
                                mapper,
                                String.format(ERROR_BOOSTER_OUTPUT_PLUGIN_ID_FORMAT, topicType),
                                String.format(ERROR_BOOSTER_TOPIC_FORMAT, topicType)
                        )
                )
        );
    }

    private static JsonNode channelWithWhenSessionLogTypeRefNode(ObjectMapper mapper,
                                                                 String logTypeName,
                                                                 JsonNode channelNode) {
        var sessionNode = nodeWithField(mapper, field(LOG_TYPE_DEF_FIELD, logTypeName));

        return nodeWithFields(
                mapper,
                List.of(
                        field("when", wrapperNode(mapper, SESSION_FIELD_NAME, sessionNode)),
                        field(CHANNEL_FIELD_NAME, channelNode)
                )
        );
    }

    private static JsonNode ydFormatFilterConfigNode(ObjectMapper mapper) {
        return nodeWithField(mapper, field("format_version", YD_FORMAT_VERSION));
    }

    private static JsonNode ydFormatFilterNode(ObjectMapper mapper) {
        return nodeWithPluginAndConfig(
                mapper,
                YD_FORMAT_FILTER_PLUGIN,
                ydFormatFilterConfigNode(mapper)
        );
    }

    private ArrayNode ydFormatPipesNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        filterWrappedNode(mapper, ydThrottlingFilterNode(mapper)),
                        filterWrappedNode(mapper, ydFormatAssignFilterNode(mapper)),
                        filterWrappedNode(mapper, ydFormatFilterNode(mapper)),
                        batchAndCompressPipeRefNode(mapper)
                )
        );
    }

    private JsonNode ydFormatChannelFieldNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                       pipeNodeField(ydFormatPipesNode(mapper)),
                       outputNodeField(mapper, YD_OUT_CHANNEL_FORMAT_ID, topicDescription.getName())
                )
        );
    }

    private JsonNode mainChannelFormatCasesNode(ObjectMapper mapper) {
        var caseNodes = List.of(
                tuple(ERROR_BOOSTER_LOG_TYPE_NAME,
                        errorBoosterChannelFieldNode(mapper, ERROR_BOOSTER_UNIVERSAL_TOPIC_TYPE)
                ),
                tuple(SENTRY_LOG_TYPE_NAME,
                        errorBoosterChannelFieldNode(mapper, ERROR_BOOSTER_SENTRY_TOPIC_TYPE)
                ),
                tuple(GRPC_LOG_TYPE_NAME,
                        ydFormatChannelFieldNode(mapper)
                )
        ).stream().map(logTypeAndChannel ->
                channelWithWhenSessionLogTypeRefNode(
                        mapper,
                        logTypeAndChannel.get1(),
                        logTypeAndChannel.get2()
                )
        ).collect(Collectors.toList());

        return arrayNode(
                mapper,
                caseNodes
        );
    }

    private JsonNode mainChannelFieldNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        pipeNodeField(mainChannelPipesNode(mapper)),
                        field("case", mainChannelFormatCasesNode(mapper))
                )
        );
    }

    private JsonNode mainChannelNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        nameField(MAIN_CHANNEL_NAME),
                        field(CHANNEL_FIELD_NAME, mainChannelFieldNode(mapper))
                )
        );
    }

    private JsonNode channelsNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        mainChannelNode(mapper)
                )
        );
    }

    private static JsonNode routeNode(ObjectMapper mapper,
                                      JsonNode inputNode,
                                      JsonNode channelNode) {
        return nodeWithFields(
                mapper,
                List.of(
                        field(INPUT_FIELD, inputNode),
                        field(CHANNEL_FIELD_NAME, channelNode)
                )
        );
    }

    private JsonNode grpcConfigNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field("uri", GRPC_ENDPOINT_URI),
                        field("shared_secret_key", staticSecret)
                )
        );
    }

    private static JsonNode grpcFlowControlNode(ObjectMapper mapper) {
        var inflightNode = nodeWithFields(
                mapper,
                List.of(
                        field("limit", INFLIGHT_LIMIT),
                        field("limit_messages", INPUT_CHANNEL_INFLIGHT_MESSAGES_LIMIT)
                )
        );

        return wrapperNode(mapper, "inflight", inflightNode);
    }

    private JsonNode grpcInputNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        pluginField(GRPC_PLUGIN),
                        field(CONFIG_FIELD_NAME, grpcConfigNode(mapper)),
                        field(FLOW_CONTROL_FIELD_NAME, grpcFlowControlNode(mapper))
                )
        );
    }

    private static JsonNode routeChannelRefNode(ObjectMapper mapper) {
        return nodeWithField(mapper, nameField(MAIN_CHANNEL_NAME));
    }


    private static JsonNode routeChannelPipesNode(ObjectMapper mapper, String formatName) {
        var assignFilterNode = assignSessionLogTypeFilterNode(mapper, LOG_TYPE_FIELD, formatName);

        return arrayNode(
                mapper,
                List.of(
                        filterWrappedNode(mapper, assignFilterNode)
                )
        );
    }

    private static JsonNode routeChannelNode(ObjectMapper mapper, String logTypeName) {
        return nodeWithFields(
                mapper,
                List.of(
                        pipeNodeField(routeChannelPipesNode(mapper, logTypeName)),
                        field(CHANNEL_REF_FIELD, routeChannelRefNode(mapper))
                )
        );
    }

    private static JsonNode syslogConfigNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        field("address", SYSLOG_ADDRESS),
                        field(FORMAT_FIELD, SYSLOG_FORMAT)
                )
        );
    }

    private static JsonNode syslogHttpConfigNode(ObjectMapper mapper, String path, long port) {
        return nodeWithFields(
                mapper,
                List.of(
                        field(PORT_FIELD, port),
                        field("path", path)
                )
        );
    }

    private static JsonNode syslogInputNode(ObjectMapper mapper) {
        return nodeWithPluginAndConfig(mapper, SYSLOG_PLUGIN, syslogConfigNode(mapper));
    }

    private static JsonNode syslogHttpInputNode(ObjectMapper mapper, String path, long port) {
        return nodeWithPluginAndConfig(mapper, HTTP_PLUGIN, syslogHttpConfigNode(mapper, path, port));
    }

    private JsonNode routesNode(ObjectMapper mapper) {
        var routeNodes = List.of(
                tuple(
                        grpcInputNode(mapper),
                        GRPC_LOG_TYPE_NAME
                ),
                tuple(
                        syslogHttpInputNode(mapper, HTTP_ROUTE_PATH, HTTP_INPUT_PORT),
                        SENTRY_LOG_TYPE_NAME
                ),
                tuple(
                        syslogInputNode(mapper),
                        ERROR_BOOSTER_LOG_TYPE_NAME
                ),
                tuple(
                        syslogHttpInputNode(mapper, SYSLOG_HTTP_ROUTE_PATH, SYSLOG_HTTP_PORT),
                        ERROR_BOOSTER_LOG_TYPE_NAME
                )
        ).stream().map(generatorsTuple ->
                routeNode(
                        mapper,
                        generatorsTuple.get1(),
                        routeChannelNode(mapper, generatorsTuple.get2())
                )
        ).collect(Collectors.toList());

        return arrayNode(
                mapper,
                routeNodes
        );
    }

    private static JsonNode dataRetentionNode(ObjectMapper mapper,
                                              DataRetentionLimits limits) {
        var ageField = OptionalField.of(
                "by_age",
                limits.getAgeLimit(),
                StringField::field
        );

        var sizeField = OptionalField.of(
                "by_size",
                limits.getSizeLimit(),
                StringField::field
        );

        return nodeWithFields(
                mapper,
                List.of(
                        ageField,
                        sizeField
                )
        );
    }

    private Optional<JsonNode> dataRetentionNodeOptional(ObjectMapper mapper){
        return dataRetentionLimitsOptional.map(
                dataRetentionLimits -> dataRetentionNode(
                        mapper,
                        dataRetentionLimits
                )
        );
    }

    private JsonNode mainStorageConfigNode(ObjectMapper mapper) {
        var dataRetentionField = OptionalField.of(
                "data_retention",
                dataRetentionNodeOptional(mapper),
                NodeField::field
        );

        return nodeWithFields(
                mapper,
                List.of(
                        field("directory", STORAGE_DIRECTORY),
                        field("max_partition_size", STORAGE_MAX_PARTITION_SIZE),
                        field("max_segment_size", STORAGE_MAX_SEGMENT_SIZE),
                        dataRetentionField,
                        field("unacknowledged_eviction_log_priority", STORAGE_UNACKNOWLEDGED_EVICTION_LOG_PRIORITY)
                )
        );
    }

    private JsonNode mainStorageNode(ObjectMapper mapper) {
        return nodeWithFields(
                mapper,
                List.of(
                        nameField(MAIN_STORAGE_NAME),
                        pluginField(MAIN_STORAGE_PLUGIN),
                        field(CONFIG_FIELD_NAME, mainStorageConfigNode(mapper))
                )
        );
    }

    private JsonNode storagesNode(ObjectMapper mapper) {
        return arrayNode(
                mapper,
                List.of(
                        mainStorageNode(mapper)
                )
        );
    }

    private static JsonNode systemNode(ObjectMapper mapper) {
        return nodeWithField(mapper, field("lock_executable_in_memory", LOCK_EXECUTABLE_IN_MEMORY));
    }

    private static JsonNode threadPoolNode(ObjectMapper mapper) {
        return nodeWithField(mapper, field("threads", NUM_OF_THREADS));
    }

    private static JsonNode flowGraphNode(ObjectMapper mapper) {
        var memoryQuotaNode = nodeWithField(mapper, field("limit", FLOW_GRAPH_MEMORY_QUOTA_MB));
        return wrapperNode(mapper, "memory_quota", memoryQuotaNode);
    }

    @Override
    public String toJsonString() {
        ObjectMapper mapper = new ObjectMapper();

        var unifiedAgentConfigNode = nodeWithFields(
                mapper,
                List.of(
                        field("monitoring", monitoringNode(mapper)),
                        field("status", statusNode(mapper)),
                        field("services", servicesNode(mapper)),
                        field("pipes", pipesNode(mapper)),
                        field("channels", channelsNode(mapper)),
                        field("routes", routesNode(mapper)),
                        field("storages", storagesNode(mapper)),
                        field("system", systemNode(mapper)),
                        field("main_thread_pool", threadPoolNode(mapper)),
                        field("flow_graph", flowGraphNode(mapper))
                )
        );

        try {
            return mapper.writeValueAsString(unifiedAgentConfigNode);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UnifiedAgentConfigV2 that = (UnifiedAgentConfigV2) o;
        return buildTaskId == that.buildTaskId
                && Objects.equal(staticSecret, that.staticSecret)
                && Objects.equal(stageId, that.stageId)
                && Objects.equal(deployUnitId, that.deployUnitId)
                && Objects.equal(throttlingLimits, that.throttlingLimits)
                && Objects.equal(clusters, that.clusters)
                && Objects.equal(topicDescription, that.topicDescription);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(
                buildTaskId,
                staticSecret,
                stageId,
                deployUnitId,
                throttlingLimits,
                clusters,
                topicDescription
        );
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static final class Builder {
        private long buildTaskId;
        private final String staticSecret;
        private final String stageId;
        private final String deployUnitId;
        private final ThrottlingLimits throttlingLimits;
        private Optional<DataRetentionLimits> dataRetentionLimitsOptional;
        private final Set<String> clusters;
        private final LogbrokerTopicDescription topicDescription;

        private Builder(UnifiedAgentConfigV2 config) {
            this.buildTaskId = config.buildTaskId;
            this.staticSecret = config.staticSecret;
            this.stageId = config.stageId;
            this.deployUnitId = config.deployUnitId;
            this.throttlingLimits = config.throttlingLimits;
            this.dataRetentionLimitsOptional = config.dataRetentionLimitsOptional;
            this.clusters = config.clusters;
            this.topicDescription = config.topicDescription;
        }

        Builder withBuildTaskId(long buildTaskId) {
            this.buildTaskId = buildTaskId;
            return this;
        }

        Builder withDataRetentionLimitsOptional(Optional<DataRetentionLimits> dataRetentionLimitsOptional) {
            this.dataRetentionLimitsOptional = dataRetentionLimitsOptional;
            return this;
        }

        UnifiedAgentConfigV2 build() {
            return new UnifiedAgentConfigV2(
                    buildTaskId,
                    staticSecret,
                    stageId,
                    deployUnitId,
                    throttlingLimits,
                    dataRetentionLimitsOptional,
                    clusters,
                    topicDescription
            );
        }
    }
}
