package ru.yandex.infra.stage.podspecs.patcher.logbroker.unified_agent_config;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import one.util.streamex.EntryStream;

final class VersionCalculatorImpl implements VersionCalculator {

    private final Map<UnifiedAgentConfigVersion, Long> versionToFirstAffectedUnifiedAgentVersion;
    private final UnifiedAgentConfigVersion defaultVersion;

    @VisibleForTesting
    VersionCalculatorImpl(
            Map<UnifiedAgentConfigVersion, Long> versionToFirstAffectedUnifiedAgentVersion,
            UnifiedAgentConfigVersion defaultVersion
    ) {
        this.versionToFirstAffectedUnifiedAgentVersion = Map.copyOf(versionToFirstAffectedUnifiedAgentVersion);
        this.defaultVersion = defaultVersion;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        VersionCalculatorImpl that = (VersionCalculatorImpl) o;
        return Objects.equals(versionToFirstAffectedUnifiedAgentVersion, that.versionToFirstAffectedUnifiedAgentVersion)
                && defaultVersion == that.defaultVersion;
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                versionToFirstAffectedUnifiedAgentVersion,
                defaultVersion
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("versionToFirstAffectedUnifiedAgentVersion", versionToFirstAffectedUnifiedAgentVersion)
                .add("defaultVersion", defaultVersion)
                .toString();
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {

        private final Map<UnifiedAgentConfigVersion, Long> versionToFirstAffectedUnifiedAgentVersion;
        private final UnifiedAgentConfigVersion defaultVersion;

        private Builder(
                VersionCalculatorImpl calculator
        ) {
            this.versionToFirstAffectedUnifiedAgentVersion = calculator.versionToFirstAffectedUnifiedAgentVersion;
            this.defaultVersion = calculator.defaultVersion;
        }

        @VisibleForTesting
        VersionCalculatorImpl build() {
            return new VersionCalculatorImpl(
                    versionToFirstAffectedUnifiedAgentVersion,
                    defaultVersion
            );
        }
    }

    private Optional<UnifiedAgentConfigVersion> calculateMaximalAffectedVersion(long releaseVersion) {
        return EntryStream.of(versionToFirstAffectedUnifiedAgentVersion)
                .filterValues(firstAffectedVersion -> firstAffectedVersion <= releaseVersion)
                .maxBy(Map.Entry::getValue)
                .map(Map.Entry::getKey);
    }

    @Override
    public UnifiedAgentConfigVersion calculate(
            boolean errorBoosterIsEnabled,
            Optional<Long> releaseVersionOptional
    ) {
        return releaseVersionOptional
                .flatMap(this::calculateMaximalAffectedVersion)
                .orElse(defaultVersion);
    }
}
