package ru.yandex.infra.stage.podspecs.patcher.logrotate;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.protobuf.ByteString;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.LogrotateConfig;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.EVolumeMountMode;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TBoxOrBuilder;
import ru.yandex.yp.client.pods.TComputeResources;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TMountedStaticResource;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TRootfsVolume;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.MEGABYTE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addLayerIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addStaticResourceIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addVolumeIfAbsent;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.buildMutableWorkload;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.rawStaticResource;

public abstract class LogrotatePatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    private static final String WORKLOAD_ID = "logrotate";

    private static final String LOGROTATE_STDOUT = "logrotate.log";

    private static final String LOGROTATE_STDERR = "logrotate.err.log";

    @VisibleForTesting
    static final List<String> POSSIBLE_VAR_LOG_MOUNT_POINTS = ImmutableList.of("var/log", "/var/log", "var/log/", "/var/log/");

    @VisibleForTesting
    static final String LOG_ROTATE_CONFIG_RESOURCE_ID = "logrotate_conf_resource";

    private static final String LOG_ROTATE_LOG_VOLUME_ID = "logrotate_log_volume";

    private static final String LOG_ROTATE_CONFIG_FILE_NAME = "box";

    private static final String LOG_ROTATE_CONFIG_MOUNT_POINT = "/etc/logrotate.d";

    @VisibleForTesting
    static final String LOG_ROTATE_APP_LAYER_ID = "logrotate_application_layer";

    private static final String INIT_D_DESTINATION_PATH = "/usr/logrotate";

    public static final String LOG_ROTATE_LOG_MOUNT_POINT = "/var/log/logrotate";

    private static final String INIT_D_LOGROTATE_SERVICE_NAME = "run.sh";

    private static final String CMD_LINE = INIT_D_DESTINATION_PATH + "/" + INIT_D_LOGROTATE_SERVICE_NAME;

    private static final int LOG_SIZE_LIMIT_BYTES = 10 * (int) MEGABYTE;

    // 1000 v_cpu is enough for writing speed to log < 2 Mb/s
    private static final long V_CPU_LIMIT = 1000;
    // 128 Mb is enough for log size <= 10 Gb (max size in xdc)
    private static final long MEMORY_LIMIT = 128 * MEGABYTE;

    private final ResourceSupplier logrotateResource;

    public LogrotatePatcherV1Base(LogrotatePatcherV1Context context) {
        this(context.getLogrotateResourceSupplier());
    }

    public LogrotatePatcherV1Base(ResourceSupplier logrotateResource) {
        this.logrotateResource = logrotateResource;
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        Map<String, LogrotateConfig> logrotateConfig = context.getSpec().getLogrotateConfig();

        TPodAgentSpec.Builder specBuilder = podTemplateSpecBuilder.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder();

        specBuilder.getBoxesBuilderList().forEach(box -> {
            LogrotateConfig config = logrotateConfig.get(box.getId());
            if (config == null) {
                return;
            }

            String resourceId = makeId(box, LOG_ROTATE_CONFIG_RESOURCE_ID);

            TResource.Builder configResource = rawStaticResource(resourceId,
                    LOG_ROTATE_CONFIG_FILE_NAME,
                    config.getRawConfig());

            TResourceGang.Builder resourceBuilder = specBuilder.getResources().toBuilder();

            addStaticResourceIfAbsent(resourceBuilder, configResource, resourceId,
                    Optional.of(box.getVirtualDiskIdRef()));

            String volumeRef = makeId(box, LOG_ROTATE_LOG_VOLUME_ID);

            addVolumeIfAbsent(specBuilder, volumeRef, Optional.of(box.getVirtualDiskIdRef()));

            String layerId = makeId(box, LOG_ROTATE_APP_LAYER_ID);

            TLayer.Builder logrotateLayer = PodSpecUtils.layer(layerId, Optional.empty(),
                    logrotateResource.get());

            addLayerIfAbsent(resourceBuilder, layerId, logrotateLayer,
                    Optional.of(box.getVirtualDiskIdRef()));

            specBuilder.setResources(resourceBuilder.build());
            List<String> stringList =
                    box.getRootfs().getLayerRefsList().asByteStringList().stream().map(ByteString::toStringUtf8).collect(Collectors.toList());
            stringList.add(0, layerId);
            TRootfsVolume.Builder layerRefs = box.getRootfsBuilder()
                    .clearLayerRefs()
                    .addAllLayerRefs(stringList);

            box.setRootfs(layerRefs);

            if (!boxHasVolumeWithVarLogMountPoint(box)) {
                box.addAllVolumes(ImmutableList.of(
                        TMountedVolume.newBuilder()
                                .setVolumeRef(volumeRef)
                                .setMountPoint(LOG_ROTATE_LOG_MOUNT_POINT)
                                .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                                .build()));
            }

            box.addStaticResources(TMountedStaticResource.newBuilder()
                    .setResourceRef(resourceId)
                    .setMountPoint(LOG_ROTATE_CONFIG_MOUNT_POINT)
                    .build());

            String workloadId = WORKLOAD_ID + "_" + box.getId();

            TUtilityContainer.Builder start = TUtilityContainer.newBuilder()
                    .setCommandLine(String.format("%s %d", CMD_LINE, config.getRunPeriodMillisecond() / 1000))
                    .setStdoutAndStderrLimit(LOG_SIZE_LIMIT_BYTES)
                    .setStdoutFile(String.format("%s/%s", LOG_ROTATE_LOG_MOUNT_POINT, LOGROTATE_STDOUT))
                    .setStderrFile(String.format("%s/%s", LOG_ROTATE_LOG_MOUNT_POINT, LOGROTATE_STDERR));
            getComputeResources().ifPresent(start::setComputeResources);

            TWorkload logbrokerWorkload = TWorkload.newBuilder()
                    .setId(workloadId)
                    .setBoxRef(box.getId())
                    .setStart(start)
                    .build();

            specBuilder.addWorkloads(logbrokerWorkload);
            specBuilder.addMutableWorkloads(buildMutableWorkload(workloadId));
        });
    }

    protected Optional<TComputeResources> getComputeResources() {
        return Optional.of(
                TComputeResources.newBuilder()
                        .setVcpuLimit(V_CPU_LIMIT)
                        .setMemoryLimit(MEMORY_LIMIT)
                        .build()
        );
    }

    public static boolean boxHasVolumeWithVarLogMountPoint(TBoxOrBuilder box) {
        return box.getVolumesList().stream().anyMatch(v -> POSSIBLE_VAR_LOG_MOUNT_POINTS.contains(v.getMountPoint()));
    }

    @VisibleForTesting
    static String makeId(TBox.Builder box, String prefix) {
        return String.format("%s_%s_%s", prefix, box.getVirtualDiskIdRef(), box.getId());
    }
}
