package ru.yandex.infra.stage.podspecs.patcher.monitoring;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TMonitoringInfo;
import ru.yandex.yp.client.api.TMonitoringInfoOrBuilder;
import ru.yandex.yp.client.api.TMonitoringUnistatEndpoint;
import ru.yandex.yp.client.api.TMonitoringUnistatEndpoint.EOutputFormat;
import ru.yandex.yp.client.api.TMonitoringWorkloadEndpoint;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.DEPLOY_UNIT_KEY;
import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.ITYPE_DEPLOY;
import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.ITYPE_KEY;
import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.ITYPE_POD_AGENT;
import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.PRJ_KEY;
import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.STAGE_KEY;
import static ru.yandex.infra.stage.podspecs.patcher.monitoring.MonitoringPatcherUtils.WORKLOAD_KEY;

abstract class MonitoringPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {

    @VisibleForTesting
    static final String PRJ_UNKNOWN = "unknown_prj";

    @VisibleForTesting
    static final String POD_AGENT_UNISTAT_PATH = "/sensors";
    @VisibleForTesting
    static final String POD_AGENT_UNISTAT_USER_PATH = "/user_sensors";
    @VisibleForTesting
    static final int POD_AGENT_UNISTAT_PORT = 1;
    @VisibleForTesting
    static final EOutputFormat POD_AGENT_OUTPUT_FORMAT = EOutputFormat.OF_YASM_JSON;

    public MonitoringPatcherV1Base(MonitoringPatcherV1Context context) {
        this();
    }

    public MonitoringPatcherV1Base() {
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        patchPodSpec(podTemplateSpecBuilder.getSpecBuilder(), context);
    }

    private void patchPodSpec(DataModel.TPodSpec.Builder builder, DeployUnitContext context) {
        TMonitoringInfo.Builder podMonitoringInfo = builder.getHostInfraBuilder().getMonitoringBuilder();

        patchPodLabels(podMonitoringInfo, builder, context);

        pushLabels(podMonitoringInfo);

        patchUnistatEndpoints(podMonitoringInfo, context);

        patchEnvVars(podMonitoringInfo, builder);
    }

    private static void patchPodLabels(
            TMonitoringInfo.Builder podMonitoringInfo, DataModel.TPodSpec.Builder builder, DeployUnitContext context
    ) {
        TPodAgentSpec.Builder agentSpec = builder.getPodAgentPayloadBuilder().getSpecBuilder();

        Optional<TWorkload.Builder> workloadForPrj = agentSpec.getWorkloadsBuilderList().stream()
                .findFirst();

        String prj = workloadForPrj
                .map(workload -> workloadPrj(context.getFullDeployUnitId(), workload.getId()))
                .orElse(PRJ_UNKNOWN);

        var podMonitoringLabels = new HashMap<>(podMonitoringInfo.getLabelsMap());
        podMonitoringLabels.putIfAbsent(ITYPE_KEY, ITYPE_DEPLOY);
        podMonitoringLabels.putIfAbsent(PRJ_KEY, prj);
        podMonitoringLabels.put(STAGE_KEY, context.getStageContext().getStageId());
        podMonitoringLabels.put(DEPLOY_UNIT_KEY, context.getDeployUnitId());

        podMonitoringInfo.putAllLabels(podMonitoringLabels);
    }

    private static String workloadPrj(String fullDeployUnitId, String workloadId) {
        return String.join(StringUtils.ID_SEPARATOR, fullDeployUnitId, workloadId);
    }

    private void pushLabels(TMonitoringInfo.Builder podMonitoringInfo) {
        podMonitoringInfo.getWorkloadsBuilderList().forEach(workloadMonitoringInfo -> patchWorkloadLabels(
                workloadMonitoringInfo, podMonitoringInfo
        ));
        podMonitoringInfo.getUnistatsBuilderList().forEach(unistatInfo -> patchUnistatLabels(
                unistatInfo, podMonitoringInfo
        ));
    }

    private static void patchWorkloadLabels(TMonitoringWorkloadEndpoint.Builder workloadMonitoringInfo,
                                            TMonitoringInfoOrBuilder podMonitoringInfo) {
        if (workloadMonitoringInfo.getInheritMissedLabels()) {
            var inheritedWorkloadLabels = inheritEndpointLabels(
                    podMonitoringInfo.getLabelsMap(),
                    workloadMonitoringInfo.getLabelsMap()
            );

            workloadMonitoringInfo.putAllLabels(inheritedWorkloadLabels);
        }
    }

    protected void patchUnistatLabels(
            TMonitoringUnistatEndpoint.Builder unistatMonitoringInfo,
            TMonitoringInfoOrBuilder podMonitoringInfo
    ) {
        patchUnistatInheritance(unistatMonitoringInfo, podMonitoringInfo);
        patchUnistatWorkloadIdLabel(unistatMonitoringInfo);
    }

    protected void patchUnistatWorkloadIdLabel(TMonitoringUnistatEndpoint.Builder unistatMonitoringInfo) {
        if (!unistatMonitoringInfo.containsLabels(WORKLOAD_KEY)) {
            unistatMonitoringInfo.putLabels(WORKLOAD_KEY, unistatMonitoringInfo.getWorkloadId());
        }
    }

    protected static void patchUnistatInheritance(TMonitoringUnistatEndpoint.Builder unistatMonitoringInfo,
                                                  TMonitoringInfoOrBuilder podMonitoringInfo) {
        if (unistatMonitoringInfo.getInheritMissedLabels()) {
            var parentLabelsMap = podMonitoringInfo.getLabelsMap();

            var inheritedUnistatLabels = inheritEndpointLabels(
                    parentLabelsMap,
                    unistatMonitoringInfo.getLabelsMap()
            );

            unistatMonitoringInfo.putAllLabels(inheritedUnistatLabels);
        }
    }

    private static Map<String, String> inheritEndpointLabels(
            Map<String, String> parentLabels,
            Map<String, String> immutableChildLabels) {
        var mutableChildLabels = new HashMap<>(immutableChildLabels);
        parentLabels.forEach(mutableChildLabels::putIfAbsent);
        return mutableChildLabels;
    }

    private static void patchUnistatEndpoints(TMonitoringInfo.Builder podMonitoringInfo, DeployUnitContext context) {
        podMonitoringInfo.addUnistats(podAgentUnistatEndpoint(POD_AGENT_UNISTAT_PATH, Map.of(
                ITYPE_KEY, ITYPE_POD_AGENT,
                PRJ_KEY, context.getFullDeployUnitId()))
        );
        if (podMonitoringInfo.hasPodAgent() && podMonitoringInfo.getPodAgent().getAddPodAgentUserSignals()) {
            Map<String, String> labels = podMonitoringInfo.getPodAgent().getLabelsMap();
            podMonitoringInfo.addUnistats(
                    podAgentUnistatEndpoint(
                            POD_AGENT_UNISTAT_USER_PATH,
                            userUnistatLabels(labels, context.getFullDeployUnitId())
                    )
            );
        }
    }

    private static Map<String, String> userUnistatLabels(Map<String, String> labels, String podSetId) {
        Map<String, String> result = new HashMap<>(labels);

        if (!labels.containsKey(ITYPE_KEY)) {
            result.put(ITYPE_KEY, ITYPE_POD_AGENT);
        }

        if (!labels.containsKey(PRJ_KEY)) {
            result.put(PRJ_KEY, podSetId);
        }
        return result;
    }

    private static TMonitoringUnistatEndpoint podAgentUnistatEndpoint(String path, Map<String, String> labels) {
        return TMonitoringUnistatEndpoint.newBuilder()
                .setPath(path)
                .setPort(POD_AGENT_UNISTAT_PORT)
                .setOutputFormat(POD_AGENT_OUTPUT_FORMAT)
                .putAllLabels(labels)
                .build();
    }

    protected void patchEnvVars(TMonitoringInfo.Builder podMonitoringInfo,
                                DataModel.TPodSpec.Builder builder) {
        TPodAgentSpec.Builder agentSpec = builder.getPodAgentPayloadBuilder().getSpecBuilder();
        final Map<String, String> labelsMap = podMonitoringInfo.getLabelsMap();

        agentSpec.getBoxesBuilderList().stream()
                .filter(PatcherUtils::notSystem)
                .forEach(boxBuilder -> labelsMap.forEach((key, value) -> boxBuilder.addEnv(PodSpecUtils.literalEnvVar(getEnvVarName(key), value))));

        var workloadById = agentSpec.getWorkloadsBuilderList().stream().collect(Collectors.toMap(
                TWorkload.Builder::getId,
                Function.identity()
        ));

        var patchedWorkloads = new HashSet<String>();
        podMonitoringInfo.getUnistatsBuilderList().forEach(unistat -> {
            var workload = workloadById.get(unistat.getWorkloadId());
            if (workload != null && !patchedWorkloads.contains(workload.getId())) {
                patchedWorkloads.add(workload.getId());
                unistat.getLabelsMap().forEach((key, value) -> workload.addEnv(PodSpecUtils.literalEnvVar(getEnvVarName(key), value)));
            }
        });
    }

    protected static String getEnvVarName(String labelName) {
        return MonitoringPatcherUtils.ENV_VARS_PREFIX + "_" + labelName.toUpperCase();
    }
}
