package ru.yandex.infra.stage.podspecs.patcher.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.protobuf.ByteString;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.BoxJugglerConfig;
import ru.yandex.infra.stage.dto.SecuritySettings;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.logrotate.LogrotatePatcherV1Base;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.EContainerIsolationMode;
import ru.yandex.yp.client.pods.EVolumeCreateMode;
import ru.yandex.yp.client.pods.EVolumeMountMode;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TGenericVolume;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TVolume;
import ru.yandex.yp.client.pods.TWorkload;
import ru.yandex.yt.ytree.TAttribute;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.addLayerIfAbsent;
import static ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils.useLogbrokerTools;
import static ru.yandex.infra.stage.podspecs.patcher.logrotate.LogrotatePatcherV1Base.boxHasVolumeWithVarLogMountPoint;


abstract class SecurityPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {

    @VisibleForTesting
    static final String USE_ENV_SECRET_KEY = "use_env_secret";

    @VisibleForTesting
    static final String POD_AGENT_FOLDERS_LAYER_BASE_ID = "pod_agent_folders_layer";

    @VisibleForTesting
    static final String PORTOSHELL_FOLDERS_LAYER_BASE_ID = "portoshell_folders_layer";
    @VisibleForTesting
    static final String PORTOSHELL_HOMES_VOLUME_BASE_ID = "portoshell_homes_volume";
    @VisibleForTesting
    static final String PORTOSHELL_HOMES_VOLUME_MOUNT_POINT = "/portoshell_homes";

    @VisibleForTesting
    static final String JUGGLER_FOLDERS_LAYER_BASE_ID = "juggler_folders_layer";
    @VisibleForTesting
    static final String JUGGLER_LOGS_VOLUME_BASE_ID = "juggler_logs";
    @VisibleForTesting
    static final String JUGGLER_LOGS_VOLUME_MOUNT_POINT = "/juggler/logs";
    @VisibleForTesting
    static final String JUGGLER_SCHEDULER_VOLUME_BASE_ID = "juggler_scheduler";
    @VisibleForTesting
    static final String JUGGLER_SCHEDULER_VOLUME_MOUNT_POINT = "/juggler/scheduler";
    @VisibleForTesting
    static final String JUGGLER_CHECKS_VOLUME_BASE_ID = "juggler_checks";
    @VisibleForTesting
    static final String JUGGLER_CHECKS_VOLUME_MOUNT_POINT = "/juggler/checks";
    @VisibleForTesting
    static final String JUGGLER_STATE_VOLUME_BASE_ID = "juggler_state";
    @VisibleForTesting
    static final String JUGGLER_STATE_VOLUME_MOUNT_POINT = "/juggler/state";

    @VisibleForTesting
    static final String COREDUMP_FOLDERS_LAYER_BASE_ID = "coredump_folders_layer";

    @VisibleForTesting
    static final String TMP_VOLUME_BASE_ID = "tmp_folder_volume";
    @VisibleForTesting
    static final String TMP_VOLUME_MOUNT_POINT = "/tmp";

    @VisibleForTesting
    static final String LOGBROKER_FOLDERS_LAYER_BASE_ID = "logbroker_folders_layer";

    @VisibleForTesting
    static final String LOGROTATE_FOLDERS_LAYER_BASE_ID = "logrotate_folders_layer";

    private final static String SOX_SERVICE_LABEL_KEY = "sox_service";

    @VisibleForTesting
    static final TAttribute SOX_SERVICE_LABEL = TAttribute.newBuilder()
            .setKey(SOX_SERVICE_LABEL_KEY)
            .setValue(ByteString.copyFrom(new YTreeBuilder().value(true).build().toBinary()))
            .build();

    private final ResourceSupplier defaultPodAgentBinarySupplier;
    private final long firstAffectedPodAgentBinaryVersion;
    private final FoldersLayerUrls folderLayerUrls;

    public SecurityPatcherV1Base(SecurityPatcherV1Context context) {
        this.defaultPodAgentBinarySupplier = context.getDefaultPodAgentBinarySupplier();
        this.firstAffectedPodAgentBinaryVersion = context.getFirstAffectedPodAgentBinaryVersion();
        this.folderLayerUrls = context.getFoldersLayerUrls();
    }

    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        Optional<Long> resourceId = PodSpecUtils.SANDBOX_RESOURCE_ID_CALCULATOR.calculate(
                context.getSpec().getPodAgentResourceInfo(),
                defaultPodAgentBinarySupplier
        );

        resourceId.ifPresent(id -> {
            if (id >= firstAffectedPodAgentBinaryVersion) {
                patchImpl(podTemplateSpecBuilder, context, labelsBuilder);
            }
        });
    }

    private void patchImpl(TPodTemplateSpec.Builder builder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        TPodAgentSpec.Builder podAgentSpec = builder.getSpecBuilder()
                .getPodAgentPayloadBuilder()
                .getSpecBuilder();

        if (useSecretEnv(context)) {
            labelsBuilder.key(USE_ENV_SECRET_KEY).value(true);
        }

        if (useChildOnly(context)) {
            enableChildOnlyIsolationForUserBoxes(podAgentSpec);
        }

        patchRoFsBoxes(builder, context);
    }

    protected boolean useSecretEnv(DeployUnitContext context) {
        return !context.getSpec().getSecuritySettings().map(SecuritySettings::disableDefaultlyEnabledSecretEnv).orElse(false) || isSox(context);
    }

    protected boolean useChildOnly(DeployUnitContext context) {
        return !context.getSpec().getSecuritySettings().map(SecuritySettings::disableDefaultlyEnabledChildOnlyIsolation).orElse(false) || isSox(context);
    }

    protected boolean isSox(DeployUnitContext context) {
        return context.getSpec().isSoxService() || hasSoxServiceLabel(context);
    }

    private static void enableChildOnlyIsolationForUserBoxes(TPodAgentSpec.Builder podAgentSpec) {
        podAgentSpec.getBoxesBuilderList()
                .stream()
                .filter(PatcherUtils::notSystem)
                .forEach(b -> b.setIsolationMode(EContainerIsolationMode.EContainerIsolationMode_CHILD_ONLY));
    }

    private void patchRoFsBoxes(TPodTemplateSpec.Builder podTemplateSpec, DeployUnitContext context) {
        TPodAgentSpec.Builder podAgentSpec =
                podTemplateSpec.getSpecBuilder().getPodAgentPayloadBuilder().getSpecBuilder();

        boolean logbrokerIsUsed = useLogbrokerTools(podAgentSpec, context.getSpec().getLogbrokerConfig());

        podAgentSpec.getBoxesBuilderList().stream()
                .filter(b -> b.hasRootfs() && b.getRootfs().getCreateMode().equals(EVolumeCreateMode.EVolumeCreateMode_READ_ONLY))
                .forEach(b -> {
                    TResourceGang.Builder resourcesBuilder = podAgentSpec.getResourcesBuilder();

                    addPodAgentFoldersLayer(b, resourcesBuilder);
                    List<String> portoshellFoldersToClear = addPortoshellFoldersLayerAndWritableVolumes(b,
                            podAgentSpec);
                    List<String> tmpFolderToClear = addTmpWritableVolume(b, podAgentSpec);
                    addLogbrokerFoldersLayerIfNeed(b, resourcesBuilder, logbrokerIsUsed);
                    addCoredumpFolderLayerIfNeed(podAgentSpec, b, resourcesBuilder, context);
                    List<String> jugglerFoldersToClear = addJugglerFolderLayerAndWritableVolumesIfNeed(b,
                            podAgentSpec, context);
                    addLogrotateFolderLayerIfNeed(b, podAgentSpec, context);

                    List<String> allFoldersToClear = Stream.of(portoshellFoldersToClear, tmpFolderToClear,
                                    jugglerFoldersToClear)
                            .flatMap(Collection::stream)
                            .collect(Collectors.toList());

                    if (!allFoldersToClear.isEmpty()) {
                        String clearCommand = createRemoveFoldersContentCommand(allFoldersToClear);
                        b.addInit(TUtilityContainer.newBuilder().setCommandLine(clearCommand));
                    }
                });
    }

    protected boolean hasSoxServiceLabel(DeployUnitContext context) {
        return context.getSpec().getDetails().getLabels().getAttributesList().stream().anyMatch(a -> a.equals(SOX_SERVICE_LABEL));
    }

    private void addPodAgentFoldersLayer(TBox.Builder box, TResourceGang.Builder resourcesBuilder) {
        addLayerToBox(box, resourcesBuilder, POD_AGENT_FOLDERS_LAYER_BASE_ID,
                folderLayerUrls.getPodAgentFoldersLayerUrl());
    }

    private List<String> addPortoshellFoldersLayerAndWritableVolumes(TBox.Builder box,
                                                                     TPodAgentSpec.Builder podAgentSpec) {
        addLayerToBox(box, podAgentSpec.getResourcesBuilder(), PORTOSHELL_FOLDERS_LAYER_BASE_ID,
                folderLayerUrls.getPortoshellFoldersLayerUrl());
        addVolumeToBox(box, podAgentSpec, PORTOSHELL_HOMES_VOLUME_BASE_ID, PORTOSHELL_HOMES_VOLUME_MOUNT_POINT);
        return ImmutableList.of(PORTOSHELL_HOMES_VOLUME_MOUNT_POINT);
    }

    private void addLogbrokerFoldersLayerIfNeed(TBox.Builder box, TResourceGang.Builder resourcesBuilder,
                                                boolean logbrokerIsUsed) {
        if (logbrokerIsUsed) {
            addLayerToBox(box, resourcesBuilder, LOGBROKER_FOLDERS_LAYER_BASE_ID,
                    folderLayerUrls.getLogbrokerFoldersLayerUrl());
        }
    }

    private static List<String> addTmpWritableVolume(TBox.Builder box, TPodAgentSpec.Builder podAgentSpec) {
        addVolumeToBox(box, podAgentSpec, TMP_VOLUME_BASE_ID, TMP_VOLUME_MOUNT_POINT);
        return ImmutableList.of(TMP_VOLUME_MOUNT_POINT);
    }

    private void addCoredumpFolderLayerIfNeed(
            TPodAgentSpec.Builder podAgentSpec,
            TBox.Builder box,
            TResourceGang.Builder resourcesBuilder,
            DeployUnitContext context
    ) {
        List<String> boxWorkloadIds = podAgentSpec.getWorkloadsList().stream()
                .filter(w -> w.getBoxRef().equals(box.getId()))
                .map(TWorkload::getId)
                .collect(Collectors.toList());

        Set<String> workloadIdsWithCoredump = context.getSpec().getCoredumpConfig().keySet();

        if (!Collections.disjoint(workloadIdsWithCoredump, boxWorkloadIds)) {
            addLayerToBox(box, resourcesBuilder, COREDUMP_FOLDERS_LAYER_BASE_ID,
                    folderLayerUrls.getCoredumpFoldersLayerUrl());
        }
    }

    private List<String> addJugglerFolderLayerAndWritableVolumesIfNeed(TBox.Builder box,
                                                                       TPodAgentSpec.Builder podAgentSpec,
                                                                       DeployUnitContext context) {
        BoxJugglerConfig jugglerConfig = context.getSpec().getBoxJugglerConfigs().get(box.getId());
        if (null == jugglerConfig) {
            return ImmutableList.of();
        }

        List<String> foldersToClear = new ArrayList<>();
        addLayerToBox(box, podAgentSpec.getResourcesBuilder(), JUGGLER_FOLDERS_LAYER_BASE_ID,
                folderLayerUrls.getJugglerFoldersLayerUrl());
        addVolumeToBox(box, podAgentSpec, JUGGLER_LOGS_VOLUME_BASE_ID, JUGGLER_LOGS_VOLUME_MOUNT_POINT);
        foldersToClear.add(JUGGLER_LOGS_VOLUME_MOUNT_POINT);
        addVolumeToBox(box, podAgentSpec, JUGGLER_SCHEDULER_VOLUME_BASE_ID, JUGGLER_SCHEDULER_VOLUME_MOUNT_POINT);
        foldersToClear.add(JUGGLER_SCHEDULER_VOLUME_MOUNT_POINT);
        addVolumeToBox(box, podAgentSpec, JUGGLER_STATE_VOLUME_BASE_ID, JUGGLER_STATE_VOLUME_MOUNT_POINT);
        foldersToClear.add(JUGGLER_STATE_VOLUME_MOUNT_POINT);
        addVolumeToBox(box, podAgentSpec, JUGGLER_CHECKS_VOLUME_BASE_ID, JUGGLER_CHECKS_VOLUME_MOUNT_POINT);
        foldersToClear.add(JUGGLER_CHECKS_VOLUME_MOUNT_POINT);

        return foldersToClear;
    }

    private void addLogrotateFolderLayerIfNeed(TBox.Builder box, TPodAgentSpec.Builder podAgentSpec,
                                               DeployUnitContext context) {
        if (context.getSpec().getLogrotateConfig().containsKey(box.getId())) {

            if (boxHasVolumeWithVarLogMountPoint(box)) {
                box.addInit(TUtilityContainer.newBuilder().setCommandLine("bash -c 'mkdir -p " + LogrotatePatcherV1Base.LOG_ROTATE_LOG_MOUNT_POINT
                        + "; chown loadbase:loadbase " + LogrotatePatcherV1Base.LOG_ROTATE_LOG_MOUNT_POINT + "'").build());
            } else {
                addLayerToBox(box, podAgentSpec.getResourcesBuilder(), LOGROTATE_FOLDERS_LAYER_BASE_ID,
                        folderLayerUrls.getLogrotateFoldersLayerUrl());
            }
        }
    }

    private static void addLayerToBox(TBox.Builder box, TResourceGang.Builder resourcesBuilder, String layerBaseId,
                                      String layerUrl) {
        String layerId = String.format("%s_%s", layerBaseId, box.getVirtualDiskIdRef());
        TLayer.Builder layer = TLayer.newBuilder().setId(layerId).setUrl(layerUrl).setChecksum("EMPTY:");
        addLayerIfAbsent(resourcesBuilder, layerId, layer, boxVirtualDiskId(box));
        box.getRootfsBuilder().addLayerRefs(layerId);
    }

    private static void addVolumeToBox(TBox.Builder box, TPodAgentSpec.Builder podAgentSpec, String volumeBaseId,
                                       String volumeMountPoint) {
        String volumeId = String.format("%s_%s", volumeBaseId, box.getId());
        TVolume.Builder volume = TVolume.newBuilder()
                .setId(volumeId)
                .setGeneric(TGenericVolume.newBuilder().build())
                .setVirtualDiskIdRef(box.getVirtualDiskIdRef());
        podAgentSpec.addVolumes(volume);

        box.addVolumes(TMountedVolume.newBuilder()
                .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                .setMountPoint(volumeMountPoint)
                .setVolumeRef(volumeId)
                .build());
    }

    private static Optional<String> boxVirtualDiskId(TBox.Builder box) {
        return Optional.ofNullable(box.getVirtualDiskIdRef()).filter(vd -> !vd.isEmpty());
    }

    @VisibleForTesting
    static String createRemoveFoldersContentCommand(List<String> folders) {
        StringBuilder command = new StringBuilder("rm -rf");
        for (var folderName : folders) {
            command.append(" ");
            command.append(folderName).append("/*");
        }
        return command.toString();
    }

}
