package ru.yandex.infra.stage.podspecs.patcher.thread_limits;

import java.util.List;
import java.util.stream.Stream;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.logbroker.LogbrokerPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.pod_agent.PodAgentPatcherUtils;
import ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils;
import ru.yandex.yp.client.pods.TBoxOrBuilder;
import ru.yandex.yp.client.pods.TPodAgentSpecOrBuilder;

public final class ThreadLimitsPatcherUtils {

    private ThreadLimitsPatcherUtils() {
    }

    @VisibleForTesting
    static final long POD_UTILS_THREAD_LIMIT = 100;

    public static final long MAX_AVAILABLE_THREADS_PER_POD = 10000;

    public static final long MIN_NUM_OF_THREADS_PER_USER_BOX = 1;

    public static long maxAvailableThreadsToUserBoxesForValidation() {
        return MAX_AVAILABLE_THREADS_PER_POD
                - PodAgentPatcherUtils.POD_AGENT_THREAD_LIMIT_V1_TO_LAST
                - POD_UTILS_THREAD_LIMIT
                - TvmPatcherUtils.TVM_THREAD_LIMIT
                - LogbrokerPatcherUtils.LOGBROKER_THREAD_LIMIT;
    }

    private static <T extends TBoxOrBuilder> Stream<T> userBoxesStream(Stream<T> boxesStream) {
        return boxesStream.filter(PatcherUtils::notSystem);
    }

    private static <T extends TBoxOrBuilder> Stream<T> userBoxesWithoutThreadLimitStream(List<T> boxesStream) {
        return userBoxesStream(boxesStream.stream()).filter(box -> box.getComputeResources().getThreadLimit() == 0);
    }

    public static long numOfUserBoxesWithoutThreadLimit(TPodAgentSpecOrBuilder podAgentSpec) {
        return userBoxesWithoutThreadLimitStream(podAgentSpec.getBoxesList()).count();
    }

    public static long threadsNumUsedByUserBoxes(TPodAgentSpecOrBuilder podAgentSpec) {
        return userBoxesStream(podAgentSpec.getBoxesList().stream())
                .mapToLong(box -> box.getComputeResources().getThreadLimit()).sum();
    }
}
