package ru.yandex.infra.stage.podspecs.patcher.tvm;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.SandboxResourceInfo;
import ru.yandex.infra.stage.dto.TvmConfig;
import ru.yandex.infra.stage.podspecs.PodSpecUtils;
import ru.yandex.infra.stage.podspecs.ResourceSupplier;
import ru.yandex.infra.stage.podspecs.SidecarDiskVolumeDescription;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatcherUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.pods.EVolumeMountMode;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TEnvVar;
import ru.yandex.yp.client.pods.THttpGet;
import ru.yandex.yp.client.pods.TLayer;
import ru.yandex.yp.client.pods.TLivenessCheck;
import ru.yandex.yp.client.pods.TMountedStaticResource;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TReadinessCheck;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TRootfsVolume;
import ru.yandex.yp.client.pods.TTimeLimit;
import ru.yandex.yp.client.pods.TUtilityContainer;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.infra.stage.podspecs.PodSpecUtils.MEGABYTE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.SYSTEM_BOX_SPECIFIC_TYPE;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.allocationAccordingToDiskIsolationLabel;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.buildMutableWorkload;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.layer;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.literalEnvVar;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.rawStaticResource;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.secretEnvVar;
import static ru.yandex.infra.stage.podspecs.PodSpecUtils.volumeWithAllocationRef;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_BASE_LAYER_ID;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_CACHE_VOLUME_ID;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_CONF_RESOURCE_ID;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_LOG_VOLUME_ID;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_THREAD_LIMIT;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_TOOL_URL_ENV;
import static ru.yandex.infra.stage.podspecs.patcher.tvm.TvmPatcherUtils.TVM_WORKLOAD_ID;

// https://wiki.yandex-team.ru/drug/envctltvm/ - a little bit old description
// Constructs tvm box from two layers - bottom is default base layer, top is custom tvm layer
// See https://st.yandex-team.ru/PASSP-25256
abstract class TvmPatcherV1Base implements SpecPatcher<TPodTemplateSpec.Builder> {
    @VisibleForTesting
    static final String TVM_LAYER_ID = "tvm_layer";
    @VisibleForTesting
    static final String TVM_CONF_FILE_NAME = "config.json";
    private static final String TVM_LOG_MOUNT_POINT = "/var/log/tvmtool";
    private static final String TVM_CACHE_MOUNT_POINT = "/var/cache/tvmtool";
    private static final int LOG_SIZE_LIMIT_BYTES = 10 * (int) MEGABYTE;

    @VisibleForTesting
    static final String TVM_KEY = "tvm";
    @VisibleForTesting
    static final String TVM_STAGE_ID_LABEL = "deploy_stage_id";
    @VisibleForTesting
    static final String TVM_INSTALLATION_LABEL = "deploy_installation";
    @VisibleForTesting
    static final String TVM_ENABLED_KEY = "tool_enabled";
    @VisibleForTesting
    static final boolean TVM_ENABLED_VALUE = true;
    @VisibleForTesting
    static final String DEPLOY_TVM_CONFIG_ENV_NAME = "DEPLOY_TVM_CONFIG";
    // https://st.yandex-team.ru/DEPLOY-1291#5daec063701665001dc379b4
    // https://st.yandex-team.ru/PASSP-23758#5d2ca307a2b79e001ed1a58a
    // For disk size we need to account for layer resource, unpacked layer and logs
    @VisibleForTesting
    static final long DISK_SIZE_MB_DEFAULT = 950;

    private final Map<String, Integer> blackboxEnvironments;
    private final ResourceSupplier tvmLayerSupplier;
    private final ResourceSupplier baseLayerSupplier;

    private final Optional<String> diskVolumeAllocationId;
    private final Optional<List<String>> allSidecarDiskAllocationIds;

    private final String installationTag;
    private final long diskSpaceMb;
    private final boolean patchBoxSpecificType;
    private final long releaseGetterTimeoutSeconds;

    public TvmPatcherV1Base(TvmPatcherV1Context context) {
        this(context.getBlackboxEnvironments(),
                context.getBaseLayerSupplier(),
                context.getTvmLayerSupplier(),
                context.getDiskSpaceMb(),
                context.getDiskVolumeAllocationId(),
                context.getAllSidecarDiskAllocationIds(),
                context.getInstallationTag(),
                context.isPatchBoxSpecificType(),
                context.getReleaseGetterTimeoutSeconds());
    }

    public TvmPatcherV1Base(Map<String, Integer> blackboxEnvironments,
                            ResourceSupplier baseLayerSupplier,
                            ResourceSupplier tvmLayerSupplier,
                            long diskSpaceMb,
                            Optional<String> diskVolumeAllocationId,
                            Optional<List<String>> allSidecarDiskAllocationIds,
                            String installationTag,
                            boolean patchBoxSpecificType,
                            long releaseGetterTimeoutSeconds) {
        this.blackboxEnvironments = blackboxEnvironments;
        this.tvmLayerSupplier = tvmLayerSupplier;
        this.baseLayerSupplier = baseLayerSupplier;
        this.diskVolumeAllocationId = diskVolumeAllocationId;
        this.allSidecarDiskAllocationIds = allSidecarDiskAllocationIds;
        this.installationTag = installationTag;
        this.diskSpaceMb = diskSpaceMb;
        this.patchBoxSpecificType = patchBoxSpecificType;
        this.releaseGetterTimeoutSeconds = releaseGetterTimeoutSeconds;
    }

    @VisibleForTesting
    private static void tvmLabels(YTreeBuilder labelsBuilder, String stageId, String installationTag) {
        YTreeNode deliveryNode = new YTreeBuilder().beginMap()
                .key(TVM_STAGE_ID_LABEL).value(stageId)
                .key(TVM_INSTALLATION_LABEL).value(installationTag)
                .key(TVM_ENABLED_KEY).value(TVM_ENABLED_VALUE)
                .endMap().build();

        labelsBuilder.key(TVM_KEY).value(deliveryNode);
    }

    protected AllComputeResources getTvmBoxComputingResource(long diskSpaceMb,
                                                             TvmConfig tvmConfig,
                                                             long threadLimit) {
        final long memory = tvmConfig.getMemoryLimitMb() * MEGABYTE;
        final long anonymousMemoryLimit = memory * 9 / 10;

        return new AllComputeResources(tvmConfig.getCpuLimit(),
                memory,
                anonymousMemoryLimit,
                diskSpaceMb * MEGABYTE,
                threadLimit
        );
    }

    protected boolean shouldAddTvmConfigEnvVar() {
        return true;
    }

    protected String getTvmBoxId() {
        return TvmPatcherUtils.TVM_BOX_ID;
    }

    @Override
    public void patch(TPodTemplateSpec.Builder podTemplateSpecBuilder, DeployUnitContext context, YTreeBuilder labelsBuilder) {
        if (context.getSpec().getTvmConfig().isPresent() && context.getSpec().getTvmConfig().get().isEnabled()) {
            Optional<String> tvmToolAllocationId = allocationAccordingToDiskIsolationLabel(podTemplateSpecBuilder,
                    diskVolumeAllocationId);
            patchPodSpec(
                    context.getStageContext().getStageId(),
                    podTemplateSpecBuilder.getSpecBuilder(),
                    tvmToolAllocationId,
                    context.getSpec().getTvmConfig().get(),
                    context.getSpec().getTvmToolResourceInfo()
            );
            tvmLabels(labelsBuilder, context.getStageContext().getStageId(), this.installationTag);
        }
    }

    private void patchPodSpec(String stageId,
                              DataModel.TPodSpec.Builder builder,
                              Optional<String> tvmToolAllocationId,
                              TvmConfig tvmConfig,
                              Optional<SandboxResourceInfo> tvmToolSidecar) {
        AllComputeResources resources = getTvmBoxComputingResource(DISK_SIZE_MB_DEFAULT, tvmConfig, 0);

        Optional<SidecarDiskVolumeDescription> sidecarDiskVolumeDescription = tvmToolAllocationId.map(diskAllocationId ->
                PodSpecUtils.calculateSidecarDiskVolumeDescription(
                        stageId, builder, diskAllocationId, tvmConfig.getSidecarVolumeSettings(),
                        allSidecarDiskAllocationIds.orElseThrow())
        );

        PodSpecUtils.addResources(builder, resources, sidecarDiskVolumeDescription);
        addTvmTool(builder, tvmToolAllocationId, tvmConfig, tvmToolSidecar);
    }

    // Method for creating box with tvmtool. See
    // https://st.yandex-team.ru/DRUG-171 for details.
    private void addTvmTool(DataModel.TPodSpec.Builder builder, Optional<String> tvmToolAllocationId,
                            TvmConfig tvmConfig,
                            Optional<SandboxResourceInfo> tvmToolSidecar) {
        // patch workloadEnv by tvmToolUrl
        TPodAgentSpec.Builder agentSpec = builder.getPodAgentPayloadBuilder().getSpecBuilder();
        String tvmToolUrl = String.format("http://localhost:%d", tvmConfig.getClientPort());
        agentSpec.getBoxesBuilderList().forEach(box -> box.addEnv(
                literalEnvVar(TVM_TOOL_URL_ENV, tvmToolUrl)));


        TLayer.Builder baseLayer = layer(TVM_BASE_LAYER_ID, Optional.empty(), baseLayerSupplier.get());
        PodSpecUtils.setDiskVolumeAllocationRef(baseLayer, tvmToolAllocationId);

        var tvmLayerResourceWithMeta = PatcherUtils.getResource(tvmLayerSupplier, tvmToolSidecar,
                releaseGetterTimeoutSeconds);

        TLayer.Builder tvmToolsLayer = layer(TVM_LAYER_ID, tvmConfig.getTvmtoolLayer(), tvmLayerResourceWithMeta);
        PodSpecUtils.setDiskVolumeAllocationRef(tvmToolsLayer, tvmToolAllocationId);

        String tvmConfigString = tvmConfig.toJsonString(blackboxEnvironments);
        if(shouldAddTvmConfigEnvVar())
            addTvmConfigEnvVar(builder, tvmConfigString);

        TResource.Builder tvmToolConfigResource = rawStaticResource(TVM_CONF_RESOURCE_ID, TVM_CONF_FILE_NAME,
                tvmConfigString);
        PodSpecUtils.setDiskVolumeAllocationRef(tvmToolConfigResource, tvmToolAllocationId);

        agentSpec.getResourcesBuilder()
                .addLayers(baseLayer)
                .addLayers(tvmToolsLayer)
                // Add static resource with tvmtool config.
                .addStaticResources(tvmToolConfigResource)
                .build();

        // Add persistent volumes for tvmtool log and cache directories.
        agentSpec.addVolumes(volumeWithAllocationRef(agentSpec, TVM_LOG_VOLUME_ID, tvmToolAllocationId));
        agentSpec.addVolumes(volumeWithAllocationRef(agentSpec, TVM_CACHE_VOLUME_ID, tvmToolAllocationId));

        // Add box with tvmtool.
        String tvmConfDir = "tvmtool.conf";
        AllComputeResources boxResources = getTvmBoxComputingResource(diskSpaceMb, tvmConfig, TVM_THREAD_LIMIT);

        TBox.Builder tvmToolsBox = TBox.newBuilder()
                .setId(getTvmBoxId())
                .setRootfs(TRootfsVolume.newBuilder()
                        .addLayerRefs(TVM_LAYER_ID)
                        .addLayerRefs(TVM_BASE_LAYER_ID)
                        .build())
                .addAllVolumes(ImmutableList.of(
                        TMountedVolume.newBuilder()
                                .setVolumeRef(TVM_LOG_VOLUME_ID)
                                .setMountPoint(TVM_LOG_MOUNT_POINT)
                                .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                                .build(),
                        TMountedVolume.newBuilder()
                                .setVolumeRef(TVM_CACHE_VOLUME_ID)
                                .setMountPoint(TVM_CACHE_MOUNT_POINT)
                                .setMode(EVolumeMountMode.EVolumeMountMode_READ_WRITE)
                                .build()))
                .addStaticResources(TMountedStaticResource.newBuilder()
                        .setResourceRef(TVM_CONF_RESOURCE_ID)
                        .setMountPoint(tvmConfDir))
                .setComputeResources(boxResources.toProto());
        if (patchBoxSpecificType) {
            tvmToolsBox.setSpecificType(SYSTEM_BOX_SPECIFIC_TYPE);
        }

        PodSpecUtils.setDiskVolumeAllocationRef(tvmToolsBox, tvmToolAllocationId);

        agentSpec.addBoxes(tvmToolsBox);

        // Make workload with tvmtool.
        String tvmConfPath = String.format("%s/%s", tvmConfDir, TVM_CONF_FILE_NAME);
        String tvmAccessLogPath = String.format("%s/tvmtool.log", TVM_LOG_MOUNT_POINT);
        String tvmErrorLogPath = String.format("%s/error.log", TVM_LOG_MOUNT_POINT);
        String tvmStartCmd = "/tvmtool_start.sh";
        List<TEnvVar> secretEnvs = tvmConfig.getClients().stream()
                .filter(c -> !c.getDestinations().isEmpty())
                .map(c -> secretEnvVar(c.getTvmClientSecretEnvName(), c.getSecretSelector()))
                .collect(Collectors.toList());
        THttpGet httpGet = THttpGet.newBuilder()
                .setPath("/tvm/ping")
                .setPort(tvmConfig.getClientPort())
                .setAny(true)
                .setTimeLimit(TTimeLimit.newBuilder()
                        .setMinRestartPeriodMs(Duration.ofSeconds(10).toMillis())
                        .setMaxRestartPeriodMs(Duration.ofSeconds(10).toMillis())
                        .setMaxExecutionTimeMs(Duration.ofMinutes(1).toMillis())
                        .build())
                .build();
        TWorkload tvmWorkload = TWorkload.newBuilder()
                .setId(TVM_WORKLOAD_ID)
                .setBoxRef(getTvmBoxId())
                .addAllEnv(secretEnvs)
                .addEnv(literalEnvVar("TVMTOOL_CONFIG_PATH", tvmConfPath))
                .addEnv(literalEnvVar("TVMTOOL_CACHEDIR_PATH", TVM_CACHE_MOUNT_POINT))
                .setStart(TUtilityContainer.newBuilder()
                        .setCommandLine(tvmStartCmd)
                        .setStdoutAndStderrLimit(LOG_SIZE_LIMIT_BYTES)
                        .setStdoutFile(tvmAccessLogPath)
                        .setStderrFile(tvmErrorLogPath)
                        .build())
                .setReadinessCheck(TReadinessCheck.newBuilder()
                        .setHttpGet(httpGet))
                .setLivenessCheck(TLivenessCheck.newBuilder()
                        .setHttpGet(httpGet))
                .build();

        // Add workloads and mutable workloads to pod agent spec.
        agentSpec.addWorkloads(tvmWorkload);
        agentSpec.addMutableWorkloads(buildMutableWorkload(TVM_WORKLOAD_ID));
    }

    private void addTvmConfigEnvVar(DataModel.TPodSpec.Builder builder, String configStr) {
        builder.getPodAgentPayloadBuilder().getSpecBuilder().getWorkloadsBuilderList().stream()
                .filter(workloadBuilder -> PatcherUtils.notSystem(workloadBuilder.getBoxRef()))
                .forEach(workloadBuilder ->
                        workloadBuilder.addEnv(PodSpecUtils.literalEnvVar(DEPLOY_TVM_CONFIG_ENV_NAME, configStr)));
    }
}
