package ru.yandex.infra.stage.podspecs.patcher.tvm;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.stage.podspecs.ResourceSupplier;

public class TvmPatcherV1Context {

    private final Map<String, Integer> blackboxEnvironments;
    private final ResourceSupplier baseLayerSupplier;
    private final ResourceSupplier tvmLayerSupplier;

    private final long diskSpaceMb;
    private final Optional<String> diskVolumeAllocationId;
    private final Optional<List<String>> allSidecarDiskAllocationIds;

    private final String installationTag;
    private final boolean patchBoxSpecificType;
    private final long releaseGetterTimeoutSeconds;

    public TvmPatcherV1Context(Map<String, Integer> blackboxEnvironments,
                               ResourceSupplier baseLayerSupplier,
                               ResourceSupplier tvmLayerSupplier,
                               long diskSpaceMb,
                               Optional<String> diskVolumeAllocationId,
                               Optional<List<String>> allSidecarDiskAllocationIds,
                               String installationTag,
                               boolean patchBoxSpecificType,
                               long releaseGetterTimeoutSeconds) {
        this.blackboxEnvironments = blackboxEnvironments;
        this.baseLayerSupplier = baseLayerSupplier;
        this.tvmLayerSupplier = tvmLayerSupplier;
        this.diskSpaceMb = diskSpaceMb;
        this.diskVolumeAllocationId = diskVolumeAllocationId;
        this.allSidecarDiskAllocationIds = allSidecarDiskAllocationIds;
        this.installationTag = installationTag;
        this.patchBoxSpecificType = patchBoxSpecificType;
        this.releaseGetterTimeoutSeconds = releaseGetterTimeoutSeconds;
    }

    public Map<String, Integer> getBlackboxEnvironments() {
        return blackboxEnvironments;
    }

    public ResourceSupplier getTvmLayerSupplier() {
        return tvmLayerSupplier;
    }

    public ResourceSupplier getBaseLayerSupplier() {
        return baseLayerSupplier;
    }

    public Optional<String> getDiskVolumeAllocationId() {
        return diskVolumeAllocationId;
    }

    public String getInstallationTag() {
        return installationTag;
    }

    public long getDiskSpaceMb() {
        return diskSpaceMb;
    }

    public boolean isPatchBoxSpecificType() {
        return patchBoxSpecificType;
    }

    public long getReleaseGetterTimeoutSeconds() {
        return releaseGetterTimeoutSeconds;
    }

    public Optional<List<String>> getAllSidecarDiskAllocationIds() {
        return allSidecarDiskAllocationIds;
    }

    @VisibleForTesting
    TvmPatcherV1Context withTvmLayerSupplier(ResourceSupplier tvmLayerSupplier) {
        return toBuilder().withTvmLayerSupplier(tvmLayerSupplier).build();
    }

    @VisibleForTesting
    Builder toBuilder() {
        return new Builder(this);
    }

    @VisibleForTesting
    static class Builder {
        private final Map<String, Integer> blackboxEnvironments;
        private ResourceSupplier baseLayerSupplier;
        private ResourceSupplier tvmLayerSupplier;

        private final long diskSpaceMb;
        private Optional<String> diskVolumeAllocationId;
        private Optional<List<String>> allSidecarDiskAllocationIds;

        private final String installationTag;
        private boolean patchBoxSpecificType;
        private final long releaseGetterTimeoutSeconds;

        Builder(TvmPatcherV1Context context) {
            this.blackboxEnvironments = context.blackboxEnvironments;
            this.baseLayerSupplier = context.baseLayerSupplier;
            this.tvmLayerSupplier = context.tvmLayerSupplier;
            this.diskSpaceMb = context.diskSpaceMb;
            this.diskVolumeAllocationId = context.diskVolumeAllocationId;
            this.allSidecarDiskAllocationIds = context.allSidecarDiskAllocationIds;
            this.installationTag = context.installationTag;
            this.patchBoxSpecificType = context.patchBoxSpecificType;
            this.releaseGetterTimeoutSeconds = context.releaseGetterTimeoutSeconds;
        }

        @VisibleForTesting
        TvmPatcherV1Context build() {
            return new TvmPatcherV1Context(
                    blackboxEnvironments,
                    baseLayerSupplier,
                    tvmLayerSupplier,
                    diskSpaceMb,
                    diskVolumeAllocationId,
                    allSidecarDiskAllocationIds,
                    installationTag,
                    patchBoxSpecificType,
                    releaseGetterTimeoutSeconds
            );
        }

        @VisibleForTesting
        Builder withTvmLayerSupplier(ResourceSupplier tvmLayerSupplier) {
            this.tvmLayerSupplier = tvmLayerSupplier;
            return this;
        }

        @VisibleForTesting
        Builder withBaseLayerSupplier(ResourceSupplier baseLayerSupplier) {
            this.baseLayerSupplier = baseLayerSupplier;
            return this;
        }

        @VisibleForTesting
        Builder withDiskVolumeAllocationId(String diskVolumeAllocationId) {
            this.diskVolumeAllocationId = Optional.ofNullable(diskVolumeAllocationId);
            return this;
        }

        @VisibleForTesting
        Builder withAllSidecarDiskAllocationIds(List<String> allSidecarDiskAllocationIds) {
            this.allSidecarDiskAllocationIds = Optional.ofNullable(allSidecarDiskAllocationIds);
            return this;
        }


        @VisibleForTesting
        Builder withPatchBoxSpecificType(boolean patchBoxSpecificType) {
            this.patchBoxSpecificType = patchBoxSpecificType;
            return this;
        }
    }
}
