package ru.yandex.infra.stage.podspecs.revision;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.protobuf.Message;
import com.typesafe.config.Config;

import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.podspecs.patcher.PatchersHolder;
import ru.yandex.infra.stage.podspecs.revision.model.CompositeInfo;
import ru.yandex.infra.stage.podspecs.revision.model.PatcherClassInfo;
import ru.yandex.infra.stage.podspecs.revision.model.Revision;
import ru.yandex.infra.stage.podspecs.revision.model.RevisionScheme;


public final class PatchersRevisionsHolderFactory<T extends Message.Builder> {

    @VisibleForTesting
    public static final String ACTIVE_REVISION_IDS_CONFIG_KEY = "active_revisions";

    public RevisionsHolder<T> from(RevisionScheme revisionScheme,
                                   Config revisionSchemeConfig, PatcherType patcherType) {
        return new PatchersRevisionsHolderFactory<T>(patchersHolder)
                .createBy(revisionScheme, revisionSchemeConfig, patcherType);
    }

    private Class<? extends SpecPatcher<T>> findPatcherClassBy(String totalClassName) {
        try {
            return (Class<? extends SpecPatcher<T>>) Class.forName(totalClassName).asSubclass(SpecPatcher.class);
        } catch (ClassNotFoundException e) {
            return null;
        }
    }

    private final PatchersHolder<T> patchersHolder;
    private final String basePatchersPackage;

    private final Map<PatcherClassInfo, Optional<Class<? extends SpecPatcher<T>>>> patcherInfoToClassCache;

    public PatchersRevisionsHolderFactory(PatchersHolder<T> patchersHolder) {
        this.patchersHolder = patchersHolder;
        this.basePatchersPackage = patchersHolder.getClass().getPackageName();
        this.patcherInfoToClassCache = new HashMap<>();
    }

    RevisionsHolderImpl<T> createBy(RevisionScheme revisionScheme, Config revisionSchemeConfig,
                                    PatcherType patcherType) {
        Map<Integer, List<SpecPatcher<T>>> revisionIdToPatchers = new HashMap<>();
        Set<Integer> activeRevisionIds = ImmutableSet.copyOf(
                revisionSchemeConfig.getIntList(ACTIVE_REVISION_IDS_CONFIG_KEY)
        );

        var activeRevisions = Arrays.stream(revisionScheme.getRevisions())
                .filter(revision -> activeRevisionIds.contains(revision.getId()));

        activeRevisions.forEach(revision -> {
            int revisionId = revision.getId();
            if (revisionIdToPatchers.containsKey(revisionId)) {
                throw new RevisionSchemeException(
                        String.format("Found duplicated revisions with id %d", revisionId)
                );
            }

            var patchers = getRevisionPatchers(revision, patcherType);

            revisionIdToPatchers.put(revisionId, patchers);
        });

        int fallbackRevisionId = revisionScheme.getFallbackRevisionId();
        if (!revisionIdToPatchers.containsKey(fallbackRevisionId)) {
            throw new RevisionSchemeException(String.format("Unknown fallback revision %d", fallbackRevisionId));
        }

        return new RevisionsHolderImpl<>(
                ImmutableMap.copyOf(revisionIdToPatchers),
                fallbackRevisionId
        );
    }

    private List<SpecPatcher<T>> getRevisionPatchers(Revision revision, PatcherType patcherType) {
        int revisionId = revision.getId();
        PatcherClassInfo[] patcherClassInfos;
        CompositeInfo compositeInfo;

        if (patcherType.equals(PatcherType.ENDPOINT_SET_SPEC)) {
            patcherClassInfos = revision.getEndpointSetSpecInfo().getPatcherClassInfos();
            compositeInfo = revision.getEndpointSetSpecInfo().getCompositeInfo();
        } else if (patcherType.equals(PatcherType.POD_SPEC)) {
            patcherClassInfos = revision.getPodSpecInfo().getPatcherClassInfos();
            compositeInfo = revision.getPodSpecInfo().getCompositeInfo();
        } else {
            throw new RuntimeException(String.format("Unknown patcherHolder type %s", patcherType));
        }


        var nameToPatcherClass = createNameToPatcherClassMap(revisionId, patcherClassInfos);


        List<Class<? extends SpecPatcher<T>>> patcherClassesOrder = getPatcherClassesOrder(
                revisionId, nameToPatcherClass, compositeInfo
        );

        return this.patchersHolder.getPatchersInExactOrder(patcherClassesOrder);
    }

    private Map<String, Class<? extends SpecPatcher<T>>> createNameToPatcherClassMap(int revisionId,
                                                                                     PatcherClassInfo[] patcherClassInfos) {
        List<Class<? extends SpecPatcher<T>>> list = new ArrayList<>();
        for (PatcherClassInfo patcherClassInfo : patcherClassInfos) {
            Class<? extends SpecPatcher<T>> patcherClassBy = getPatcherClassBy(patcherClassInfo);
            list.add(patcherClassBy);
        }
        List<Class<? extends SpecPatcher<T>>> patcherClasses = Collections.unmodifiableList(list);

        assertNoPatcherClassDuplications(revisionId, patcherClasses);
        assertNoPatcherPackageDuplications(revisionId, patcherClasses);

        return patcherClasses.stream()
                .collect(Collectors.toMap(Class::getSimpleName, Function.identity()));
    }

    private Class<? extends SpecPatcher<T>> getPatcherClassBy(PatcherClassInfo patcherClassInfo) {
        return getPatcherClassOptionalBy(patcherClassInfo)
                .orElseThrow(() -> new RevisionSchemeException(
                                String.format("Patcher class with info %s not found", patcherClassInfo.toString())
                        )
                );
    }

    private Optional<Class<? extends SpecPatcher<T>>> getPatcherClassOptionalBy(PatcherClassInfo patcherClassInfo) {
        return patcherInfoToClassCache.computeIfAbsent(patcherClassInfo, this::findPatcherClassOptionalBy);
    }

    private Optional<Class<? extends SpecPatcher<T>>> findPatcherClassOptionalBy(PatcherClassInfo patcherClassInfo) {
        String totalClassName = patcherClassInfo.getPatcherFullClassName(basePatchersPackage);
        Class<? extends SpecPatcher<T>> patcherClass = findPatcherClassBy(totalClassName);
        return Optional.ofNullable(patcherClass);
    }

    private void assertNoPatcherClassDuplications(int revisionId,
                                                  List<Class<? extends SpecPatcher<T>>> patcherClasses) {
        Set<Class<? extends SpecPatcher<T>>> classes = new HashSet<>();
        Map<Package, Class<? extends SpecPatcher<T>>> packageToPatcherClass = new HashMap<>();

        for (Class<? extends SpecPatcher<T>> patcherClass : patcherClasses) {
            if (!classes.add(patcherClass)) {
                throw new RevisionSchemeException(
                        String.format(
                                "Revision %d: found duplicated patcher classes %s",
                                revisionId,
                                patcherClass.getCanonicalName()
                        )
                );
            }

            var patcherPackage = patcherClass.getPackage();
            var alreadyFoundClass = packageToPatcherClass.put(patcherPackage, patcherClass);
            if (null != alreadyFoundClass) {
                throw new RevisionSchemeException(
                        String.format(
                                "Revision %d: found different patcher classes from one package %s: %s and %s",
                                revisionId,
                                patcherPackage.getName(),
                                patcherClass.getCanonicalName(),
                                alreadyFoundClass.getCanonicalName()
                        )
                );
            }
        }
    }

    private void assertNoPatcherPackageDuplications(int revisionId,
                                                    List<Class<? extends SpecPatcher<T>>> patcherClasses) {
        Map<Package, Class<? extends SpecPatcher<T>>> packageToPatcherClass = new HashMap<>();

        for (Class<? extends SpecPatcher<T>> patcherClass : patcherClasses) {
            var patcherPackage = patcherClass.getPackage();
            var alreadyFoundClass = packageToPatcherClass.put(patcherPackage, patcherClass);
            if (null != alreadyFoundClass) {
                throw new RevisionSchemeException(
                        String.format(
                                "Revision %d: found different patcher classes from one package %s: %s and %s",
                                revisionId,
                                patcherPackage.getName(),
                                patcherClass.getCanonicalName(),
                                alreadyFoundClass.getCanonicalName()
                        )
                );
            }
        }
    }


    private List<Class<? extends SpecPatcher<T>>> getPatcherClassesOrder(int revisionId,
                                                                         Map<String, Class<?
                                                                                 extends SpecPatcher<T>>> nameToPatcherClass,
                                                                         CompositeInfo compositeInfo) {
        var patcherNamesOrder = compositeInfo.getPatcherNamesOrder();

        var patcherClassesOrderBuilder = ImmutableList.<Class<? extends SpecPatcher<T>>>builder();
        for (String patcherName : patcherNamesOrder) {
            var patcherClass = nameToPatcherClass.get(patcherName);
            if (null == patcherClass) {
                throw new RevisionSchemeException(
                        String.format(
                                "Revision %d: not found class for patcher name %s in composite order",
                                revisionId, patcherName
                        )
                );
            }

            patcherClassesOrderBuilder.add(patcherClass);
        }

        var patcherClassesOrder = patcherClassesOrderBuilder.build();

        assertNoPatcherClassDuplications(revisionId, patcherClassesOrder);
        assertAllClassesWereUsed(revisionId, nameToPatcherClass, patcherClassesOrder);

        return patcherClassesOrder;
    }

    private void assertAllClassesWereUsed(int revisionId,
                                          Map<String, Class<? extends SpecPatcher<T>>> nameToPatcherClass,
                                          List<Class<? extends SpecPatcher<T>>> patcherClassesOrder) {
        var patcherClassesOrderSet = ImmutableSet.copyOf(patcherClassesOrder);

        var notUsedClassNames = nameToPatcherClass.entrySet()
                .stream()
                .filter(nameClassEntry -> !patcherClassesOrderSet.contains(nameClassEntry.getValue()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());

        if (!notUsedClassNames.isEmpty()) {
            throw new RevisionSchemeException(
                    String.format(
                            "Revision %d: not all patchers are used in order:\n[%s]",
                            revisionId,
                            String.join("\n", notUsedClassNames)
                    )
            );
        }
    }
}
