package ru.yandex.infra.stage.primitives;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import ru.yandex.infra.stage.deployunit.Readiness;
import ru.yandex.infra.stage.dto.DeployProgress;

// Wrapper for DeployPrimitiveStatus with unfolded perCluster statuses
public class AggregatedRawStatus<T> {
    private final DeployPrimitiveStatus<T> primitiveStatus;
    private final Map<String, DeployPrimitiveStatus<T>> perClusterStatuses;

    public static <T> AggregatedRawStatus<T> merge(Map<String, DeployPrimitiveStatus<T>> map) {
        return new AggregatedRawStatus<>(map);
    }

    public static <T> AggregatedRawStatus<T> fromClusterStatus(String clusterId, DeployPrimitiveStatus<T> status) {
        return new AggregatedRawStatus<>(status, Map.of(clusterId, status));
    }

    public AggregatedRawStatus(DeployPrimitiveStatus<T> primitiveStatus, Map<String, DeployPrimitiveStatus<T>> perClusterStatuses) {
        this.primitiveStatus = primitiveStatus;
        this.perClusterStatuses = ImmutableMap.copyOf(perClusterStatuses);
    }

    public AggregatedRawStatus(Map<String, DeployPrimitiveStatus<T>> perClusterStatuses) {
        this.perClusterStatuses = ImmutableMap.copyOf(perClusterStatuses);
        this.primitiveStatus = DeployPrimitiveStatus.merge(perClusterStatuses);
    }

    public AggregatedRawStatus(DeployPrimitiveStatus<T> primitiveStatus) {
        this.perClusterStatuses = Collections.emptyMap();
        this.primitiveStatus = primitiveStatus;
    }

    public Readiness getReadiness() {
        return primitiveStatus.getReadiness();
    }

    public DeployProgress getProgress() {
        return primitiveStatus.getProgress();
    }

    public Map<String, Optional<T>> getClusterStatuses() {
        return EntryStream.of(perClusterStatuses).mapValues(DeployPrimitiveStatus::getRawStatus).toMap();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AggregatedRawStatus<?> that = (AggregatedRawStatus<?>) o;
        return Objects.equals(primitiveStatus, that.primitiveStatus) &&
                Objects.equals(perClusterStatuses, that.perClusterStatuses);
    }

    @Override
    public int hashCode() {
        return Objects.hash(primitiveStatus, perClusterStatuses);
    }

}
