package ru.yandex.infra.stage.primitives;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.MoreObjects;
import one.util.streamex.EntryStream;

import ru.yandex.infra.stage.deployunit.Readiness;
import ru.yandex.infra.stage.deployunit.ReadyStatus;
import ru.yandex.infra.stage.dto.DeployProgress;

public class DeployPrimitiveStatus<YpStatus> implements ReadyStatus {
    private final DeployProgress progress;
    private final Readiness readiness;
    private final Optional<YpStatus> rawStatus;

    public DeployPrimitiveStatus(Readiness readiness, DeployProgress progress, Optional<YpStatus> rawStatus) {
        this.progress = progress;
        this.readiness = readiness;
        this.rawStatus = rawStatus;
    }

    public Readiness getReadiness() {
        return readiness;
    }

    public Optional<YpStatus> getRawStatus() {
        return rawStatus;
    }

    public DeployProgress getProgress() {
        return progress;
    }

    public static <YpStatus> DeployPrimitiveStatus<YpStatus> merge(Map<String, DeployPrimitiveStatus<YpStatus>> perClusterStatuses) {
        DeployProgress totalProgress = perClusterStatuses.values().stream()
                .map(DeployPrimitiveStatus::getProgress)
                .reduce(DeployProgress::sum).orElse(DeployProgress.EMPTY);
        Readiness totalReadiness = Readiness.mergeRelated(EntryStream.of(perClusterStatuses)
                .mapValues(DeployPrimitiveStatus::getReadiness)
                .toMap()
        );
        return new DeployPrimitiveStatus<>(totalReadiness, totalProgress, Optional.empty());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DeployPrimitiveStatus)) {
            return false;
        }
        DeployPrimitiveStatus that = (DeployPrimitiveStatus) o;
        return readiness.equals(that.readiness) &&
                Objects.equals(progress, that.progress) &&
                Objects.equals(rawStatus, that.rawStatus);

    }

    @Override
    public int hashCode() {
        return Objects.hash(readiness, progress, rawStatus);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("readiness", readiness)
                .add("progress", progress)
                .toString();
    }

    @Override
    public boolean isReady() {
        return readiness.isReady();
    }
}
