package ru.yandex.infra.stage.primitives;

import java.util.Map;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.infra.stage.StageContext;
import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.deployunit.DeployUnitStats;
import ru.yandex.infra.stage.deployunit.ObjectController;
import ru.yandex.infra.stage.dto.McrsUnitSpec;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetStatus;
import ru.yandex.yp.client.api.TPodTemplateSpec;

// TODO: this may be merged with MultiClusterReplicaSetController, because it does only spec patching
public class McrsDeployPrimitiveController implements DeployPrimitiveController<McrsUnitSpec> {
    private final ObjectController<TMultiClusterReplicaSetSpec, DeployPrimitiveStatus<TMultiClusterReplicaSetStatus>> mcrsController;
    private final Converter converter;
    private final SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher;

    public McrsDeployPrimitiveController(ObjectController<TMultiClusterReplicaSetSpec,
            DeployPrimitiveStatus<TMultiClusterReplicaSetStatus>> mcrsController,
                                         SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher,
                                         Converter converter) {
        this.mcrsController = mcrsController;
        this.converter = converter;
        this.podSpecPatcher = podSpecPatcher;
    }

    @Override
    public void sync(McrsUnitSpec spec, DeployUnitContext context) {
         mcrsController.sync(mergeSpec(spec, context, podSpecPatcher, converter),
                context.getStageContext(),
                getLabels(context.getStageContext()));
    }

    private Map<String, YTreeNode> getLabels(StageContext stageContext) {
        YTreeBuilder deployLabelBuilder = YTree.mapBuilder()
                .key(STAGE_LABEL_KEY)
                .value(stageContext.getStageId());

        if (!stageContext.getDisabledClusters().isEmpty()) {
            final YTreeBuilder listBuilder = YTree.listBuilder();
            stageContext.getDisabledClusters().forEach(listBuilder::value);
            deployLabelBuilder.key(DISABLED_CLUSTERS_LABEL_KEY).value(listBuilder.buildList());
        }

        return Map.of(DEPLOY_LABEL_KEY, deployLabelBuilder.buildMap());
    }

    @Override
    public void shutdown() {
        mcrsController.shutdown();
    }

    @Override
    public AggregatedRawStatus<TMultiClusterReplicaSetStatus> getStatus() {
        return new AggregatedRawStatus<>(mcrsController.getStatus());
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        mcrsController.addStats(builder);
    }

    @VisibleForTesting
    static TMultiClusterReplicaSetSpec mergeSpec(McrsUnitSpec spec,
                                                 DeployUnitContext context,
                                                 SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher,
                                                 Converter converter) {
        TMultiClusterReplicaSetSpec.Builder builder = spec.getSpec().toBuilder();
        podSpecPatcher.patch(builder.getPodTemplateSpecBuilder(), context, new YTreeBuilder());
        builder.setRevision(context.getSpec().getRevision());
        builder.setAccountId(context.getStageContext().getAccountId());

        boolean shouldPatchConstraints = spec.getSpec().getClustersList().stream()
                .allMatch(preferences -> preferences.getSpec().getConstraints().getAntiaffinityConstraintsCount() == 0);

        if (shouldPatchConstraints) {
            builder.getClustersBuilderList().forEach(clusterBuilder ->
                clusterBuilder.getSpecBuilder().getConstraintsBuilder()
                        .addAntiaffinityConstraints(converter.toProto(DEFAULT_CONSTRAINT))
            );
        }

        return builder.build();
    }
}
