package ru.yandex.infra.stage.primitives;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.infra.stage.deployunit.DeployUnitContext;
import ru.yandex.infra.stage.deployunit.DeployUnitStats;
import ru.yandex.infra.stage.deployunit.MultiplexingController;
import ru.yandex.infra.stage.deployunit.MultiplexingController.Factory;
import ru.yandex.infra.stage.dto.DeployUnitSpec.DeploySettings;
import ru.yandex.infra.stage.dto.DeployUnitSpec.DeploySettings.ClusterSettings;
import ru.yandex.infra.stage.dto.ReplicaSetUnitSpec;
import ru.yandex.infra.stage.dto.RuntimeDeployControls;
import ru.yandex.infra.stage.podspecs.SpecPatcher;
import ru.yandex.infra.stage.protobuf.Converter;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.TDeployUnitSpec;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;

import static ru.yandex.infra.stage.util.GeneralUtils.CLUSTER_SEQUENCE;

public class ReplicaSetDeployPrimitiveController implements DeployPrimitiveController<ReplicaSetUnitSpec> {
    public static final String AUTOSCALER_ID_LABEL_KEY = "horizontal_pod_autoscaler_id";
    private static final String EMPTY_ID = "";

    private final MultiplexingController<TReplicaSetSpec, DeployPrimitiveStatus<TReplicaSetStatus>> perCluster;
    private final SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher;
    private final Converter converter;
    private final String id;

    public ReplicaSetDeployPrimitiveController(String id, Factory<TReplicaSetSpec, DeployPrimitiveStatus<TReplicaSetStatus>> perCluster,
                                               SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher,
                                               Consumer<AggregatedRawStatus<TReplicaSetStatus>> updateNotifier,
                                               Converter converter) {
        this.perCluster = new MultiplexingController<>(id, perCluster, "replica set", status -> {
            AggregatedRawStatus<TReplicaSetStatus> st = AggregatedRawStatus.fromClusterStatus(id, status);
            updateNotifier.accept(st);
        });
        this.converter = converter;
        this.podSpecPatcher = podSpecPatcher;
        this.id = id;
    }

    public ReplicaSetDeployPrimitiveController(String id,
                                               MultiplexingController<TReplicaSetSpec, DeployPrimitiveStatus<TReplicaSetStatus>> perCluster,
                                               SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher,
                                               Converter converter) {
        this.perCluster = perCluster;
        this.converter = converter;
        this.podSpecPatcher = podSpecPatcher;
        this.id = id;
    }

    private List<Pair<String, Boolean>> extractClusterSequence(DeployUnitContext context,
                                                               ReplicaSetUnitSpec spec) {
        RuntimeDeployControls runtimeDeployControls = context.getStageContext().getRuntimeDeployControls();
        if (runtimeDeployControls.getDeploySettings().containsKey(context.getDeployUnitId())) {
            DeploySettings deploySettings = runtimeDeployControls.getDeploySettings().get(context.getDeployUnitId());
            return convertDeploySettingsToClusterSequence(spec, deploySettings);
        }

        Optional<DeploySettings> deploySettings = context.getSpec().getDeploySettings();
        if (deploySettings.isPresent()) {
            DeploySettings settings = deploySettings.get();
            return convertDeploySettingsToClusterSequence(spec, settings);
        }
        List<String> fullClusterSequence = new ArrayList<>(spec.extractClusters());
        return fullClusterSequence.stream().map(v -> Pair.of(v, false)).collect(Collectors.toList());
    }

    private List<Pair<String, Boolean>> convertDeploySettingsToClusterSequence(ReplicaSetUnitSpec spec,
                                                                               DeploySettings settings) {
        List<String> fullClusterSequence = completeClusterSequence(spec,
                settings.getClustersSettings().stream().map(ClusterSettings::getName).collect(Collectors.toList()));
        Map<String, Boolean> clusterMap = settings.getClustersSettings().stream()
                .collect(Collectors.toMap(ClusterSettings::getName,
                        ClusterSettings::isNeedApprove));
        return fullClusterSequence.stream().map(t -> Pair.of(t, clusterMap.getOrDefault(t, false))).collect(Collectors.toList());
    }

    @Deprecated
    private List<Pair<String, Boolean>> extractDeprecatedClusterSequence(DeployUnitContext context,
                                                                         ReplicaSetUnitSpec spec) {
        YTreeNode yTreeNode = context.getStageContext().getLabels().get(CLUSTER_SEQUENCE);
        List<String> fullClusterSequence = new ArrayList<>(spec.extractClusters());
        if (yTreeNode != null && yTreeNode.isListNode()) {
            List<String> clusterSequence =
                    (yTreeNode.asList().stream().map(YTreeNode::stringValue)).collect(Collectors.toList());
            fullClusterSequence = completeClusterSequence(spec, clusterSequence);
        }
        return fullClusterSequence.stream().map(v -> Pair.of(v, false)).collect(Collectors.toList());
    }

    private List<String> completeClusterSequence(ReplicaSetUnitSpec spec, List<String> clusterSequence) {
        List<String> fullClusterSequence = new ArrayList<>(clusterSequence);
        Set<String> clusters = new HashSet<>(spec.extractClusters());
        if (fullClusterSequence.size() < clusters.size()) {
            fullClusterSequence.forEach(clusters::remove);
            fullClusterSequence.addAll(clusters);
        }
        if (!Sets.difference(Set.copyOf(fullClusterSequence), spec.extractClusters()).isEmpty()) {
            fullClusterSequence = new ArrayList<>(clusters);
        }
        return fullClusterSequence;
    }

    private DeployStrategy getTypeOfDeployStrategy(DeployUnitContext context) {
        RuntimeDeployControls runtimeDeployControls = context.getStageContext().getRuntimeDeployControls();
        if (runtimeDeployControls.getDeploySettings().containsKey(context.getDeployUnitId())) {
            DeploySettings deploySettings = runtimeDeployControls.getDeploySettings().get(context.getDeployUnitId());
            return getDeployStrategy(deploySettings.getDeployStrategy());
        }
        if (context.getSpec().getDeploySettings().isPresent()) {
            return getDeployStrategy(context.getSpec().getDeploySettings().get().getDeployStrategy());
        }
        YTreeNode yTreeNode = context.getStageContext().getLabels().get(CLUSTER_SEQUENCE);
        if (yTreeNode != null && yTreeNode.isListNode()) {
            return DeployStrategy.DEPRECATED_SEQUENTIAL;
        }
        return DeployStrategy.PARALLEL;
    }

    private DeployStrategy getDeployStrategy(TDeployUnitSpec.TDeploySettings.EDeployStrategy strategy) {
        if (strategy.equals(TDeployUnitSpec.TDeploySettings.EDeployStrategy.SEQUENTIAL)) {
            return DeployStrategy.SEQUENTIAL;
        }
        if (strategy.equals(TDeployUnitSpec.TDeploySettings.EDeployStrategy.PARALLEL)) {
            return DeployStrategy.PARALLEL;
        }
        throw new IllegalStateException(String.format("Unsupported type of deploy strategy %s",
                strategy));
    }

    @Override
    public void sync(ReplicaSetUnitSpec spec, DeployUnitContext context) {
        DeployStrategy typeOfDeployStrategy = getTypeOfDeployStrategy(context);
        final Map<String, TReplicaSetSpec> newSpecs = StreamEx.of(spec.extractClusters())
                .toMap(cluster -> mergeSpec(spec, context, cluster, podSpecPatcher, converter));
        switch (typeOfDeployStrategy) {
            case DEPRECATED_SEQUENTIAL:
                perCluster.syncSequential(
                        newSpecs,
                        context.getStageContext(),
                        context.getStageContext().getRuntimeDeployControls()
                                .getApprovedClusterList().getOrDefault(context.getDeployUnitId(),
                                Collections.emptySet()),
                        labels(spec, context),
                        extractDeprecatedClusterSequence(context, spec));
                break;
            case SEQUENTIAL:
                perCluster.syncSequential(
                        newSpecs,
                        context.getStageContext(),
                        context.getStageContext().getRuntimeDeployControls()
                                .getApprovedClusterList().getOrDefault(context.getDeployUnitId(),
                                Collections.emptySet()),
                        labels(spec, context),
                        extractClusterSequence(context, spec));
                break;
            case PARALLEL:
                perCluster.syncParallel(
                        newSpecs,
                        context.getStageContext(),
                        context.getStageContext().getRuntimeDeployControls()
                                .getApprovedClusterList().getOrDefault(context.getDeployUnitId(),
                                Collections.emptySet()),
                        labels(spec, context),
                        extractClusterSequence(context, spec));
                break;

            default:
                throw new IllegalStateException(String.format("Unsupported strategy %s", typeOfDeployStrategy));
        }
    }

    private enum DeployStrategy {
        DEPRECATED_SEQUENTIAL,
        SEQUENTIAL,
        PARALLEL
    }

    private Map<String, Map<String, YTreeNode>> labels(ReplicaSetUnitSpec spec, DeployUnitContext context) {
        return StreamEx.of(spec.extractClusters())
                .toMap(cluster -> Map.of(
                        DEPLOY_LABEL_KEY,
                        YTree.mapBuilder()
                                .key(AUTOSCALER_ID_LABEL_KEY)
                                .value(spec.getSettingsForCluster(cluster).hasAutoscale() ? id : EMPTY_ID)
                                .key(STAGE_LABEL_KEY)
                                .value(context.getStageContext().getStageId())
                                .buildMap()));
    }

    @Override
    public void shutdown() {
        perCluster.shutdown();
    }

    @Override
    public AggregatedRawStatus getStatus() {
        return AggregatedRawStatus.merge(perCluster.getClusterStatuses());
    }

    @Override
    public void addStats(DeployUnitStats.Builder builder) {
        perCluster.addStats(builder);
    }

    @VisibleForTesting
    static TReplicaSetSpec mergeSpec(ReplicaSetUnitSpec spec, DeployUnitContext context, String cluster,
                                     SpecPatcher<TPodTemplateSpec.Builder> podSpecPatcher, Converter converter) {
        TReplicaSetSpec.Builder builder = spec.getSpecTemplate().toBuilder();
        spec.getPerClusterSettings().get(cluster).getDeploymentStrategy().ifPresent(strategy ->
                builder.setDeploymentStrategy(converter.toProto(strategy)));
        podSpecPatcher.patch(builder.getPodTemplateSpecBuilder(), context, new YTreeBuilder());
        builder.setRevisionId(getDeducedRevisionString(context.getSpec().getRevision()));

        ReplicaSetUnitSpec.PerClusterSettings perClusterSettings = spec.getPerClusterSettings().get(cluster);
        if (perClusterSettings.hasPodCount()) {
            builder.setReplicaCount(perClusterSettings.getPodCount().get());
        }
        builder.setAccountId(context.getStageContext().getAccountId());
        if (!builder.hasConstraints() || builder.getConstraints().getAntiaffinityConstraintsCount() == 0) {
            builder.getConstraintsBuilder().addAntiaffinityConstraints(converter.toProto(DEFAULT_CONSTRAINT));
        }
        return builder.build();
    }

    // uint32 to int conversion string representation fix DEPLOY-2091
    public static String getDeducedRevisionString(int revisionId) {
        if (revisionId >= 0) {
            return String.valueOf(revisionId);
        }
        long pow = 1L << 32;
        long deducedRevisionId = pow + revisionId;
        return String.valueOf(deducedRevisionId);
    }
}
