package ru.yandex.infra.stage.protobuf;

import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.Timestamp;
import com.typesafe.config.Config;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.bolts.collection.Either;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.RelationMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.stage.TDockerImageContents;
import ru.yandex.infra.stage.TDockerState;
import ru.yandex.infra.stage.TSecret;
import ru.yandex.infra.stage.TSingleResourceState;
import ru.yandex.infra.stage.deployunit.DeployUnitTimeline;
import ru.yandex.infra.stage.docker.DockerState;
import ru.yandex.infra.stage.dto.AggregatedCondition;
import ru.yandex.infra.stage.dto.AllComputeResources;
import ru.yandex.infra.stage.dto.BoxJugglerConfig;
import ru.yandex.infra.stage.dto.Checksum;
import ru.yandex.infra.stage.dto.Condition;
import ru.yandex.infra.stage.dto.CoredumpConfig;
import ru.yandex.infra.stage.dto.CoredumpOutputPolicy;
import ru.yandex.infra.stage.dto.DeployProgress;
import ru.yandex.infra.stage.dto.DeployReadyCriterion;
import ru.yandex.infra.stage.dto.DeploySpeed;
import ru.yandex.infra.stage.dto.DeployUnitOverrides;
import ru.yandex.infra.stage.dto.DeployUnitOverrides.PerClusterOverrides;
import ru.yandex.infra.stage.dto.DeployUnitSpec;
import ru.yandex.infra.stage.dto.DeployUnitSpecDetails;
import ru.yandex.infra.stage.dto.DeployUnitStatus;
import ru.yandex.infra.stage.dto.DeployUnitStatusDetails;
import ru.yandex.infra.stage.dto.DeploymentStrategy;
import ru.yandex.infra.stage.dto.DockerImageContents;
import ru.yandex.infra.stage.dto.DockerImageDescription;
import ru.yandex.infra.stage.dto.DownloadableResource;
import ru.yandex.infra.stage.dto.DynamicResourceMeta;
import ru.yandex.infra.stage.dto.DynamicResourceRevisionStatus;
import ru.yandex.infra.stage.dto.DynamicResourceSpec;
import ru.yandex.infra.stage.dto.DynamicResourceStatus;
import ru.yandex.infra.stage.dto.HorizontalPodAutoscalerMeta;
import ru.yandex.infra.stage.dto.InfraComponents;
import ru.yandex.infra.stage.dto.LogbrokerCommunalTopicRequest;
import ru.yandex.infra.stage.dto.LogbrokerConfig;
import ru.yandex.infra.stage.dto.LogbrokerCustomTopicRequest;
import ru.yandex.infra.stage.dto.LogbrokerDestroyPolicy;
import ru.yandex.infra.stage.dto.LogbrokerTopicDescription;
import ru.yandex.infra.stage.dto.LogbrokerTopicRequest;
import ru.yandex.infra.stage.dto.LogrotateConfig;
import ru.yandex.infra.stage.dto.McrsUnitSpec;
import ru.yandex.infra.stage.dto.McrsUnitStatus;
import ru.yandex.infra.stage.dto.NetworkDefaults;
import ru.yandex.infra.stage.dto.PodAgentConfig;
import ru.yandex.infra.stage.dto.ReplicaSetDeploymentStrategy;
import ru.yandex.infra.stage.dto.ReplicaSetUnitSpec;
import ru.yandex.infra.stage.dto.ReplicaSetUnitStatus;
import ru.yandex.infra.stage.dto.RuntimeDeployControls;
import ru.yandex.infra.stage.dto.SandboxResourceInfo;
import ru.yandex.infra.stage.dto.Secret;
import ru.yandex.infra.stage.dto.SecretRef;
import ru.yandex.infra.stage.dto.SecretSelector;
import ru.yandex.infra.stage.dto.SecuritySettings;
import ru.yandex.infra.stage.dto.SidecarVolumeSettings;
import ru.yandex.infra.stage.dto.StageSpec;
import ru.yandex.infra.stage.dto.StageStatus;
import ru.yandex.infra.stage.dto.TvmApp;
import ru.yandex.infra.stage.dto.TvmClient;
import ru.yandex.infra.stage.dto.TvmConfig;
import ru.yandex.infra.stage.dto.datamodel.AntiaffinityConstraint;
import ru.yandex.infra.stage.podspecs.ResourceWithMeta;
import ru.yandex.infra.stage.podspecs.SandboxResourceMeta;
import ru.yandex.infra.stage.util.OptionalUtils;
import ru.yandex.yp.client.api.Autogen;
import ru.yandex.yp.client.api.Autogen.THorizontalPodAutoscalerMeta;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.DynamicResource;
import ru.yandex.yp.client.api.EChecksumType;
import ru.yandex.yp.client.api.EConditionStatus;
import ru.yandex.yp.client.api.TAggregatedCondition;
import ru.yandex.yp.client.api.TBoxJugglerConfig;
import ru.yandex.yp.client.api.TChecksum;
import ru.yandex.yp.client.api.TCondition;
import ru.yandex.yp.client.api.TCoredumpAggregator;
import ru.yandex.yp.client.api.TCoredumpPolicy;
import ru.yandex.yp.client.api.TCoredumpProcessor;
import ru.yandex.yp.client.api.TDeployProgress;
import ru.yandex.yp.client.api.TDeployReadyCriterion;
import ru.yandex.yp.client.api.TDeploySpeed;
import ru.yandex.yp.client.api.TDeployUnitApproval;
import ru.yandex.yp.client.api.TDeployUnitOverrides;
import ru.yandex.yp.client.api.TDeployUnitSpec;
import ru.yandex.yp.client.api.TDeployUnitSpec.TReplicaSetDeploy.TPerClusterSettings;
import ru.yandex.yp.client.api.TDeployUnitStatus;
import ru.yandex.yp.client.api.TDockerImageDescription;
import ru.yandex.yp.client.api.TDownloadableResource;
import ru.yandex.yp.client.api.TInfraComponents;
import ru.yandex.yp.client.api.TLogbrokerConfig;
import ru.yandex.yp.client.api.TLogrotateConfig;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TNetworkDefaults;
import ru.yandex.yp.client.api.TReplicaSetScaleSpec;
import ru.yandex.yp.client.api.TReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;
import ru.yandex.yp.client.api.TRuntimeDeployControls;
import ru.yandex.yp.client.api.TSandboxResourceInfo;
import ru.yandex.yp.client.api.TSidecarVolume;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.client.api.TTvmApp;
import ru.yandex.yp.client.api.TTvmClient;
import ru.yandex.yp.client.api.TTvmConfig;
import ru.yandex.yp.client.pods.TSandboxResource;

import static ru.yandex.infra.stage.protobuf.ConverterUtils.convertList;
import static ru.yandex.infra.stage.protobuf.ConverterUtils.convertMap;

public class Converter {

    @VisibleForTesting
    public static final String PROTO_CONVERTER_CONFIG_KEY = "proto_converter";
    @VisibleForTesting
    public static final String WRITE_FAILURE_CONDITION_CONFIG_KEY = "write_failure_condition";
    @VisibleForTesting
    public static final String PROJECT_PROMISE_CONFIG_KEY = "project_promise";
    @VisibleForTesting
    public static final String USE_EXTENDED_REPLICA_SET_STATUSES_CONFIG_KEY = "use_extended_replica_set_statuses";

    public static final String ADD_DEPLOY_UNIT_TIMELINE_STATUSES_CONFIG_KEY = "add_deploy_unit_timeline_statuses";

    private final OneofDerivedConverter<DeployUnitSpecDetails, TDeployUnitSpec> deployUnitSpecConverter;

    private final OneofDerivedConverter<DeployUnitStatusDetails, TDeployUnitStatus> deployUnitStatusConverter;

    private final OneofDerivedConverter<LogbrokerTopicRequest, TLogbrokerConfig> logbrokerTopicRequestConverter;

    private static final EnumByMapConverter<Condition.Status, EConditionStatus> CONDITION_STATUS_CONVERTER =
            new EnumByMapConverter<>(Condition.Status.class, EConditionStatus.class, ImmutableMap.of(
                    Condition.Status.UNRECOGNIZED, EConditionStatus.UNRECOGNIZED,
                    Condition.Status.FALSE, EConditionStatus.CS_FALSE,
                    Condition.Status.TRUE, EConditionStatus.CS_TRUE,
                    Condition.Status.UNKNOWN, EConditionStatus.CS_UNKNOWN
            ));


    private static final EnumByMapConverter<TvmConfig.Mode, TTvmConfig.EMode> TVM_CONFIG_MODE_CONVERTER =
            new EnumByMapConverter<>(TvmConfig.Mode.class, TTvmConfig.EMode.class, ImmutableMap.of(
                    TvmConfig.Mode.UNRECOGNIZED, TTvmConfig.EMode.UNRECOGNIZED,
                    TvmConfig.Mode.UNKNOWN, TTvmConfig.EMode.UNKNOWN,
                    TvmConfig.Mode.DISABLED, TTvmConfig.EMode.DISABLED,
                    TvmConfig.Mode.ENABLED, TTvmConfig.EMode.ENABLED
            ));

    private static final EnumByMapConverter<Checksum.Type, EChecksumType> CHECKSUM_TYPE_CONVERTER =
            new EnumByMapConverter<>(Checksum.Type.class, EChecksumType.class, ImmutableMap.of(
                    Checksum.Type.UNRECOGNIZED, EChecksumType.UNRECOGNIZED,
                    Checksum.Type.EMPTY, EChecksumType.EMPTY,
                    Checksum.Type.MD5, EChecksumType.MD5,
                    Checksum.Type.SHA256, EChecksumType.SHA256
            ));

    private static final EnumByMapConverter<LogbrokerConfig.SidecarBringupMode, TLogbrokerConfig.ESidecarBringupMode> LOGBROKER_SIDECAR_BRINGUP_MODE_CONVERTER =
            new EnumByMapConverter<>(LogbrokerConfig.SidecarBringupMode.class, TLogbrokerConfig.ESidecarBringupMode.class, ImmutableMap.of(
                    LogbrokerConfig.SidecarBringupMode.UNRECOGNIZED, TLogbrokerConfig.ESidecarBringupMode.UNRECOGNIZED,
                    LogbrokerConfig.SidecarBringupMode.UNKNOWN, TLogbrokerConfig.ESidecarBringupMode.UNKNOWN,
                    LogbrokerConfig.SidecarBringupMode.DEFAULT, TLogbrokerConfig.ESidecarBringupMode.DEFAULT,
                    LogbrokerConfig.SidecarBringupMode.MANDATORY, TLogbrokerConfig.ESidecarBringupMode.MANDATORY
            ));

    private static final EnumByMapConverter<SidecarVolumeSettings.StorageClass, TSidecarVolume.EStorageClass> SIDECAR_VOLUME_STORAGE_CLASS_CONVERTER =
            new EnumByMapConverter<>(SidecarVolumeSettings.StorageClass.class, TSidecarVolume.EStorageClass.class, ImmutableMap.of(
                    SidecarVolumeSettings.StorageClass.UNRECOGNIZED, TSidecarVolume.EStorageClass.UNRECOGNIZED,
                    SidecarVolumeSettings.StorageClass.AUTO, TSidecarVolume.EStorageClass.AUTO,
                    SidecarVolumeSettings.StorageClass.HDD, TSidecarVolume.EStorageClass.HDD,
                    SidecarVolumeSettings.StorageClass.SSD, TSidecarVolume.EStorageClass.SSD
            ));

    private final boolean writeFailureCondition;
    private final Config projectPromiseConfig;

    public Converter(Config config) {
        this(config, new OneofDerivedConverter<>(
                        TDeployUnitSpec.getDescriptor().getOneofs().get(0),
                        ImmutableMap.of(
                                McrsUnitSpec.class, TDeployUnitSpec.TMultiClusterReplicaSetDeploy.class,
                                ReplicaSetUnitSpec.class, TDeployUnitSpec.TReplicaSetDeploy.class
                        )
                ),
                new OneofDerivedConverter<>(
                        TDeployUnitStatus.getDescriptor().getOneofs().get(0),
                        ImmutableMap.of(
                                McrsUnitStatus.class, TDeployUnitStatus.TMultiClusterReplicaSetDeploy.class,
                                ReplicaSetUnitStatus.class, TDeployUnitStatus.TReplicaSetDeploy.class
                        )
                ),
                new OneofDerivedConverter<>(
                        TLogbrokerConfig.getDescriptor().getOneofs().get(0),
                        ImmutableMap.of(
                                LogbrokerCommunalTopicRequest.class, TLogbrokerConfig.TLogbrokerCommunalTopicRequest.class,
                                LogbrokerCustomTopicRequest.class, TLogbrokerConfig.TLogbrokerCustomTopicRequest.class
                        )
                ));
    }

    public Converter(Config config,
                     OneofDerivedConverter<DeployUnitSpecDetails, TDeployUnitSpec> deployUnitSpecConverter,
                     OneofDerivedConverter<DeployUnitStatusDetails, TDeployUnitStatus> deployUnitStatusConverter,
                     OneofDerivedConverter<LogbrokerTopicRequest, TLogbrokerConfig> logbrokerTopicRequestConverter) {
        this.writeFailureCondition = config.getConfig(PROTO_CONVERTER_CONFIG_KEY).getBoolean(WRITE_FAILURE_CONDITION_CONFIG_KEY);
        this.projectPromiseConfig = config.getConfig(PROJECT_PROMISE_CONFIG_KEY);

        this.deployUnitSpecConverter = deployUnitSpecConverter;
        this.deployUnitStatusConverter = deployUnitStatusConverter;
        this.logbrokerTopicRequestConverter = logbrokerTopicRequestConverter;
    }

    private boolean getProjectPromiseConfigFlag(String flagConfigKey) {
        return projectPromiseConfig.getBoolean(flagConfigKey);
    }

    public AntiaffinityConstraint fromProto(DataModel.TAntiaffinityConstraint proto) {
        return new AntiaffinityConstraint(AntiaffinityConstraint.Zone.fromYpName(proto.getKey()), proto.getMaxPods());
    }

    public DataModel.TAntiaffinityConstraint toProto(AntiaffinityConstraint dto) {
        return DataModel.TAntiaffinityConstraint.newBuilder()
                .setKey(dto.getZone().getYpName())
                .setMaxPods(dto.getMaxPods())
                .build();
    }

    public static DockerImageDescription fromProto(TDockerImageDescription proto) {
        return new DockerImageDescription(proto.getRegistryHost(), proto.getName(), proto.getTag());
    }

    public static TDockerImageDescription toProto(DockerImageDescription dto) {
        return TDockerImageDescription.newBuilder()
                .setRegistryHost(dto.getRegistryHost())
                .setName(dto.getName())
                .setTag(dto.getTag())
                .build();
    }

    public DockerState fromProto(TDockerState proto) {
        Map<DockerImageDescription, DockerImageContents> images = new HashMap<>();
        proto.getImagesList().forEach(item ->
                images.put(fromProto(item.getDescription()), fromProto(item)));
        return new DockerState(images);
    }

    public static TSecret toProto(Secret dto) {
        var secretRef = dto.getSecretRef();

        return TSecret.newBuilder()
                .setId(secretRef.getUuid())
                .setVersion(secretRef.getVersion())
                .setDelegationToken(dto.getDelegationToken())
                .build();
    }

    public static Secret fromProto(TSecret proto) {
        var secretRef = new SecretRef(proto.getId(), proto.getVersion());
        return new Secret(secretRef, proto.getDelegationToken());
    }

    public static DataModel.TPodSpec.TSecret toYpProto(SecretRef secretRef, String delegationToken) {
        return toYpProto(new Secret(secretRef, delegationToken));
    }

    public static DataModel.TPodSpec.TSecret toYpProto(Secret dto) {
        var secretRef = toProto(dto.getSecretRef());

        return DataModel.TPodSpec.TSecret.newBuilder()
                .setSecretId(secretRef.getSecretId())
                .setSecretVersion(secretRef.getSecretVersion())
                .setDelegationToken(dto.getDelegationToken())
                .build();
    }

    public static Secret fromYpProto(DataModel.TPodSpec.TSecret proto) {
        var secretRef = fromProto(secretToRef(proto));
        return new Secret(secretRef, proto.getDelegationToken());
    }

    public static DataModel.TSecretRef secretToRef(DataModel.TPodSpec.TSecret proto) {
        return DataModel.TSecretRef.newBuilder()
                .setSecretId(proto.getSecretId())
                .setSecretVersion(proto.getSecretVersion())
                .build();
    }

    public static DataModel.TSecretRef toProto(SecretRef dto) {
        return DataModel.TSecretRef.newBuilder()
                .setSecretId(dto.getUuid())
                .setSecretVersion(dto.getVersion())
                .build();
    }

    public static SecretRef fromProto(DataModel.TSecretRef proto) {
        return new SecretRef(
                proto.getSecretId(),
                proto.getSecretVersion()
        );
    }

    public static ru.yandex.yp.client.pods.SecretSelector toProto(SecretSelector dto) {
        return ru.yandex.yp.client.pods.SecretSelector.newBuilder()
                .setAlias(dto.getAlias())
                .setId(dto.getKey())
                .build();
    }

    public SecretSelector fromProto(ru.yandex.yp.client.pods.SecretSelector proto) {
        return new SecretSelector(proto.getAlias(), proto.getId());
    }

    public TDockerState toProto(DockerState dto) {
        TDockerState.Builder builder = TDockerState.newBuilder();
        dto.getImages().forEach((description, contents) -> builder.addImages(toProto(contents)));
        return builder.build();
    }

    public static Checksum fromProto(TChecksum proto) {
        return new Checksum(proto.getValue(), CHECKSUM_TYPE_CONVERTER.fromProto(proto.getType()));
    }

    public static TChecksum toProto(Checksum dto) {
        return TChecksum.newBuilder()
                .setType(CHECKSUM_TYPE_CONVERTER.toProto(dto.getType()))
                .setValue(dto.getValue())
                .build();
    }

    public static DownloadableResource fromProto(TDownloadableResource proto) {
        return new DownloadableResource(proto.getUrl(), fromProto(proto.getChecksum()));
    }

    public static TDownloadableResource toProto(DownloadableResource dto) {
        return TDownloadableResource.newBuilder()
                .setChecksum(toProto(dto.getChecksum()))
                .setUrl(dto.getUrl())
                .build();
    }

    public SidecarVolumeSettings fromProto(TSidecarVolume proto) {
        return new SidecarVolumeSettings(SIDECAR_VOLUME_STORAGE_CLASS_CONVERTER.fromProto(proto.getStorageClass()));
    }

    public TSidecarVolume toProto(SidecarVolumeSettings dto) {
        return TSidecarVolume.newBuilder()
                .setStorageClass(SIDECAR_VOLUME_STORAGE_CLASS_CONVERTER.toProto(dto.getStorageClass()))
                .build();
    }

    public static DockerImageContents fromProto(TDockerImageContents proto) {
        return new DockerImageContents(fromProto(proto.getDescription()),
                convertList(proto.getLayersList(), Converter::fromProto),
                proto.getCommandList(),
                proto.getEntryPointList(),
                OptionalUtils.emptyStringAsEmptyOptional(proto.getUser()),
                OptionalUtils.emptyStringAsEmptyOptional(proto.getGroup()),
                OptionalUtils.emptyStringAsEmptyOptional(proto.getWorkingDir()),
                proto.getEnvironmentMap(),
                OptionalUtils.emptyStringAsEmptyOptional(proto.getDigest()));
    }

    public static TDockerImageContents toProto(DockerImageContents dto) {
        TDockerImageContents.Builder builder = TDockerImageContents.newBuilder();
        builder.setDescription(toProto(dto.getDescription()));
        builder.addAllLayers(convertList(dto.getLayers(), Converter::toProto));
        builder.addAllCommand(dto.getCommand());
        builder.addAllEntryPoint(dto.getEntryPoint());
        dto.getUser().ifPresent(builder::setUser);
        dto.getGroup().ifPresent(builder::setGroup);
        dto.getWorkingDir().ifPresent(builder::setWorkingDir);
        builder.putAllEnvironment(dto.getEnvironment());
        dto.getDigest().ifPresent(builder::setDigest);
        return builder.build();
    }

    public PodAgentConfig fromProto(DataModel.TPodSpec.TPodAgentDeploymentMeta proto) {
        Optional<SidecarVolumeSettings> volumeSettings = Optional.empty();

        if (proto.hasSidecarVolume()) {
            volumeSettings = Optional.of(fromProto(proto.getSidecarVolume()));
        }

        return new PodAgentConfig(volumeSettings);
    }

    public DataModel.TPodSpec.TPodAgentDeploymentMeta.Builder toProtoPartial(PodAgentConfig dto, DataModel.TPodSpec.TPodAgentDeploymentMeta.Builder builder) {
        dto.getSidecarVolumeSettings().ifPresent(volSettings -> builder.setSidecarVolume(toProto(volSettings)));
        return builder;
    }


    public static BoxJugglerConfig fromProto(TBoxJugglerConfig proto) {
        Optional<DownloadableResource> jugglerBinary = Optional.empty();
        if (proto.hasJugglerAgentBinary()) {
            jugglerBinary = Optional.of(fromProto(proto.getJugglerAgentBinary()));
        }
        return new BoxJugglerConfig(convertList(proto.getArchivedChecksList(), Converter::fromProto),
                OptionalUtils.zeroAsEmptyOptional(proto.getPort()), jugglerBinary);
    }

    public static TBoxJugglerConfig toProto(BoxJugglerConfig dto) {
        TBoxJugglerConfig.Builder builder = TBoxJugglerConfig.newBuilder()
                .addAllArchivedChecks(convertList(dto.getArchivedChecks(), Converter::toProto));
        dto.getPort().ifPresent(builder::setPort);
        dto.getJugglerBinary().ifPresent(jugglerBinary -> builder.setJugglerAgentBinary(toProto(jugglerBinary)));
        return builder.build();
    }

    public LogbrokerDestroyPolicy fromProto(TLogbrokerConfig.TLogbrokerDestroyPolicy proto) {
        OptionalInt maxTries = OptionalUtils.zeroAsEmptyOptional(proto.getMaxTries());
        OptionalLong restartPeriodMs = OptionalUtils.zeroAsEmptyOptional(proto.getRestartPeriodMs());

        return new LogbrokerDestroyPolicy(maxTries, restartPeriodMs);
    }

    public TLogbrokerConfig.TLogbrokerDestroyPolicy toProto(LogbrokerDestroyPolicy logbrokerDestroyPolicy) {
        var builder = TLogbrokerConfig.TLogbrokerDestroyPolicy.newBuilder();
        builder.setMaxTries(logbrokerDestroyPolicy.getMaxTries());
        builder.setRestartPeriodMs(logbrokerDestroyPolicy.getRestartPeriodMs());
        return builder.build();
    }

    public LogbrokerCommunalTopicRequest fromProto(TLogbrokerConfig.TLogbrokerCommunalTopicRequest proto) {
        return LogbrokerCommunalTopicRequest.INSTANCE;
    }

    public TLogbrokerConfig.TLogbrokerCommunalTopicRequest toProto(LogbrokerCommunalTopicRequest dto) {
        return TLogbrokerConfig.TLogbrokerCommunalTopicRequest.newBuilder()
                .build();
    }

    public LogbrokerCustomTopicRequest fromProto(TLogbrokerConfig.TLogbrokerCustomTopicRequest proto) {
        var topicDescription = new LogbrokerTopicDescription(proto.getTvmClientId(), proto.getTopicName());
        return new LogbrokerCustomTopicRequest(
                topicDescription,
                fromProto(proto.getSecretSelector())
        );
    }

    public TLogbrokerConfig.TLogbrokerCustomTopicRequest toProto(LogbrokerCustomTopicRequest dto) {
        var topicDescription = dto.getTopicDescription();

        return TLogbrokerConfig.TLogbrokerCustomTopicRequest.newBuilder()
                .setTvmClientId(topicDescription.getTvmClientId())
                .setTopicName(topicDescription.getName())
                .setSecretSelector(toProto(dto.getSecretSelector()))
                .build();
    }

    public static AllComputeResources fromProto(DataModel.TPodSpec.TResourceRequestsOrBuilder proto) {
        return new AllComputeResources(
            proto.getVcpuGuarantee(),
            proto.getVcpuLimit(),
            proto.getMemoryGuarantee(),
            proto.getMemoryLimit(),
            proto.getAnonymousMemoryLimit(),
            AllComputeResources.UNKNOWN_DISK_CAPACITY,
            proto.getThreadLimit()
        );
    }

    public static DataModel.TPodSpec.TResourceRequests toProto(AllComputeResources dto) {
        return DataModel.TPodSpec.TResourceRequests.newBuilder()
                .setVcpuGuarantee(dto.getVcpuGuarantee())
                .setVcpuLimit(dto.getVcpuLimit())
                .setMemoryGuarantee(dto.getMemoryGuarantee())
                .setMemoryLimit(dto.getMemoryLimit())
                .setAnonymousMemoryLimit(dto.getAnonymousMemoryLimit())
                .setThreadLimit(dto.getThreadLimit())
                .build();
    }

    public LogbrokerConfig fromProto(TLogbrokerConfig proto) {
        Optional<DownloadableResource> pushAgentBinary = Optional.empty();
        Optional<SidecarVolumeSettings> sidecarVolumeSettings = Optional.empty();

        if (proto.hasPushAgentLayer()) {
            pushAgentBinary = Optional.of(fromProto(proto.getPushAgentLayer()));
        }

        if (proto.hasSidecarVolume()) {
            sidecarVolumeSettings = Optional.of(fromProto(proto.getSidecarVolume()));
        }

        Optional<String> logsVirtualDiskIdRef = OptionalUtils.emptyStringAsEmptyOptional(proto.getLogsVirtualDiskIdRef());
        LogbrokerConfig.SidecarBringupMode sidecarBringupMode = LOGBROKER_SIDECAR_BRINGUP_MODE_CONVERTER.fromProto(proto.getSidecarBringupMode());
        LogbrokerDestroyPolicy destroyPolicy = fromProto(proto.getDestroyPolicy());

        var topicRequest = logbrokerTopicRequestConverter.fromProto(
                proto, this, LogbrokerCommunalTopicRequest.INSTANCE
        );

        Optional<AllComputeResources> podAdditionalResourcesRequest = Optional.empty();
        if (proto.hasPodAdditionalResourcesRequest()) {
            podAdditionalResourcesRequest = Optional.of(
                    fromProto(proto.getPodAdditionalResourcesRequest())
            );
        }

        return new LogbrokerConfig(
                pushAgentBinary,
                sidecarBringupMode,
                logsVirtualDiskIdRef,
                sidecarVolumeSettings,
                destroyPolicy,
                topicRequest,
                podAdditionalResourcesRequest
        );
    }

    public TLogbrokerConfig toProto(LogbrokerConfig dto) {
        TLogbrokerConfig.Builder builder = TLogbrokerConfig.newBuilder();
        dto.getLogbrokerAgentLayer().ifPresent(pushAgentBinary -> builder.setPushAgentLayer(toProto(pushAgentBinary)));
        dto.getLogsVirtualDiskIdRef().ifPresent(builder::setLogsVirtualDiskIdRef);
        dto.getSidecarVolumeSettings().ifPresent(volSettings -> builder.setSidecarVolume(toProto(volSettings)));
        builder.setSidecarBringupMode(LOGBROKER_SIDECAR_BRINGUP_MODE_CONVERTER.toProto(dto.getSidecarBringupMode()));
        builder.setDestroyPolicy(toProto(dto.getDestroyPolicy()));
        logbrokerTopicRequestConverter.setProtoField(builder, dto.getTopicRequest(), this);
        dto.getPodAdditionalResourcesRequest().map(Converter::toProto).ifPresent(builder::setPodAdditionalResourcesRequest);
        return builder.build();
    }

    public SecuritySettings fromProto(TDeployUnitSpec.TSecuritySettings proto) {
        return new SecuritySettings(proto.getDisableChildOnlyIsolationIfEnabledByDefault(), proto.getDisableSecretEnvIfEnabledByDefault());
    }

    public TDeployUnitSpec.TSecuritySettings toProto(SecuritySettings dto) {
         return TDeployUnitSpec.TSecuritySettings.newBuilder()
                 .setDisableChildOnlyIsolationIfEnabledByDefault(dto.disableDefaultlyEnabledChildOnlyIsolation())
                 .setDisableSecretEnvIfEnabledByDefault(dto.disableDefaultlyEnabledSecretEnv())
                 .build();
    }

    public DeployUnitSpec fromProto(TDeployUnitSpec proto) {
        Optional<DeployUnitSpec.DeploySettings> deploySettings = Optional.empty();
        if (proto.hasDeploySettings()) {
            deploySettings = Optional.of(fromProto(proto.getDeploySettings()));
        }
        Optional<TvmConfig> tvmConfig = Optional.empty();
        if (proto.hasTvmConfig()) {
            tvmConfig = Optional.of(fromProto(proto.getTvmConfig()));
        }
        Optional<SandboxResourceInfo> tvmToolSidecar = Optional.empty();
        if (proto.hasTvmSandboxInfo()) {
            TSandboxResourceInfo tvmSandboxInfo = proto.getTvmSandboxInfo();
            tvmToolSidecar = Optional.of(fromProto(tvmSandboxInfo));
        }

        Optional<SandboxResourceInfo> podAgentResourceInfo = proto.hasPodAgentSandboxInfo() && proto.getPodAgentSandboxInfo().getRevision() != 0 ?
                Optional.of(fromProto(proto.getPodAgentSandboxInfo())) : Optional.empty();

        Optional<SandboxResourceInfo> podAgentLayerResourceInfo = proto.hasPodAgentLayerSandboxInfo() && proto.getPodAgentLayerSandboxInfo().getRevision() != 0 ?
                Optional.of(fromProto(proto.getPodAgentLayerSandboxInfo())) : Optional.empty();

        Optional<SandboxResourceInfo> logbrokerToolsResourceInfo = proto.hasLogbrokerToolsSandboxInfo() && proto.getLogbrokerToolsSandboxInfo().getRevision() != 0 ?
                Optional.of(fromProto(proto.getLogbrokerToolsSandboxInfo())) : Optional.empty();

        Optional<SandboxResourceInfo> druLayerResourceInfo = proto.hasDynamicResourceUpdaterSandboxInfo() && proto.getDynamicResourceUpdaterSandboxInfo().getRevision() != 0 ?
                Optional.of(fromProto(proto.getDynamicResourceUpdaterSandboxInfo())) : Optional.empty();

        Map<String, LogrotateConfig> logrotateConfig = convertMap(proto.getLogrotateConfigsMap(), this::fromProto);

        Map<String, CoredumpConfig> coredumpConfig = convertMap(proto.getCoredumpConfigMap(), this::fromProto);

        Optional<SandboxResourceInfo> coredumpToolSidecar =
                proto.hasCoredumpToolsSandboxInfo() && proto.getCoredumpToolsSandboxInfo().getRevision() != 0 ?
                        Optional.of(fromProto(proto.getCoredumpToolsSandboxInfo())) : Optional.empty();

        Optional<SandboxResourceInfo> gdbLayerSidecar =
                proto.hasGdbLayerSandboxInfo() && proto.getGdbLayerSandboxInfo().getRevision() != 0 ?
                        Optional.of(fromProto(proto.getGdbLayerSandboxInfo())) : Optional.empty();

        Optional<SecuritySettings> securitySettings = Optional.empty();
        if (proto.hasSecuritySettings()) {
            securitySettings = Optional.of(fromProto(proto.getSecuritySettings()));
        }

        return new DeployUnitSpec(
                proto.getRevision(),
                proto.getPatchersRevision(),
                fromProto(proto.getNetworkDefaults()), tvmConfig,
                deployUnitSpecConverter.fromProto(proto, this),
                convertMap(proto.getImagesForBoxesMap(), Converter::fromProto),
                proto.getEndpointSetsList().stream().collect(Collectors.toMap(
                        TDeployUnitSpec.TEndpointSetTemplate::getId,
                        endpointSet -> {
                            DataModel.TEndpointSetSpec.Builder endpointSetSpec = DataModel.TEndpointSetSpec.newBuilder();
                            if (endpointSet.getPort() != 0) {
                                endpointSetSpec.setPort(endpointSet.getPort());
                            }
                            if (endpointSet.getLivenessLimitRatio() != 0) {
                                endpointSetSpec.setLivenessLimitRatio(endpointSet.getLivenessLimitRatio());
                            }
                            if (!endpointSet.getProtocol().equals("")) {
                                endpointSetSpec.setProtocol(endpointSet.getProtocol());
                            }
                            return endpointSetSpec.build();
                        })),
                convertMap(proto.getBoxJugglerConfigsMap(), Converter::fromProto), logrotateConfig,
                fromProto(proto.getLogbrokerConfig()),
                securitySettings,
                proto.getSoxService(),
                proto.getCollectPortometricsFromSidecars(), coredumpConfig,
                tvmToolSidecar,
                podAgentResourceInfo,
                podAgentLayerResourceInfo,
                logbrokerToolsResourceInfo,
                druLayerResourceInfo,
                coredumpToolSidecar,
                gdbLayerSidecar,
                deploySettings);
    }

    public DeployUnitSpec.DeploySettings fromProto(TDeployUnitSpec.TDeploySettings deploySettings) {
        List<DeployUnitSpec.DeploySettings.ClusterSettings> clusters = deploySettings.getClusterSequenceList().stream()
                .map(v -> new DeployUnitSpec.DeploySettings.ClusterSettings(v.getYpCluster(), v.getNeedApproval()))
                .collect(Collectors.toList());
        return new DeployUnitSpec.DeploySettings(clusters, deploySettings.getDeployStrategy());
    }

    public NetworkDefaults fromProto(TNetworkDefaults value) {
        return new NetworkDefaults(value.getNetworkId(),
                value.getOverrideIp6AddressRequests(),
                value.getOverrideIp6SubnetRequests(),
                Optional.ofNullable(value.getIp4AddressPoolId()).filter(StringUtils::isNotEmpty),
                value.getVirtualServiceIdsList());
    }

    public TNetworkDefaults toProto(NetworkDefaults dto) {
        TNetworkDefaults.Builder builder = TNetworkDefaults.newBuilder()
                .setNetworkId(dto.getNetworkId())
                .setOverrideIp6AddressRequests(dto.isOverrideIp6AddressRequests())
                .setOverrideIp6SubnetRequests(dto.isOverrideIp6SubnetRequests())
                .addAllVirtualServiceIds(dto.getVirtualServiceIds());
        dto.getIp4AddressPoolId().ifPresent(builder::setIp4AddressPoolId);
        return builder.build();
    }

    public LogrotateConfig fromProto(TLogrotateConfig value) {
        return new LogrotateConfig(value.getRawConfig(), value.getRunPeriodMillisecond());
    }

    public TLogrotateConfig toProto(LogrotateConfig dto) {
        return TLogrotateConfig.newBuilder()
                .setRawConfig(dto.getRawConfig())
                .setRunPeriodMillisecond(dto.getRunPeriodMillisecond())
                .build();
    }

    public CoredumpConfig fromProto(TCoredumpPolicy value) {
        TCoredumpProcessor cdp = value.getCoredumpProcessor();

        Optional<String> outputPath = Optional.ofNullable(cdp.getOutput()).filter(StringUtils::isNotEmpty);
        Optional<String> outputVolumeId = Optional.ofNullable(cdp.getVolumeId()).filter(StringUtils::isNotEmpty);

        Optional<CoredumpOutputPolicy> outputPolicy = Optional.empty();
        if (outputPath.isPresent()) {
            outputPolicy = outputPath.map(CoredumpOutputPolicy::outputPath);
        } else if (outputVolumeId.isPresent()) {
            outputPolicy = outputVolumeId.map(CoredumpOutputPolicy::outputVolumeId);
        }

        return new CoredumpConfig(cdp.getCountLimit(), cdp.getTotalSizeLimitMegabytes(), cdp.getProbability(),
                cdp.getCleanupTtlSeconds(),
                cdp.getAggregator().getEnabled(),
                cdp.getAggregator().getUrl(),
                Optional.ofNullable(cdp.getAggregator().getServiceName()).filter(StringUtils::isNotEmpty),
                Optional.ofNullable(cdp.getAggregator().getCtype()).filter(StringUtils::isNotEmpty),
                outputPolicy);
    }

    public TCoredumpPolicy toProto(CoredumpConfig dto) {
        TCoredumpAggregator.Builder aggregatorBuilder = TCoredumpAggregator.newBuilder()
                .setEnabled(dto.isAggregatorEnabled())
                .setUrl(dto.getAggregatorUrl());
        dto.getServiceName().ifPresent(aggregatorBuilder::setServiceName);
        dto.getCtype().ifPresent(aggregatorBuilder::setCtype);

        TCoredumpProcessor.Builder builder = TCoredumpProcessor.newBuilder()
                .setTotalSizeLimitMegabytes(dto.getTotalSizeLimit())
                .setProbability(dto.getProbability())
                .setCountLimit(dto.getCountLimit())
                .setCleanupTtlSeconds(dto.getTtlSeconds())
                .setAggregator(aggregatorBuilder);

        dto.getOutputPolicy().ifPresent(outPolicy -> {
            String outputValue = outPolicy.getValue();
            if (outPolicy.isOutputPath()) {
                builder.setOutput(outputValue);
            } else if (outPolicy.isOutputVolumeId()) {
                builder.setVolumeId(outputValue);
            }
        });

        return TCoredumpPolicy.newBuilder()
                .setCoredumpProcessor(builder)
                .build();
    }

    public TDeployUnitSpec toProto(DeployUnitSpec dto) {
        TDeployUnitSpec.Builder builder = TDeployUnitSpec.newBuilder()
                .setRevision(dto.getRevision())
                .setPatchersRevision(dto.getPatchersRevision())
                .setLogbrokerConfig(toProto(dto.getLogbrokerConfig()))
                .setNetworkDefaults(toProto(dto.getNetworkDefaults()));
        dto.getTvmConfig().ifPresent(c -> builder.setTvmConfig(toProto(c)));
        dto.getSecuritySettings().ifPresent(s -> builder.setSecuritySettings(toProto(s)));
        dto.getTvmToolResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setTvmSandboxInfo(toProto(s)));
        dto.getPodAgentResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setPodAgentSandboxInfo(toProto(s)));
        dto.getPodAgentLayerResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setPodAgentLayerSandboxInfo(toProto(s)));
        dto.getLogbrokerToolsResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setLogbrokerToolsSandboxInfo(toProto(s)));
        dto.getDruLayerResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setDynamicResourceUpdaterSandboxInfo(toProto(s)));
        dto.getCoredumpToolResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setCoredumpToolsSandboxInfo(toProto(s)));
        dto.getGdbLayerResourceInfo().filter(s -> s.getRevision() != 0).map(s -> builder.setGdbLayerSandboxInfo(toProto(s)));
        dto.getDeploySettings().ifPresent(v -> builder.setDeploySettings(toProto(v)));
        deployUnitSpecConverter.setProtoField(builder, dto.getDetails(), this);
        builder.putAllImagesForBoxes(convertMap(dto.getImagesForBoxes(), Converter::toProto));
        dto.getDeploySettings().ifPresent(v -> builder.setDeploySettings(toProto(v)));
        dto.getEndpointSets().forEach((key, value) -> {
            TDeployUnitSpec.TEndpointSetTemplate.Builder endpointSet = TDeployUnitSpec.TEndpointSetTemplate.newBuilder();
            endpointSet.setId(key);
            if (value.getPort() != 0) {
                endpointSet.setPort(value.getPort());
            }
            if (value.getLivenessLimitRatio() != 0) {
                endpointSet.setLivenessLimitRatio(value.getLivenessLimitRatio());
            }
            if (!value.getProtocol().equals("")) {
                endpointSet.setProtocol(value.getProtocol());
            }
            builder.addEndpointSets(endpointSet.build());
        });
        builder.putAllLogrotateConfigs(convertMap(dto.getLogrotateConfig(), this::toProto));
        builder.putAllBoxJugglerConfigs(convertMap(dto.getBoxJugglerConfigs(), Converter::toProto));
        builder.putAllCoredumpConfig(convertMap(dto.getCoredumpConfig(), this::toProto));
        builder.setSoxService(dto.isSoxService());
        builder.setCollectPortometricsFromSidecars(dto.isCollectPortometricsFromSidecars());
        return builder.build();
    }

    private TDeployUnitSpec.TDeploySettings toProto(DeployUnitSpec.DeploySettings deploySettings) {
        return TDeployUnitSpec.TDeploySettings.newBuilder()
                .setDeployStrategy(deploySettings.getDeployStrategy())
                .addAllClusterSequence(deploySettings.getClustersSettings().stream()
                        .map(v -> TDeployUnitSpec.TClusterSettings.newBuilder()
                                .setYpCluster(v.getName())
                                .setNeedApproval(v.isNeedApprove()).build())
                        .collect(Collectors.toList()))
                .build();
    }

    public DeployUnitStatus fromProto(TDeployUnitStatus proto) {
        long targetRevision = ConverterUtils.fromUInt32ToLong(proto.getTargetRevision());
        long latestDeployedRevision = ConverterUtils.fromUInt32ToLong(proto.getLatestDeployedRevision());
        Condition ready = fromProto(proto.getReady());

        DeployUnitTimeline.Status status = ready.isTrue()
                ? DeployUnitTimeline.Status.DEPLOYED
                : DeployUnitTimeline.Status.DEPLOYING;

        return new DeployUnitStatus(
                fromProto(proto.getInProgress()),
                fromProto(proto.getFailed()),
                fromProto(proto.getCurrentTarget()),
                deployUnitStatusConverter.fromProto(proto, this),
                fromProto(proto.getProgress()),
                proto.getYasmItype(),
                new DeployUnitTimeline(
                        targetRevision,
                        Instant.ofEpochMilli(proto.getTargetSpecTimestamp()),
                        Optional.empty(),
                        status,
                        ready,
                        latestDeployedRevision
                )
        );
//        TODO: read from deploy unit timeline after yp release
    }

    public TDeployUnitStatus toProto(DeployUnitStatus dto) {
        int targetRevision = (int) dto.getTargetRevision();
        int latestDeployedRevision = (int) dto.getLatestDeployedRevision();

        TDeployUnitStatus.Builder builder = TDeployUnitStatus.newBuilder()
                .setCurrentTarget(toProto(dto.getCurrentTarget()))
                .setInProgress(toProto(dto.getInProgress()))
                .setReady(toProto(dto.getReady()))
                .setProgress(toProto(dto.getDeployProgress()))
                .setTargetRevision(targetRevision)
                .setTargetSpecTimestamp(dto.getTargetSpecTimestamp())
                .setYasmItype(dto.getYasmItype())
                .setLatestDeployedRevision(latestDeployedRevision);

        if (writeFailureCondition) {
            builder.setFailed(toProto(dto.getFailure()));
        }
        if (getProjectPromiseConfigFlag(ADD_DEPLOY_UNIT_TIMELINE_STATUSES_CONFIG_KEY)) {
            builder.setDeployUnitTimeline(toProto(dto.getDeployUnitTimeline()));
        }
        deployUnitStatusConverter.setProtoField(builder, dto.getDetails(), this);

        return builder.build();
    }

    public DeployUnitTimeline fromProto(TDeployUnitStatus.TDeployUnitTimeline proto) {
        long targetRevision = ConverterUtils.fromUInt32ToLong(proto.getTargetRevision());
        long latestDeployedRevision = ConverterUtils.fromUInt32ToLong(proto.getLatestDeployedRevision());

        Optional<Instant> finishTimestamp = proto.getFinishTimestamp() != 0L
                ? Optional.of(Instant.ofEpochMilli(proto.getFinishTimestamp()))
                : Optional.empty();

        DeployUnitTimeline.Status status = proto.getStatus() == TDeployUnitStatus.TDeployUnitTimeline.EStatus.DEPLOYED
                ? DeployUnitTimeline.Status.DEPLOYED
                : DeployUnitTimeline.Status.DEPLOYING;

        return new DeployUnitTimeline(
                targetRevision,
                Instant.ofEpochMilli(proto.getStartTimestamp()),
                finishTimestamp,
                status,
                fromProto(proto.getLatestReadyCondition()),
                latestDeployedRevision
        );
    }

    public TDeployUnitStatus.TDeployUnitTimeline toProto(DeployUnitTimeline dto) {
        int targetRevision = (int) dto.getTargetRevision();
        int latestDeployedRevision = (int) dto.getLatestDeployedRevision();

        TDeployUnitStatus.TDeployUnitTimeline.EStatus status = dto.getStatus() == DeployUnitTimeline.Status.DEPLOYED
                ? TDeployUnitStatus.TDeployUnitTimeline.EStatus.DEPLOYED
                : TDeployUnitStatus.TDeployUnitTimeline.EStatus.DEPLOYING;

        TDeployUnitStatus.TDeployUnitTimeline.Builder builder = TDeployUnitStatus.TDeployUnitTimeline.newBuilder()
                .setTargetRevision(targetRevision)
                .setStartTimestamp(dto.getStartTimestamp().toEpochMilli())
                .setStatus(status)
                .setLatestReadyCondition(toProto(dto.getLatestReadyCondition()))
                .setLatestDeployedRevision(latestDeployedRevision);

        if (dto.getFinishTimestamp().isPresent()) {
            builder.setFinishTimestamp(dto.getFinishTimestamp().get().toEpochMilli());
        }
        return builder.build();
    }

    public Condition fromProto(TCondition proto) {
        return new Condition(CONDITION_STATUS_CONVERTER.fromProto(proto.getStatus()),
                proto.getReason(), proto.getMessage(), fromProto(proto.getLastTransitionTime()));
    }

    public TCondition toProto(Condition dto) {
        return TCondition.newBuilder()
                .setStatus(CONDITION_STATUS_CONVERTER.toProto(dto.getStatus()))
                .setReason(dto.getReason())
                .setMessage(dto.getMessage())
                .setLastTransitionTime(toProto(dto.getTimestamp()))
                .build();
    }

    public DeployProgress fromProto(TDeployProgress proto) {
        return new DeployProgress(proto.getPodsReady(), proto.getPodsInProgress(), proto.getPodsTotal());
    }

    public TDeployProgress toProto(DeployProgress dto) {
        return TDeployProgress.newBuilder()
                .setPodsReady(dto.getPodsReady())
                .setPodsInProgress(dto.getPodsInProgress())
                .setPodsTotal(dto.getPodsTotal())
                .build();
    }

    public static ResourceWithMeta fromProto(TSingleResourceState proto) {
        Optional<SandboxResourceMeta> sandboxInfo = Optional.empty();
        if (proto.hasSandboxInfo()) {
            sandboxInfo = Optional.of(fromProto(proto.getSandboxInfo()));
        }
        return new ResourceWithMeta(fromProto(proto.getDownloadInfo()), sandboxInfo);
    }

    public static TSingleResourceState toProto(ResourceWithMeta dto) {
        TSingleResourceState.Builder builder = TSingleResourceState.newBuilder()
                .setDownloadInfo(toProto(dto.getResource()));
        dto.getMeta().ifPresent(meta -> builder.setSandboxInfo(toProto(meta)));
        return builder.build();
    }

    public TvmApp fromProto(TTvmApp proto) {
        return new TvmApp(proto.getAppId(), proto.getAlias());
    }

    public TTvmApp toProto(TvmApp dto) {
        return TTvmApp.newBuilder()
                .setAppId(dto.getAppId())
                .setAlias(dto.getAlias())
                .build();
    }

    public TvmClient fromProto(TTvmClient proto) {
        return new TvmClient(
                fromProto(proto.getSecretSelector()),
                fromProto(proto.getSource()),
                convertList(proto.getDestinationsList(), this::fromProto),
                proto.getRolesForIdmSlug());
    }

    public TTvmClient toProto(TvmClient dto) {
        return TTvmClient.newBuilder()
                .setSecretSelector(toProto(dto.getSecretSelector()))
                .setSource(toProto(dto.getSource()))
                .setRolesForIdmSlug(dto.getRolesForIdmSlug())
                .addAllDestinations(convertList(dto.getDestinations(), this::toProto))
                .build();
    }

    public TvmConfig fromProto(TTvmConfig proto) {
        Optional<DownloadableResource> tvmtoolLayer = Optional.empty();
        if (proto.hasTvmtoolLayer()) {
            tvmtoolLayer = Optional.of(fromProto(proto.getTvmtoolLayer()));
        }

        Optional<SidecarVolumeSettings> sidecarVolumeSettings = Optional.empty();
        if (proto.hasSidecarVolume()) {
            sidecarVolumeSettings = Optional.of(fromProto(proto.getSidecarVolume()));
        }

        return new TvmConfig(TVM_CONFIG_MODE_CONVERTER.fromProto(proto.getMode()),
                proto.getBlackboxEnvironment(),
                convertList(proto.getClientsList(), this::fromProto),
                OptionalUtils.zeroAsEmptyOptional(proto.getClientPort()),
                OptionalUtils.zeroAsEmptyOptional(proto.getCpuLimit()),
                OptionalUtils.zeroAsEmptyOptional(proto.getMemoryLimitMb()),
                tvmtoolLayer,
                OptionalUtils.zeroAsEmptyOptional(proto.getStageTvmId()),
                OptionalUtils.zeroAsEmptyOptional(proto.getSolomonTvmId()),
                OptionalUtils.zeroAsEmptyOptional(proto.getMonitoringPort()),
                sidecarVolumeSettings,
                proto.getUseSystemCerts());
    }

    public TTvmConfig toProto(TvmConfig dto) {
        TTvmConfig.Builder builder = TTvmConfig.newBuilder()
                .setMode(TVM_CONFIG_MODE_CONVERTER.toProto(dto.getMode()))
                .setBlackboxEnvironment(dto.getBlackboxEnvironment())
                .addAllClients(convertList(dto.getClients(), this::toProto))
                .setClientPort(dto.getClientPort())
                .setSolomonTvmId(dto.getSolomonTvmId())
                .setStageTvmId(dto.getStageTvmId())
                .setCpuLimit(dto.getCpuLimit())
                .setMemoryLimitMb(dto.getMemoryLimitMb())
                .setMonitoringPort(dto.getMonitoringPort())
                .setUseSystemCerts(dto.getUseSystemCerts());
        dto.getTvmtoolLayer().ifPresent(tvmtoolLayer -> builder.setTvmtoolLayer(toProto(tvmtoolLayer)));
        dto.getSidecarVolumeSettings().ifPresent(volumeSettings -> builder.setSidecarVolume(toProto(volumeSettings)));
        return builder.build();
    }

    public SandboxResourceInfo fromProto(TSandboxResourceInfo proto) {
        return new SandboxResourceInfo(
                proto.getRevision(), proto.getOverrideMap());
    }

    public TSandboxResourceInfo toProto(SandboxResourceInfo dto) {
        return TSandboxResourceInfo.newBuilder()
                .setRevision(dto.getRevision())
                .putAllOverride(dto.getOverride())
                .build();
    }

    public StageSpec fromProto(TStageSpec proto) {
        return fromProto(proto, proto.getAccountId());
    }

    public StageSpec fromProto(TStageSpec proto, String accountId) {
        return new StageSpec(
                convertMap(proto.getDeployUnitsMap(), this::fromProto),
                accountId,
                proto.getRevision(),
                proto.getSoxService(),
                convertMap(proto.getDynamicResourcesMap(), this::fromProto),
                proto.getEnvMap()
        );
    }

    public DynamicResourceSpec fromProto(TStageSpec.TStageDynamicResourceSpec proto) {
        return new DynamicResourceSpec(proto.getDeployUnitRef(), proto.getDynamicResource());
    }


    public TStageSpec.TStageDynamicResourceSpec toProto(DynamicResourceSpec dto) {
        return TStageSpec.TStageDynamicResourceSpec.newBuilder()
                .setDeployUnitRef(dto.getDeployUnitRef())
                .setDynamicResource(dto.getDynamicResource())
                .build();
    }

    public DynamicResourceMeta fromProto(Autogen.TDynamicResourceMeta proto) {
        return DynamicResourceMeta.fromProto(proto);
    }

    public Autogen.TDynamicResourceMeta toProto(DynamicResourceMeta dto) {
        return Autogen.TDynamicResourceMeta.newBuilder()
                .setId(dto.getId())
                .setFqid(dto.getFqid())
                .setUuid(dto.getUuid())
                .setCreationTime(dto.getCreationTime())
                .setPodSetId(dto.getPodSetId())
                .addAllAcl(dto.getAcl().getEntries())
                .build();
    }

    public InfraComponents fromProto(TInfraComponents proto) {
        if (proto.getUpdatePolicy() == TInfraComponents.EUpdatePolicy.UNKNOWN) {
            return new InfraComponents(false);
        }
        return new InfraComponents(proto.getAllowAutomaticUpdates());
    }

    public TInfraComponents toProto(InfraComponents dto) {
        return TInfraComponents.newBuilder()
                .setAllowAutomaticUpdates(dto.isAllowAutomaticUpdate())
                .build();
    }

    public TStageSpec toProto(StageSpec dto) {
        return TStageSpec.newBuilder()
                .putAllDeployUnits(convertMap(dto.getDeployUnits(), this::toProto))
                .putAllDynamicResources(convertMap(dto.getDynamicResources(), this::toProto))
                .setAccountId(dto.getAccountId())
                .setRevision(dto.getRevision())
                .setSoxService(dto.isSoxService())
                .putAllEnv(dto.getEnvVars())
                .build();
    }

    public StageStatus fromProto(TStageStatus proto) {
        return new StageStatus(convertMap(proto.getDeployUnitsMap(), this::fromProto),
                convertMap(proto.getDynamicResourcesMap(), this::fromProto),
                proto.getRevision(),
                fromProto(proto.getValidated()),
                proto.getSpecTimestamp());
    }

    public RuntimeDeployControls fromProto(Map<String, TRuntimeDeployControls> protoMap) {
        Map<String, Set<String>> approvedMap =
                protoMap.entrySet().stream()
                        .flatMap(v -> v.getValue().getDeployUnitApprovalsMap().values().stream()
                                .filter(TDeployUnitApproval::hasUser)
                                .filter(TDeployUnitApproval::hasPayload)
                                .filter(tDeployUnitApproval -> tDeployUnitApproval.getUser().getStatus() == TDeployUnitApproval.TUserApproval.EUserApprovalStatus.APPROVED)
                                .map(tDeployUnitApproval -> Pair.of(v.getKey(), tDeployUnitApproval))
                        )
                        .map(v -> Pair.of(v.getKey(),
                                v.getValue().getPayload().getCluster()))
                        .collect(Collectors.groupingBy(Pair::getKey, Collectors.mapping(Pair::getValue,
                                Collectors.toUnmodifiableSet())));

        Map<String, DeployUnitOverrides> deployUnitOverrides =
                protoMap.entrySet().stream()
                        .filter(v -> v.getValue().hasDeployUnitOverrides())
                        .filter(v -> v.getValue().getDeployUnitOverrides().getPerClusterOverridesMap().values().stream()
                                .allMatch(TDeployUnitOverrides.TPerClusterOverrides::hasMaxUnavailable))
                        .collect(Collectors.toMap(
                                Map.Entry::getKey,
                                v -> new DeployUnitOverrides(
                                        v.getValue().getDeployUnitOverrides().getPerClusterOverridesMap().entrySet().stream()
                                                .collect(Collectors.toMap(
                                                        Map.Entry::getKey,
                                                        perClusterEntry -> new PerClusterOverrides(
                                                                perClusterEntry.getValue().getMaxUnavailable().getValue()))),
                                        v.getValue().getDeployUnitOverrides().getRevisionToOverride())));

        Map<String, DeployUnitSpec.DeploySettings> deploySettings =
                protoMap.entrySet().stream()
                        .filter(v -> v.getValue().hasDeployUnitOverrides())
                        .filter(v -> v.getValue().getDeployUnitOverrides().hasDeploySettingsOverride())
                        .filter(v -> v.getValue().getDeployUnitOverrides().getDeploySettingsOverride().hasDeploySettings())
                        .collect(Collectors.toMap(
                                Map.Entry::getKey,
                                v -> fromProto(v.getValue().getDeployUnitOverrides().getDeploySettingsOverride().getDeploySettings())
                        ));

        return new RuntimeDeployControls(
                Collections.unmodifiableMap(approvedMap),
                Collections.unmodifiableMap(deployUnitOverrides),
                Collections.unmodifiableMap(deploySettings));
    }

    public TStageStatus toProto(StageStatus dto) {
        return TStageStatus.newBuilder()
                .putAllDeployUnits(convertMap(dto.getDeployUnits(), this::toProto))
                .putAllDynamicResources(convertMap(dto.getDynamicResources(), this::toProto))
                .setRevision(dto.getRevision())
                .setValidated(toProto(dto.getValidated()))
                .setSpecTimestamp(dto.getSpecTimestamp())
                .build();
    }

    public DynamicResourceStatus fromProto(TStageStatus.TStageDynamicResourceStatus proto) {
        return new DynamicResourceStatus(fromProto(proto.getCurrentTarget()), fromProto(proto.getStatus()));
    }

    public TStageStatus.TStageDynamicResourceStatus toProto(DynamicResourceStatus dto) {
        return TStageStatus.TStageDynamicResourceStatus.newBuilder()
                .setCurrentTarget(toProto(dto.getCurrentTarget()))
                .setStatus(toProto(dto.getStatus()))
                .build();
    }

    public AggregatedCondition fromProto(TAggregatedCondition proto) {
        return new AggregatedCondition(proto.getPodCount(), Optional.of(fromProto(proto.getCondition())));
    }

    public TAggregatedCondition toProto(AggregatedCondition dto) {
        TAggregatedCondition.Builder builder = TAggregatedCondition.newBuilder();
        builder.setPodCount(dto.getPodCount());
        dto.getCondition().ifPresent(cnd -> builder.setCondition(toProto(cnd)));
        return builder.build();
    }

    public DynamicResourceRevisionStatus fromProto(DynamicResource.TDynamicResourceStatus.TRevisionStatus proto) {
        return new DynamicResourceRevisionStatus(proto.getRevision(),
                fromProto(proto.getReady()),
                fromProto(proto.getInProgress()),
                fromProto(proto.getError())
        );
    }

    public DynamicResource.TDynamicResourceStatus.TRevisionStatus toProto(DynamicResourceRevisionStatus dto) {
        return DynamicResource.TDynamicResourceStatus.TRevisionStatus.newBuilder()
                .setRevision(dto.getRevision())
                .setReady(toProto(dto.getReady()))
                .setInProgress(toProto(dto.getInProgress()))
                .setError(toProto(dto.getError()))
                .build();
    }

    public Instant fromProto(Timestamp proto) {
        return Instant.ofEpochSecond(proto.getSeconds(), proto.getNanos());
    }

    public Timestamp toProto(Instant dto) {
        return Timestamp.newBuilder()
                .setSeconds(dto.getEpochSecond())
                .setNanos(dto.getNano())
                .build();
    }

    public McrsUnitSpec fromProto(TDeployUnitSpec.TMultiClusterReplicaSetDeploy proto) {
        return new McrsUnitSpec(proto.getReplicaSet(), this::fromProto);
    }

    public TDeployUnitSpec.TMultiClusterReplicaSetDeploy toProto(McrsUnitSpec dto) {
        return TDeployUnitSpec.TMultiClusterReplicaSetDeploy.newBuilder()
                .setReplicaSet(dto.getSpec())
                .build();
    }

    public McrsUnitStatus fromProto(TDeployUnitStatus.TMultiClusterReplicaSetDeploy proto) {
        return new McrsUnitStatus(proto.getReplicaSetId(), convertMap(proto.getClusterStatusesMap(),
                perClusterProtoStatus ->
                        new McrsUnitStatus.PerClusterStatus(perClusterProtoStatus.getEndpointSetIdsList())));
    }

    public TDeployUnitStatus.TMultiClusterReplicaSetDeploy toProto(McrsUnitStatus dto) {
        return TDeployUnitStatus.TMultiClusterReplicaSetDeploy.newBuilder()
                .setReplicaSetId(dto.getReplicaSetRef())
                .putAllClusterStatuses(convertMap(dto.getClusterStatuses(), perClusterStatus -> {
                            var builder = TDeployUnitStatus.TMultiClusterReplicaSetDeploy.TPerClusterStatus.newBuilder()
                                    .addAllEndpointSetIds(perClusterStatus.getEndpointSetRefs());

                            perClusterStatus.getEndpointSetRefs().stream().findFirst().ifPresent(builder::setEndpointSetId);

                            return builder.build();
                        }
                ))
                .build();
    }

    public ReplicaSetUnitSpec fromProto(TDeployUnitSpec.TReplicaSetDeploy proto) {
        return new ReplicaSetUnitSpec(proto.getReplicaSetTemplate(), convertMap(proto.getPerClusterSettingsMap(),
                this::fromProto), this::fromProto);
    }

    public TDeployUnitSpec.TReplicaSetDeploy toProto(ReplicaSetUnitSpec dto) {
        return TDeployUnitSpec.TReplicaSetDeploy.newBuilder()
                .setReplicaSetTemplate(dto.getSpecTemplate())
                .putAllPerClusterSettings(convertMap(dto.getPerClusterSettings(), this::toProto))
                .build();
    }

    public ReplicaSetUnitStatus fromProto(TDeployUnitStatus.TReplicaSetDeploy proto) {
        return new ReplicaSetUnitStatus(convertMap(proto.getClusterStatusesMap(), perClusterProtoStatus -> {
                    Optional<TReplicaSetStatus> status = Optional.empty();
                    if (getProjectPromiseConfigFlag(USE_EXTENDED_REPLICA_SET_STATUSES_CONFIG_KEY) &&
                            perClusterProtoStatus.hasStatus()) {
                        status = Optional.of(perClusterProtoStatus.getStatus());
                    }
                    return new ReplicaSetUnitStatus.PerClusterStatus(perClusterProtoStatus.getReplicaSetId(),
                            perClusterProtoStatus.getEndpointSetIdsList(),
                            perClusterProtoStatus.getHorizontalPodAutoscalerId(),
                            status);
                }

        ));
    }

    public TDeployUnitStatus.TReplicaSetDeploy toProto(ReplicaSetUnitStatus dto) {
        return TDeployUnitStatus.TReplicaSetDeploy.newBuilder()
                .putAllClusterStatuses(convertMap(dto.getClusterStatuses(), perClusterStatus -> {
                            var builder = TDeployUnitStatus.TReplicaSetDeploy.TPerClusterStatus.newBuilder()
                                    .setReplicaSetId(perClusterStatus.getReplicaSetRef())
                                    .addAllEndpointSetIds(perClusterStatus.getEndpointSetRefs())
                                    .setHorizontalPodAutoscalerId(perClusterStatus.getHorizontalPodAutoscalerRef());

                            if (getProjectPromiseConfigFlag(USE_EXTENDED_REPLICA_SET_STATUSES_CONFIG_KEY)) {
                                perClusterStatus.getRawStatus().ifPresent(builder::setStatus);
                            }

                            perClusterStatus.getEndpointSetRefs().stream().findFirst().ifPresent(builder::setEndpointSetId);

                            return builder.build();
                        }
                ))
                .build();
    }

    public ReplicaSetUnitSpec.PerClusterSettings fromProto(TPerClusterSettings proto) {
        Optional<ReplicaSetDeploymentStrategy> strategy = Optional.empty();
        if (proto.hasDeploymentStrategy()) {
            strategy = Optional.of(fromProto(proto.getDeploymentStrategy()));
        }
        Either<Integer, TReplicaSetScaleSpec> podCountPolicy = proto.hasAutoscale()
                ? Either.right(proto.getAutoscale())
                : Either.left(proto.getPodCount());

        return new ReplicaSetUnitSpec.PerClusterSettings(podCountPolicy, strategy);
    }

    public TPerClusterSettings toProto(ReplicaSetUnitSpec.PerClusterSettings dto) {
        TPerClusterSettings.Builder builder = TPerClusterSettings.newBuilder();
        if (dto.hasAutoscale()) {
            builder.setAutoscale(dto.getAutoscale().orElseThrow());
        } else {
            builder.setPodCount(dto.getPodCount().orElseThrow());
        }
        dto.getDeploymentStrategy().ifPresent(strategy -> builder.setDeploymentStrategy(toProto(strategy)));
        return builder.build();
    }

    public static SandboxResourceMeta fromProto(TSandboxResource proto) {
        return new SandboxResourceMeta(Long.parseLong(proto.getTaskId()), Long.parseLong(proto.getResourceId()),
                proto.getAttributesMap());
    }

    public static TSandboxResource toProto(SandboxResourceMeta dto) {
        return TSandboxResource.newBuilder()
                .setResourceId(String.valueOf(dto.getResourceId()))
                .setTaskId(String.valueOf(dto.getTaskId()))
                .putAllAttributes(dto.getAttributes())
                .build();
    }

    public SchemaMeta fromProto(Autogen.TSchemaMeta proto) {
        return SchemaMeta.fromProto(proto);
    }

    public Autogen.TSchemaMeta toProto(SchemaMeta dto) {
        return Autogen.TSchemaMeta.newBuilder()
                .setId(dto.getId())
                .addAllAcl(dto.getAcl().getEntries())
                .setFqid(dto.getFqid())
                .setUuid(dto.getUuid())
                .setCreationTime(dto.getCreationTime())
                .build();
    }

    public ProjectMeta fromProto(Autogen.TProjectMeta proto) {
        return ProjectMeta.fromProto(proto);
    }

    public Autogen.TProjectMeta toProto(ProjectMeta dto) {
        return Autogen.TProjectMeta.newBuilder()
                .setId(dto.getId())
                .addAllAcl(dto.getAcl().getEntries())
                .setFqid(dto.getFqid())
                .setUuid(dto.getUuid())
                .setCreationTime(dto.getCreationTime())
                .setOwnerId(dto.getOwnerId())
                .build();
    }

    public RelationMeta fromProto(Autogen.TRelationMeta proto) {
        return RelationMeta.fromProto(proto);
    }

    public Autogen.TRelationMeta toProto(RelationMeta dto) {
        return Autogen.TRelationMeta.newBuilder()
                .setId(dto.getId())
                .addAllAcl(dto.getAcl().getEntries())
                .setFqid(dto.getFqid())
                .setUuid(dto.getUuid())
                .setCreationTime(dto.getCreationTime())
                .setFromFqid(dto.getFromFqid())
                .setToFqid(dto.getToFqid())
                .build();
    }

    public StageMeta fromProto(Autogen.TStageMeta proto) {
        return StageMeta.fromProto(proto);
    }

    public Autogen.TStageMeta toProto(StageMeta dto) {
        return Autogen.TStageMeta.newBuilder()
                .setId(dto.getId())
                .addAllAcl(dto.getAcl().getEntries())
                .setFqid(dto.getFqid())
                .setUuid(dto.getUuid())
                .setCreationTime(dto.getCreationTime())
                .setProjectId(dto.getProjectId())
                .setAccountId(dto.getAccountId())
                .build();
    }

    public HorizontalPodAutoscalerMeta fromProto(THorizontalPodAutoscalerMeta proto) {
        return HorizontalPodAutoscalerMeta.fromProto(proto);
    }

    public THorizontalPodAutoscalerMeta toProto(HorizontalPodAutoscalerMeta dto) {
        return THorizontalPodAutoscalerMeta.newBuilder()
                .setId(dto.getId())
                .setFqid(dto.getFqid())
                .setUuid(dto.getUuid())
                .setCreationTime(dto.getCreationTime())
                .setReplicaSetId(dto.getReplicaSetId())
                .addAllAcl(dto.getAcl().getEntries())
                .build();
    }

    public DeployReadyCriterion fromProto(TDeployReadyCriterion proto) {
        return new DeployReadyCriterion(
                proto.getReadyThreshold().getType().isEmpty() ? Optional.empty() : Optional.of(proto.getReadyThreshold().getType()),
                proto.getReadyThreshold().hasMinReadyPodsPercent() ? Optional.of(proto.getReadyThreshold().getMinReadyPodsPercent().getValue()) : Optional.empty(),
                proto.getReadyThreshold().hasMaxNotReadyPods() ? Optional.of(proto.getReadyThreshold().getMaxNotReadyPods().getValue()) : Optional.empty());
    }

    public ReplicaSetDeploymentStrategy fromProto(TReplicaSetSpec.TDeploymentStrategy proto) {
        return new ReplicaSetDeploymentStrategy(proto.getMinAvailable(),
                                                proto.getMaxUnavailable(),
                                                proto.getMaxSurge(),
                                                proto.getMinCreatedSeconds(),
                                                proto.getMaxTolerableDowntimeSeconds(),
                                                proto.getMaxTolerableDowntimePods(),
                                                proto.hasDeploySpeed() ? Optional.of(fromProto(proto.getDeploySpeed())) : Optional.empty(),
                                                proto.hasReadyCriterion() ? Optional.of(fromProto(proto.getReadyCriterion())) : Optional.empty());
    }

    public TDeployReadyCriterion toProto(DeployReadyCriterion dto) {
        TDeployReadyCriterion.Builder deployReadyCriterionBuilder = TDeployReadyCriterion.newBuilder();
        TDeployReadyCriterion.TReadyThreshold.Builder readyThresholdBuilder = TDeployReadyCriterion.TReadyThreshold.newBuilder();

        dto.getType().ifPresent(readyThresholdBuilder::setType);
        dto.getMinReadyPodsPercent().ifPresent(mrpp -> readyThresholdBuilder.setMinReadyPodsPercent(TDeployReadyCriterion.TReadyThreshold.TMinReadyPodsPercent.newBuilder().setValue(mrpp)));
        dto.getMaxNotReadyPods().ifPresent(mnrp -> readyThresholdBuilder.setMaxNotReadyPods(TDeployReadyCriterion.TReadyThreshold.TMaxNotReadyPods.newBuilder().setValue(mnrp)));

        deployReadyCriterionBuilder.setReadyThreshold(readyThresholdBuilder);
        return deployReadyCriterionBuilder.build();
    }

    public TReplicaSetSpec.TDeploymentStrategy toProto(ReplicaSetDeploymentStrategy dto) {
        TReplicaSetSpec.TDeploymentStrategy.Builder builder = TReplicaSetSpec.TDeploymentStrategy.newBuilder()
                .setMinAvailable(dto.getMinUnavailable())
                .setMaxUnavailable(dto.getMaxUnavailable())
                .setMaxSurge(dto.getMaxSurge())
                .setMinCreatedSeconds(dto.getMinCreatedSeconds())
                .setMaxTolerableDowntimeSeconds(dto.getMaxTolerableDowntimeSeconds())
                .setMaxTolerableDowntimePods(dto.getMaxTolerableDowntimePods());
        dto.getDeployReadyCriterion().ifPresent(drc -> builder.setReadyCriterion(toProto(drc)));
        dto.getDeploySpeed().ifPresent(ds -> builder.setDeploySpeed(toProto(ds)));
        return builder.build();
    }

    public TMultiClusterReplicaSetSpec.TDeploymentStrategy toProto(DeploymentStrategy dto) {
        TMultiClusterReplicaSetSpec.TDeploymentStrategy.Builder builder = TMultiClusterReplicaSetSpec.TDeploymentStrategy.newBuilder()
                .setMaxUnavailable(dto.getMaxUnavailable())
                .setMinCreatedSeconds(dto.getMinCreatedSeconds())
                .setMaxTolerableDowntimeSeconds(dto.getMaxTolerableDowntimeSeconds())
                .setMaxTolerableDowntimePods(dto.getMaxTolerableDowntimePods());
        dto.getDeploySpeed().ifPresent(ds -> builder.setDeploySpeed(toProto(ds)));
        return builder.build();
    }

    public DeploymentStrategy fromProto(TMultiClusterReplicaSetSpec.TDeploymentStrategy proto) {
        return new DeploymentStrategy(proto.getMaxUnavailable(),
                                      proto.getMinCreatedSeconds(),
                                      proto.getMaxTolerableDowntimeSeconds(),
                                      proto.getMaxTolerableDowntimePods(),
                                      proto.hasDeploySpeed() ? Optional.of(fromProto(proto.getDeploySpeed())) : Optional.empty());
    }

    public DeploySpeed fromProto(TDeploySpeed proto) {
        return new DeploySpeed(proto.getUpdatePortion(), proto.getMinDelay());
    }

    public TDeploySpeed toProto(DeploySpeed dto) {
        return TDeploySpeed.newBuilder()
                .setUpdatePortion(dto.getUpdatePortion())
                .setMinDelay(dto.getMinDelaySeconds())
                .build();
    }

}
