package ru.yandex.infra.stage.rest;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.infra.controller.yp.YpRequestWithPaging;
import ru.yandex.infra.stage.yp.AsyncYpClientsMap;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpPayloadFormat;
import ru.yandex.yp.model.YpSelectStatement;

import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class RelationsServlet extends HttpServlet {
    private static final int YP_REQUEST_PAGE_SIZE = 1 << 16;
    private static final Logger LOG = LoggerFactory.getLogger(RelationsServlet.class);

    //Todo: use application config entries instead of predefined constants used in production
    private static final String FILTER_BY_LABELS_FOR_PRODUCTION_STAGES =
            "[/labels/deploy_engine] = \"env_controller\"";
    private static final String FILTER_BY_LABELS_FOR_PRODUCTION_STAGE_CHILD_OBJECTS =
            "[/labels/created_by] = \"stage_controller\" AND [/labels/stage_tag] = \"prod\"";
    private static final String FILTER_BY_LABELS_FOR_PRODUCTION_POD_SETS =
            "[/labels/deploy_engine] = \"MCRSC\" OR [/labels/deploy_engine] = \"RSC\"";

    private final AsyncYpClientsMap ypClients;

    public RelationsServlet(AsyncYpClientsMap ypClients) {
        this.ypClients = ypClients;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        final long start = System.currentTimeMillis();
        try {
            YpRawObjectService ypXdcClient = ypClients.getMultiClusterClient();

            Map<String, Set<String>> parentRelations = YpRequestWithPaging.selectObjects(ypXdcClient, YP_REQUEST_PAGE_SIZE,
                            YpSelectStatement.builder(YpObjectType.RELATION, YpPayloadFormat.YSON)
                                    .addSelector("/meta/from_fqid")
                                    .addSelector("/meta/to_fqid"),
                            payloads -> {
                                String fromFqid = payloadToYson(payloads.get(0)).stringValue();
                                String toFqid = payloadToYson(payloads.get(1)).stringValue();
                                return Tuple2.tuple(fromFqid, toFqid);
                            })
                    .thenApply(objects -> objects.stream()
                            .collect(Collectors.groupingBy(t -> t._2)))
                    .thenApply(map -> Maps.transformValues(map, list -> list.stream().map(t -> t._1).collect(Collectors.toSet())))
                    .get();
            LOG.info("Loaded {} unique objects used as 'to_fqid' in relations", parentRelations.size());

            Set<String> stages = loadFqids(ypXdcClient, YpObjectType.STAGE, FILTER_BY_LABELS_FOR_PRODUCTION_STAGES);
            LOG.info("Loaded {} stage fqids", stages.size());
            Set<String> replicaSets = loadFqids(ypXdcClient, YpObjectType.MULTI_CLUSTER_REPLICA_SET,
                    FILTER_BY_LABELS_FOR_PRODUCTION_STAGE_CHILD_OBJECTS);
            LOG.info("Loaded {} mcrs fqids", replicaSets.size());

            Set<String> podSets = new HashSet<>();
            ypClients.getClusters().forEach(cluster -> {
                YpRawObjectService ypClient = ypClients.getClusterClient(cluster);
                try {
                    Set<String> rs = loadFqids(ypClient, YpObjectType.REPLICA_SET,
                            FILTER_BY_LABELS_FOR_PRODUCTION_STAGE_CHILD_OBJECTS);
                    LOG.info("[{}] Loaded {} replicaSet fqids", cluster, rs.size());
                    replicaSets.addAll(rs);
                    Set<String> ps = loadFqids(ypClient, YpObjectType.POD_SET,
                            FILTER_BY_LABELS_FOR_PRODUCTION_POD_SETS);
                    LOG.info("[{}] Loaded {} podSets fqids", cluster, ps.size());
                    podSets.addAll(ps);
                } catch (ExecutionException | InterruptedException e) {
                    LOG.error("{} failed to load objects from YP", req.getMethod(), e);
                }
            });
            LOG.info("Found {} replicaSets/mcrs on all cluster", replicaSets.size());
            LOG.info("Found {} podSets on all cluster", podSets.size());

            List<String> replicaSetsWithoutRelation = replicaSets.stream()
                    .filter(fqid -> isRelationMissed(fqid, stages, parentRelations))
                    .sorted(Comparator.naturalOrder())
                    .collect(Collectors.toList());
            List<String> podSetsWithoutRelation = podSets.stream()
                    .filter(fqid -> isRelationMissed(fqid, replicaSets, parentRelations))
                    .sorted(Comparator.naturalOrder())
                    .collect(Collectors.toList());
            LOG.info("Found {} replicaSets/mcrs without parent relation", replicaSetsWithoutRelation.size());
            LOG.info("Found {} podSets without parent relation", podSetsWithoutRelation.size());

            PrintWriter writer = resp.getWriter();
            replicaSetsWithoutRelation.forEach(writer::println);
            podSetsWithoutRelation.forEach(writer::println);

            resp.setStatus(HttpServletResponse.SC_OK);
        } catch (Exception ex) {
            LOG.error("{} failed", req.getMethod(), ex);
            resp.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
        LOG.info("Processing {} {} request took: {} ms", req.getMethod(), req.getServletPath(), System.currentTimeMillis() - start);
    }

    private boolean isRelationMissed(String childFqid, Set<String> parentObjectsFqids, Map<String, Set<String>> parentRelations) {
        Set<String> parentsFoundByRelations = parentRelations.get(childFqid);
        if (parentsFoundByRelations == null) {
            return true;
        }
        return parentsFoundByRelations.stream().noneMatch(parentObjectsFqids::contains);
    }

    private static Set<String> loadFqids(YpRawObjectService ypClient, YpObjectType objectType, String filter) throws ExecutionException, InterruptedException {
        return YpRequestWithPaging.selectObjects(ypClient, YP_REQUEST_PAGE_SIZE,
                        YpSelectStatement.builder(objectType, YpPayloadFormat.YSON)
                                .addSelector("/meta/fqid")
                                .setFilter(filter),
                        payloads -> payloadToYson(payloads.get(0)).stringValue())
                .thenApply(HashSet::new)
                .get();
    }

}
