package ru.yandex.infra.stage.util;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;

import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Yet another rate limiter.
 *
 * Current rate is not fixed and depends on failed requests count during previous time interval.
 */
public class AdaptiveRateLimiterImpl implements AdaptiveRateLimiter {
    private static final Logger LOG = LoggerFactory.getLogger(AdaptiveRateLimiter.class);

    private final Clock clock;
    private final int minRPS, maxRPS, defaultRPS;
    private final double rateUpMultiplier, rateDownMultiplier;
    private final Duration reviewInterval;
    private final Optional<Duration> autoShutdownInterval;
    private final Optional<Integer> maxConcurrentRequests;
    private final String name;

    private double currentRate = Double.NaN;
    private int executionsInSecond;
    private volatile long lastExecutionSecond = -1L;
    private final AtomicInteger activeRequests = new AtomicInteger();

    private Instant lastFailedResponseTimestamp;
    private Instant lastRateReviewTimestamp;
    private Instant limiterActivationTimestamp;

    public AdaptiveRateLimiterImpl(Clock clock, Config config) {
        this.clock = clock;

        this.minRPS = config.getInt("min_rps");
        this.maxRPS = config.getInt("max_rps");
        this.defaultRPS = config.getInt("default_rps");
        this.reviewInterval = config.getDuration("review_interval");
        if (config.hasPath("auto_shutdown_interval")) {
            this.autoShutdownInterval = Optional.of(config.getDuration("auto_shutdown_interval"));
        } else {
            this.autoShutdownInterval = Optional.empty();
        }
        if (config.hasPath("max_concurrent_requests")) {
            int limit = config.getInt("max_concurrent_requests");
            this.maxConcurrentRequests = limit >= 0 ? Optional.of(limit) : Optional.empty();
        } else {
            this.maxConcurrentRequests = Optional.empty();
        }

        this.rateUpMultiplier = config.getDouble("rate_up_multiplier");
        this.rateDownMultiplier = config.getDouble("rate_down_multiplier");
        if (config.hasPath("name")) {
            this.name = config.getString("name");
        } else {
            this.name = String.format("defaultRPS=%d, minRPS=%d, maxRPS=%d", defaultRPS, minRPS, maxRPS);
        }

        if(config.getBoolean("activate_after_start")) {
            setActive(true);
        }
    }

    @Override
    public boolean tryAcquire() {
        reviewLimits();

        if(!isActive()) {
            return true;
        }

        if (maxConcurrentRequests.map(limit -> activeRequests.get() >= limit)
                .orElse(false)) {
            return false;
        }

        long nowSeconds = clock.instant().getEpochSecond();

        if(nowSeconds == lastExecutionSecond) {
            executionsInSecond++;
        } else {
            lastExecutionSecond = nowSeconds;
            executionsInSecond = 1;
        }

        return executionsInSecond <= currentRate;
    }

    @Override
    public int incrementAndGet() {
        return activeRequests.incrementAndGet();
    }

    @Override
    public int decrementAndGet() {
        return activeRequests.decrementAndGet();
    }

    @Override
    public void registerFailedResponse() {
        lastFailedResponseTimestamp = clock.instant();
        if(!isActive()) {
            setActive(true);
        }
    }

    @Override
    public double getRate() {
        return currentRate;
    }

    @Override
    public int getActiveRequestsCount() {
        return activeRequests.get();
    }

    @Override
    public boolean isActive() {
        return !Double.isNaN(currentRate);
    }

    private void setActive(boolean value) {
        if(value) {
            LOG.info("[{}] RateLimiter is enabled. Current rate = {} rps", name, defaultRPS);
            setRate(defaultRPS);
            limiterActivationTimestamp = clock.instant();
            lastRateReviewTimestamp = limiterActivationTimestamp;//should wait some time until next rate review
        }
        else {
            currentRate = Double.NaN;
            LOG.info("[{}] RateLimiter was disabled. Last failed request time is {}", name, lastFailedResponseTimestamp);
        }
    }

    private void reviewLimits() {
        if(!isActive()) return;

        var now = clock.instant();

        var latestFailOrLimiterStart = lastFailedResponseTimestamp != null && lastFailedResponseTimestamp.isAfter(limiterActivationTimestamp)
                ? lastFailedResponseTimestamp : limiterActivationTimestamp;
        // Turn off limiter if no failures during last autoShutdownInterval
        if (autoShutdownInterval.isPresent()) {
            if (now.isAfter(latestFailOrLimiterStart.plus(autoShutdownInterval.get()))) {
                setActive(false);
                return;
            }
        }

        if(Duration.between(lastRateReviewTimestamp, now).compareTo(reviewInterval) < 0) {
            return;
        }

        boolean noNewFailedRequestsSincePreviousReview = lastFailedResponseTimestamp == null ||
                Duration.between(lastFailedResponseTimestamp, now).compareTo(reviewInterval) > 0;

        var multiplier = noNewFailedRequestsSincePreviousReview ? rateUpMultiplier : rateDownMultiplier;
        setRate(currentRate * multiplier);

        lastRateReviewTimestamp = now;
    }

    private void setRate(double permitsPerSecond) {
        currentRate = fitIntoRange(permitsPerSecond);
        LOG.info("[{}] Rate was updated to {} rps", name, currentRate);
    }

    private double fitIntoRange(double permits) {
        if (permits < minRPS) {
            return minRPS;
        }
        if (permits > maxRPS) {
            return maxRPS;
        }
        return permits;
    }
}
