package ru.yandex.infra.stage.util;

import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.stage.HttpServiceMetrics;

public class JsonHttpGetter {
    private static final int VALID_RESPONSE_CODE = 200;
    private static final Logger LOG = LoggerFactory.getLogger(JsonHttpGetter.class);

    private final HttpServiceMetrics.Source source;
    private final AsyncHttpClient httpClient;
    private final HttpServiceMetrics metrics;

    public JsonHttpGetter(HttpServiceMetrics.Source source, AsyncHttpClient httpClient, HttpServiceMetrics metrics) {
        this.source = source;
        this.httpClient = httpClient;
        this.metrics = metrics;
    }

    public CompletableFuture<JsonNode> get(String url) {
        return get(url, Optional.empty());
    }

    public CompletableFuture<JsonNode> get(String url, String token) {
        return get(url, Optional.of(token));
    }

    private CompletableFuture<JsonNode> get(String url, Optional<String> token) {
        LOG.info("Sending request: {}", url);
        metrics.addRequest(source);

        var requestBuilder = httpClient.prepareGet(url);
        token.ifPresent(t -> requestBuilder.addHeader("Authorization", String.format("OAuth %s", t)));

        return httpClient.executeRequest(requestBuilder.build()).toCompletableFuture()
                .thenApply(response -> {
                    if (response.getStatusCode() != VALID_RESPONSE_CODE) {
                        metrics.addErrorResponse(source);
                        LOG.warn("{} failed to get {} - response code {} {}",
                                source,
                                response.getUri(),
                                response.getStatusCode(),
                                response.getStatusText());
                        throw new JsonHttpGetterException(
                                String.format("%d %s",
                                        response.getStatusCode(),
                                        response.getStatusText()));
                    }
                    try {
                        return (new ObjectMapper()).readTree(response.getResponseBodyAsBytes());
                    } catch (IOException e) {
                        metrics.addParsingError(source);
                        LOG.error("{} failed to parse JSON response of {}: {}",
                                source,
                                response.getUri(),
                                e.getMessage());
                        throw new JsonHttpGetterException("Failed to parse JSON response", e);
                    }
                });
    }

    private static class JsonHttpGetterException extends RuntimeException {
        public JsonHttpGetterException(String msg) {
            super(msg);
        }

        public JsonHttpGetterException(String msg, Exception cause) {
            super(msg, cause);
        }
    }
}
