package ru.yandex.infra.stage.util;

import java.util.Map;

import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import org.slf4j.Logger;

// Applies specs to managed map using provided managing functions
public class MapManager {
    private MapManager() {
    }

    public interface CreateFunction<Item, Spec> {
        Item create(String id, Spec spec);
    }

    public interface UpdateFunction<Item, Spec> {
        void update(String id, Item item, Spec spec);
    }

    public interface ShutdownFunction<Item> {
        void shutdown(String id, Item item);
    }

    public static <Item, Spec> void apply(Map<String, Item> managed, Map<String, Spec> newSpecs,
                                                   CreateFunction<Item, Spec> onCreate, UpdateFunction<Item, Spec> onUpdate,
                                                   ShutdownFunction<Item> onShutdown, Logger logger, String description) {
        MapDifference<String, Object> difference = Maps.difference(managed, newSpecs);

        difference.entriesOnlyOnLeft().forEach((itemId, item) -> withErrorLogging(itemId, "remove", () -> {
            onShutdown.shutdown(itemId, (Item)item);
            managed.remove(itemId);
        }, logger, description));

        difference.entriesOnlyOnRight().forEach((itemId, spec) -> withErrorLogging(itemId, "create",
                () -> managed.put(itemId, onCreate.create(itemId, (Spec)spec)), logger, description));

        difference.entriesDiffering().forEach((itemId, valueDifference) -> withErrorLogging(itemId, "update",
                () -> onUpdate.update(itemId, (Item)valueDifference.leftValue(), (Spec)valueDifference.rightValue()),
                logger, description));

        if (!difference.entriesInCommon().isEmpty()) {
            logger.warn("Unexpected {} common entries in MapManager", difference.entriesInCommon().size());
        }
    }

    private static void withErrorLogging(String itemId, String actionString, Runnable action, Logger logger, String description) {
        try {
            action.run();
        } catch (Exception e) {
            logger.error("Could not {} {}", actionString, String.format(description, itemId), e);
        }
    }
}
