package ru.yandex.infra.stage.yp;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import com.typesafe.config.Config;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.util.AclUtils;
import ru.yandex.infra.stage.ConfigUtils;
import ru.yandex.infra.stage.util.StringUtils;
import ru.yandex.yp.client.api.AccessControl;

public class AppendingAclUpdater implements AclUpdater {
    private final AccessControl.TAccessControlEntry addedEntry;

    public AppendingAclUpdater(AccessControl.TAccessControlEntry entry) {
        this.addedEntry = entry;
    }

    public AppendingAclUpdater(Collection<String> subjects, Collection<AccessControl.EAccessControlPermission> permissions) {
        this(AccessControl.TAccessControlEntry.newBuilder()
                .addAllSubjects(subjects)
                .addAllPermissions(permissions)
                .setAction(AccessControl.EAccessControlAction.ACA_ALLOW)
                .build());
    }

    public AccessControl.TAccessControlEntry getEntry() {
        return addedEntry;
    }

    @Override
    public Acl update(Acl value) {
        List<AccessControl.TAccessControlEntry> entries = new ArrayList<>(value.getEntries());
        entries.add(addedEntry);
        return new Acl(entries);
    }

    public static AclUpdater configure(Config config) {
        Set<AccessControl.EAccessControlPermission> permissions;
        if (config.hasPath("permissions")) {
            permissions = ConfigUtils.unwrapStringList(config.getList("permissions")).stream()
                    .map(value -> AclUtils.permissionToProto(value).orElseThrow(() -> {
                            String message = String.format("Unknown yp permission '%s', allowed are %s", value,
                                StringUtils.joinStrings(AclUtils.PERMISSION_NAMES.keySet()));
                        throw new IllegalArgumentException(message);
                    }))
                    .collect(Collectors.toSet());
        } else {
            permissions = ImmutableSet.of(AccessControl.EAccessControlPermission.ACP_READ, AccessControl.EAccessControlPermission.ACA_WRITE);
        }

        List<String> subjects = ConfigUtils.unwrapStringList(config.getList("subjects"));
        if (subjects.isEmpty()) {
            return AclUpdater.IDENTITY;
        } else {
            return new AppendingAclUpdater(subjects, permissions);
        }
    }
}
