package ru.yandex.infra.stage.yp;

import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.Message;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.yp.EpochDecoratorRepository;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.ObjectBuilderDescriptor;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.dto.DynamicResourceMeta;
import ru.yandex.infra.stage.dto.HorizontalPodAutoscalerMeta;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.Autogen;
import ru.yandex.yp.client.api.Autogen.TSchemaMeta;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.DynamicResource;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerSpec;
import ru.yandex.yp.client.api.THorizontalPodAutoscalerStatus;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetSpec;
import ru.yandex.yp.client.api.TMultiClusterReplicaSetStatus;
import ru.yandex.yp.client.api.TReplicaSetSpec;
import ru.yandex.yp.client.api.TReplicaSetStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;

public class EpochDecoratorRepositoryFactory implements RepositoryFactory {
    static final Map<YpObjectType, ObjectBuilderDescriptor> DESCRIPTORS = ImmutableMap.copyOf(Map.of(
            YpObjectType.DYNAMIC_RESOURCE, new ObjectBuilderDescriptor<>(
                    DynamicResource.TDynamicResourceSpec::newBuilder,
                    DynamicResource.TDynamicResourceStatus::newBuilder,
                    DynamicResourceMeta::fromProto,
                    Autogen.TDynamicResourceMeta.getDefaultInstance()),
            YpObjectType.REPLICA_SET, new ObjectBuilderDescriptor<>(TReplicaSetSpec::newBuilder,
                    TReplicaSetStatus::newBuilder, SchemaMeta::fromProto, TSchemaMeta.getDefaultInstance()),
            YpObjectType.ENDPOINT_SET, new ObjectBuilderDescriptor<>(DataModel.TEndpointSetSpec::newBuilder,
                    DataModel.TEndpointSetStatus::newBuilder, SchemaMeta::fromProto, TSchemaMeta.getDefaultInstance()),
            YpObjectType.MULTI_CLUSTER_REPLICA_SET,
            new ObjectBuilderDescriptor<>(TMultiClusterReplicaSetSpec::newBuilder,
                    TMultiClusterReplicaSetStatus::newBuilder, SchemaMeta::fromProto, TSchemaMeta.getDefaultInstance()),
            YpObjectType.HORIZONTAL_POD_AUTOSCALER, new ObjectBuilderDescriptor<>(
                    THorizontalPodAutoscalerSpec::newBuilder,
                    THorizontalPodAutoscalerStatus::newBuilder,
                    HorizontalPodAutoscalerMeta::fromProto,
                    Autogen.THorizontalPodAutoscalerMeta.getDefaultInstance()),
            YpObjectType.STAGE, new ObjectBuilderDescriptor<>(
                    TStageSpec::newBuilder, TStageStatus::newBuilder, StageMeta::fromProto,
                    Autogen.TStageMeta.getDefaultInstance())
    ));

    private final Map<String, String> labelsMap;
    private final int pageSize;
    private final Supplier<Long> epochGetter;
    private final String epochKey;
    private final AsyncYpClientsMap ypClients;
    private final Optional<String> vcsPrefix;
    private final GaugeRegistry gaugeRegistry;

    public EpochDecoratorRepositoryFactory(Map<String, String> labelsMap, int pageSize,
                                           Supplier<Long> epochGetter, String epochKey,
                                           AsyncYpClientsMap ypClients, Optional<String> vcsPrefix,
                                           GaugeRegistry gaugeRegistry) {
        this.labelsMap = labelsMap;
        this.pageSize = pageSize;
        this.epochGetter = epochGetter;
        this.epochKey = epochKey;
        this.vcsPrefix = vcsPrefix;
        this.ypClients = ypClients;
        this.gaugeRegistry = gaugeRegistry;
    }

    @Override
    public <Meta extends SchemaMeta, Spec extends Message, Status extends Message> EpochDecoratorRepository<Meta, Spec, Status> create(
            ClusterAndType clusterAndType
    ) {
        YpRawObjectService ypClient = ypClients.getMultiOrClusterClient(clusterAndType.getCluster());
        final LabelBasedRepository labelBasedRepository = new LabelBasedRepository<>(clusterAndType.getType(),
                labelsMap,
                vcsPrefix,
                ypClient,
                DESCRIPTORS.get(clusterAndType.getType()),
                pageSize,
                gaugeRegistry,
                clusterAndType.getCluster().orElse(LabelBasedRepository.ALL_CLUSTERS));
        return new EpochDecoratorRepository<>(labelBasedRepository, clusterAndType.getType(), epochGetter, epochKey);
    }
}
