package ru.yandex.infra.stage.yp;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

import com.google.protobuf.Message;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.yp.CreateObjectRequest;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.stage.rest.GarbageRequestReceiver;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

// Interface for interaction between controller code and external systems (for now only YP, at least it depends on YP ACL).
// Controller code subscribes to an object. may modify it via create/update.
// Objects without subscribers are considered garbage and eventually deleted.
// Garbage collection is triggered externally when it is decided that repository is up to date.
public interface ObjectLifeCycleManager<Meta extends SchemaMeta, Spec extends Message, Status extends Message> extends GarbageRequestReceiver {
    // Create object in underlying system
    void create(String id, CreateObjectRequest<Spec> request, Runnable onSuccess, Consumer<Throwable> onFailure);
    void create(String id, Function<YpObjectTransactionalRepository<Meta, Spec, Status>, CompletableFuture<?>> createAction, Runnable onSuccess, Consumer<Throwable> onFailure);
    void updateSpecWithLabels(String id, Spec spec, Map<String, YTreeNode> labels, Optional<Long> prerequisiteTimestamp,
            Runnable onSuccess, Consumer<Throwable> onFailure);
    void updateAcl(String id, Acl acl, Optional<Long> metaPrerequisiteTimestamp,
            Runnable onSuccess, Consumer<Throwable> onFailure);
    void removeLabel(String id, String label, Runnable onSuccess, Consumer<Throwable> onFailure);

    // Mark object as managed, subscribe to its updates. This does not create a new object.
    // Absense of value indicates that the object is not present
    // Use 'withInitialGet' to get object ASAP, without waiting for it to be listed
    void startManaging(String id, BiConsumer<Optional<SpecStatusMeta<Meta, Spec, Status>>, ObjectLifeCycleEventType> subscriber);
    // Unsubscribe from updates. Object will be marked as garbage and eventually removed.
    void stopManaging(String id);
    // Get objects from underlying repository and pass updates to subscribers
    // CompletableFuture is returned after all subscribers are processed to simplify testing
    CompletableFuture<?> startPolling();
    CompletableFuture<?> processPollingResults();

    void startGcCycle();

    boolean isInCache(int hash);

    Optional<String> getCluster();

    void addError(int hash, String description);
}
