package ru.yandex.infra.stage.yp;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.util.ExceptionUtils;
import ru.yandex.infra.controller.yp.CreateObjectRequest;
import ru.yandex.infra.controller.yp.Paths;
import ru.yandex.infra.controller.yp.UpdateYpObjectRequest;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.controller.yp.YpObjectSettings;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.controller.yp.YpObjectsCache;
import ru.yandex.infra.stage.concurrent.SerialExecutor;
import ru.yandex.infra.stage.dto.ClusterAndType;
import ru.yandex.infra.stage.rest.MetricUtils;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.model.YpAttributeTimestampPrerequisite;
import ru.yandex.yp.model.YpRemoveUpdate;

import static java.util.Optional.empty;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.SPEC_STATUS_META_LABELS;

public class ObjectLifeCycleManagerImpl<Meta extends SchemaMeta, Spec extends Message, Status extends Message>
        implements ObjectLifeCycleManager<Meta, Spec, Status> {

    private static final Logger LOG = LoggerFactory.getLogger(ObjectLifeCycleManagerImpl.class);
    private static final int TIME_TO_DELETE = 10;

    static final String METRIC_RUNNING_OPERATIONS = "running_operations_count";
    static final String METRIC_MANAGED_OBJECTS = "managed_objects_count";
    static final String METRIC_TOTAL_OBJECTS = "total_objects_count";
    static final String METRIC_GARBAGE_OBJECTS = "garbage_objects_count";
    static final String METRIC_OBJECTS_WITH_FAILED_OPERATIONS_COUNT = "objects_with_failed_operations";
    static final String METRIC_LIST_DELAY_SECONDS = "list_delay_seconds";
    static final String METRIC_GARBAGE_LIMIT_EXCEEDED = "garbage_limit_exceeded";
    static final String METRIC_MODIFY_OPERATION_ERRORS_COUNT = "modify_operation_errors_count";
    static final String METRIC_MODIFY_OPERATION_COUNT = "modify_operation_count";

    //Metrics
    private volatile Instant metricLastSuccessfulObjectsProcessingTimestamp;
    private final AtomicLong metricModifyOperationErrorsCount = new AtomicLong();
    private final AtomicLong metricModifyOperationCount = new AtomicLong();
    private volatile Integer metricTotalObjectsCount;

    private final YpObjectTransactionalRepository<Meta, Spec, Status> ypRepository;
    private final String description;
    private final SerialExecutor serialExecutor;
    private final Clock clock;
    private final GarbageApprover approver;
    private final ClusterAndType clusterAndType;
    // An error is most likely to manifest itself during startup, so first iteration requires as strict checks as possible.
    // Mass deletions sometimes happen, so controller must be able to deal with them without manual interference,
    // but they should very rarely coincide with restarts.
    private final int garbageCountLimit;
    private final int initialGarbageCountLimit;

    // Modifiable current state.
    private boolean isFirstGarbageCollection = true;
    private volatile boolean removeManyObjects = false;
    private CompletableFuture<Map<String, Try<YpObject<Meta, Spec, Status>>>> ypLoadObjectsFuture;
    private final YpObjectsCache<Meta, Spec, Status> objectsCache;

    // Collections are concurrent because metric collection is asynchronous.
    private final Map<String, BiConsumer<Optional<SpecStatusMeta<Meta, Spec, Status>>, ObjectLifeCycleEventType>> subscribers = new ConcurrentHashMap<>();
    private final Map<String, BiConsumer<Optional<SpecStatusMeta<Meta, Spec, Status>>, ObjectLifeCycleEventType>> subscribersForRemovedEvent = new ConcurrentHashMap<>();
    private final Set<String> objectsBeingModified = ConcurrentHashMap.newKeySet();
    private final Set<String> garbage = ConcurrentHashMap.newKeySet();
    private final Set<String> objectsWithFailedOperations = ConcurrentHashMap.newKeySet();
    private final LoadingCache<Integer, String> cache;

    public ObjectLifeCycleManagerImpl(YpObjectTransactionalRepository<Meta, Spec, Status> ypRepository,
                                      SerialExecutor serialExecutor,
                                      GaugeRegistry metricRegistry,
                                      Clock clock, GarbageApprover approver,
                                      ClusterAndType clusterAndType,
                                      int garbageCountLimit,
                                      int initialGarbageCountLimit,
                                      String description,
                                      YpObjectSettings ypObjectSettings) {
        this.ypRepository = ypRepository;
        this.description = description;
        this.serialExecutor = serialExecutor;
        this.clock = clock;
        this.approver = approver;
        this.clusterAndType = clusterAndType;
        this.garbageCountLimit = garbageCountLimit;
        this.initialGarbageCountLimit = initialGarbageCountLimit;
        metricLastSuccessfulObjectsProcessingTimestamp = clock.instant();

        objectsCache = new YpObjectsCache<>(ypRepository, ypObjectSettings, metricRegistry, SPEC_STATUS_META_LABELS);

        metricRegistry.add(METRIC_RUNNING_OPERATIONS, objectsBeingModified::size);
        metricRegistry.add(METRIC_MANAGED_OBJECTS, subscribers::size);
        metricRegistry.add(METRIC_TOTAL_OBJECTS, new GolovanableGauge<>(() -> metricTotalObjectsCount, "axxx"));
        metricRegistry.add(METRIC_OBJECTS_WITH_FAILED_OPERATIONS_COUNT, objectsWithFailedOperations::size);
        metricRegistry.add(METRIC_LIST_DELAY_SECONDS, () -> Duration.between(metricLastSuccessfulObjectsProcessingTimestamp, clock.instant()).toSeconds());
        metricRegistry.add(METRIC_MODIFY_OPERATION_COUNT, new GolovanableGauge<>(metricModifyOperationCount::get, "dmmm"));
        metricRegistry.add(METRIC_MODIFY_OPERATION_ERRORS_COUNT, new GolovanableGauge<>(metricModifyOperationErrorsCount::get, "dmmm"));

        // Return subscription-based garbage size without filtering out unapproved to avoid checking for retention outside serializer thread.
        // This is upper estimate of final garbage list size (they should be equal in absence of bugs).
        metricRegistry.add(METRIC_GARBAGE_LIMIT_EXCEEDED, () -> MetricUtils.booleanToInt(isOverGarbageLimit(garbage.size())));
        metricRegistry.add(METRIC_GARBAGE_OBJECTS, garbage::size);

        cache = CacheBuilder.newBuilder()
                .expireAfterWrite(TIME_TO_DELETE, TimeUnit.MINUTES)
                .build(new CacheLoader<>() {
                    @Override
                    public String load(Integer key) {
                        return key.toString();
                    }
                });
    }

    public void addError(int hash, String description) {
        cache.put(hash, description);
    }

    public boolean isInCache(int hash) {
        return cache.getIfPresent(hash) != null;
    }

    @Override
    public Optional<String> getCluster() {
        return clusterAndType.getCluster();
    }

    @Override
    public void create(String id, CreateObjectRequest<Spec> request, Runnable onSuccess,
                       Consumer<Throwable> onFailure) {
        runModifyingOperation(id, () -> ypRepository.createObject(id, request), "create", ignored -> onSuccess.run(),
                onFailure);
    }

    @Override
    public void create(String id,
                       Function<YpObjectTransactionalRepository<Meta, Spec, Status>, CompletableFuture<?>> createAction, Runnable onSuccess, Consumer<Throwable> onFailure) {
        runModifyingOperation(id, () -> createAction.apply(ypRepository), "create", ignored -> onSuccess.run(),
                onFailure);
    }

    @Override
    public void updateSpecWithLabels(String id, Spec newSpec, Map<String, YTreeNode> labels,
            Optional<Long> prerequisiteTimestamp, Runnable onSuccess, Consumer<Throwable> onFailure) {
        if (!subscribers.containsKey(id)) {
            throw new IllegalArgumentException(String.format("%s does not contain object %s, cannot update",
                    description, id));
        }
        UpdateYpObjectRequest.Builder<Spec, Status> builder = new UpdateYpObjectRequest.Builder<Spec, Status>()
                .setSpec(newSpec)
                .setLabels(labels);
        prerequisiteTimestamp.ifPresent(timestamp ->
                builder.addPrerequisite(new YpAttributeTimestampPrerequisite("/spec", timestamp)));
        runModifyingOperation(id, () -> ypRepository.updateObject(id, builder.build()),
                "update",
                ignored -> onSuccess.run(),
                onFailure);
    }

    @Override
    public void removeLabel(String id, String label, Runnable onSuccess, Consumer<Throwable> onFailure) {
        if (!subscribers.containsKey(id)) {
            throw new IllegalArgumentException(String.format("%s does not contain object %s, cannot update",
                    description, id));
        }
        UpdateYpObjectRequest.Builder<Spec, Status> builder = new UpdateYpObjectRequest.Builder<Spec, Status>()
                .setRequestExtender(request -> {
                    request.addRemoveUpdate(new YpRemoveUpdate(Paths.LABELS + Paths.DELIMITER + label));
                });
        runModifyingOperation(id, () -> ypRepository.updateObject(id, builder.build()),
                "remove /labels/" + label, ignored -> onSuccess.run(), onFailure);
    }

    @Override
    public void updateAcl(String id, Acl acl, Optional<Long> metaPrerequisiteTimestamp,
            Runnable onSuccess, Consumer<Throwable> onFailure) {
        if (!subscribers.containsKey(id)) {
            throw new IllegalArgumentException(String.format("%s does not contain object %s, cannot update",
                    description, id));
        }
        UpdateYpObjectRequest.Builder<Spec, Status> builder =
                new UpdateYpObjectRequest.Builder<Spec, Status>().setAcl(acl);
        metaPrerequisiteTimestamp.ifPresent(metaTimestamp ->
                builder.addPrerequisite(new YpAttributeTimestampPrerequisite("/meta", metaTimestamp)));
        runModifyingOperation(id, () -> ypRepository.updateObject(id, builder.build()),
                "update acl", ignored -> onSuccess.run(), onFailure);
    }

    @Override
    public void stopManaging(String id) {
        var removedSubscriber = subscribers.remove(id);
        if (removedSubscriber == null) {
            LOG.warn("{} does not contain object '{}', nothing to remove", description, id);
        } else {
            LOG.info("{} stops managing object '{}', will delete from YP on garbage collection", description, id);
            subscribersForRemovedEvent.put(id, removedSubscriber);
            garbage.add(id);
        }
    }

    @Override
    public void startManaging(String id, BiConsumer<Optional<SpecStatusMeta<Meta, Spec, Status>>, ObjectLifeCycleEventType> subscriber) {
        if (subscribers.containsKey(id)) {
            LOG.warn("Replacing existing subscription for id {}. There is probably a bug in calling code.", id);
        }
        subscribers.put(id, subscriber);
        if (garbage.contains(id)) {
            LOG.info("{} got a subscriber for garbage object '{}'", description, id);
            garbage.remove(id);
        }
    }

    @Override
    public void startGcCycle() {
        boolean removeManyObjectsCopy = removeManyObjects;
        removeManyObjects = false;

        garbage.removeAll(subscribers.keySet());
        garbage.removeAll(objectsBeingModified);

        List<String> garbageAfterApprove = garbage.stream()
                .filter(id -> {
                    Retainment retainment = approver.shouldRetain(id, clusterAndType);
                    if (!retainment.isRetained()) {
                        LOG.debug("{}: deletion of {} confirmed", description, id);
                        return true;
                    } else {
                        LOG.info("{}: deletion of {} denied, removing from garbage - {}", description, id, retainment.getReason());
                        return false;
                    }
                })
                .collect(Collectors.toList());
        if (garbageAfterApprove.isEmpty()) {
            return;
        }
        if (isOverGarbageLimit(garbageAfterApprove.size())) {
            if (removeManyObjectsCopy) {
                LOG.info("{}: Force garbage collect many objects", description);
            } else {
                LOG.warn("{}: Skip garbage collect. Total objects count - {}. Garbage objects count {}", description,
                        metricTotalObjectsCount, garbageAfterApprove.size());
                return;
            }
        }

        garbageAfterApprove.forEach(id -> runModifyingOperation(id, () -> ypRepository.removeObject(id),
                "delete", ignored -> {
                    var subscriber = subscribersForRemovedEvent.get(id);
                    if (subscriber != null) {
                        subscriber.accept(empty(), ObjectLifeCycleEventType.REMOVED);
                        subscribersForRemovedEvent.remove(id);
                    }
                }, ignored -> {}));

        garbageAfterApprove.forEach(object -> {
            objectsWithFailedOperations.remove(object);
            garbage.remove(object);
        });

        isFirstGarbageCollection = false;
    }

    @Override
    public CompletableFuture<?> startPolling() {
        ypLoadObjectsFuture = objectsCache.selectObjects(Optional.empty());
        return ypLoadObjectsFuture;
    }

    @Override
    public CompletableFuture<?> processPollingResults() {
        return serialExecutor.submitFuture(ypLoadObjectsFuture, objects -> {
            metricTotalObjectsCount = objects.size();
            // Do not toggle subscriber if an action is running on object to avoid subscriber racing with itself
            Set<String> subscribersToNotify = new HashSet<>(Sets.difference(subscribers.keySet(), objectsBeingModified));

            Sets.difference(subscribersToNotify, objects.keySet()).forEach(key -> {
                try {
                    subscribers.get(key).accept(empty(), ObjectLifeCycleEventType.MISSED);
                } catch (Exception e) {
                    LOG.error("Subscriber for '{}' failed to handle absent object: ", key, e);
                }
            });
            Sets.intersection(subscribersToNotify, objects.keySet()).forEach(key -> {
                try {
                    Try<YpObject<Meta, Spec, Status>> trySpecStatus = objects.get(key);
                    if (trySpecStatus.isFailure()) {
                        LOG.error("Could not parse spec and status of '{}': ", key, trySpecStatus.getThrowable());
                    } else {
                        subscribers.get(key).accept(Optional.of(SpecStatusMeta.fromYpObject(trySpecStatus.get())), ObjectLifeCycleEventType.RELOADED);
                    }
                } catch (Exception e) {
                    LOG.error("Subscriber for '{}' failed to handle response: ", key, e);
                }
            });
            Sets.difference(objects.keySet(), subscribers.keySet()).forEach(unknownId -> {
                if (objectsBeingModified.contains(unknownId)) {
                    LOG.info("{} will not mark '{}' as garbage because it is currently updated", description, unknownId);
                } else if (!garbage.contains(unknownId)) {
                    Retainment retainment = approver.shouldRetain(unknownId, clusterAndType);
                    if (retainment.isRetained()) {
                        LOG.debug("{} will not mark id '{}' to garbage - denied by approver", description, unknownId);
                    } else {
                        LOG.info("{} is allowed to move '{}' to garbage: {}", description, unknownId, retainment.getReason());
                        garbage.add(unknownId);
                    }
                }
            });
            metricLastSuccessfulObjectsProcessingTimestamp = clock.instant();
        }, error -> LOG.error("{} could not retrieve objects from yp: ", description, error));
    }

    @Override
    public ClusterAndType getClusterAndType() {
        return clusterAndType;
    }

    @Override
    public Set<String> getGarbage() {
        return garbage;
    }

    @Override
    public CompletableFuture<?> forceCollectGarbage() {
        return serialExecutor.submitFuture(CompletableFuture.completedFuture(null), ignore -> {
            LOG.info("{}: Receive request to force garbage collect", clusterAndType.toString());
            removeManyObjects = true;
        }, error -> {});
    }

    private <T> void runModifyingOperation(String id, Supplier<CompletableFuture<T>> supplier, String operation,
                                           Consumer<T> onSuccess, Consumer<Throwable> onFailure) {
        if (objectsBeingModified.contains(id)) {
            throw new IllegalStateException(String.format("Another operation is already running on '%s'", id));
        }
        metricModifyOperationCount.incrementAndGet();
        LOG.info("{} - will {} id '{}'", description, operation, id);
        objectsBeingModified.add(id);
        garbage.remove(id);
        serialExecutor.submitFuture(supplier.get(),
                result -> {
                    objectsWithFailedOperations.remove(id);
                    objectsBeingModified.remove(id);
                    LOG.info("{} - {} id '{}' finished successfully", description, operation, id);
                    onSuccess.accept(result);
                },
                error -> {
                    metricModifyOperationErrorsCount.incrementAndGet();
                    objectsWithFailedOperations.add(id);
                    objectsBeingModified.remove(id);
                    Throwable cause = ExceptionUtils.stripCompletionException(error);
                    LOG.warn(String.format("%s - failed to %s '%s'", description, operation, id), cause);
                    onFailure.accept(cause);
                }
        );
    }

    private boolean isOverGarbageLimit(int garbageCount) {
        return (isFirstGarbageCollection && garbageCount > initialGarbageCountLimit)
                || (garbageCount > garbageCountLimit);
    }

    @Override
    public String toString() {
        return description;
    }
}
