package ru.yandex.infra.stage.yp;

import java.time.Duration;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.infra.controller.RepeatedTask;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.util.ExceptionUtils;
import ru.yandex.infra.controller.util.YsonUtils;
import ru.yandex.infra.controller.yp.Paths;
import ru.yandex.infra.controller.yp.YpRequestWithPaging;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.infra.controller.yp.YpTransactionClientImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeProtoUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.Autogen;
import ru.yandex.yp.model.YpErrorCodes;
import ru.yandex.yp.model.YpGetStatement;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpObjectUpdate;
import ru.yandex.yp.model.YpPayloadFormat;
import ru.yandex.yp.model.YpSelectStatement;
import ru.yandex.yp.model.YpSetUpdate;
import ru.yandex.yp.model.YpTransaction;
import ru.yandex.yp.model.YpTypedId;

import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class RelationControllerImpl implements RelationController {
    private static final Logger LOG = LoggerFactory.getLogger(RelationControllerImpl.class);

    static final String METRIC_RELATIONS = "count";
    static final String METRIC_RELOAD_TIME_MS = "reload_time_ms";
    static final String METRIC_ADD_DEPLOY_CHILD_COUNT = "add_deploy_child_count";
    static final String METRIC_REMOVE_RELATION_COUNT = "remove_relation_count";
    static final String METRIC_SUCCESS_ADD_DEPLOY_CHILD_COUNT = "success_add_deploy_child_count";
    static final String METRIC_FAILED_ADD_DEPLOY_CHILD_COUNT = "failed_add_deploy_child_count";
    static final String METRIC_FAILED_REMOVE_RELATION_COUNT = "failed_remove_relation_count";
    static final String METRIC_RELATIONS_RELOAD_COUNT = "relations_reload_count";
    static final String METRIC_FAILED_RELATIONS_RELOAD_COUNT = "failed_relations_reload_count";
    static final String METRIC_MISSED_RELATION_EVENTS_COUNT = "missed_relation_events_count";
    private final AtomicLong metricAddDeployChildCount = new AtomicLong();
    private final AtomicLong metricRemoveRelationCount = new AtomicLong();
    private final AtomicLong metricSuccessAddDeployChildCount = new AtomicLong();
    private final AtomicLong metricFailedAddDeployChildCount = new AtomicLong();
    private final AtomicLong metricFailedRemoveRelationCount = new AtomicLong();
    private final AtomicLong metricRelationsReloadCount = new AtomicLong();
    private final AtomicLong metricFailedRelationsReloadCount = new AtomicLong();
    private final AtomicLong metricMissedRelationEventsCount = new AtomicLong();
    private Long metricReloadTimeMilliseconds;
    private Integer metricRelationsCount;

    private final YpRawObjectService ypClient;
    private final YpTransactionClient ypTransactionClient;
    private final boolean addMissedRelations;
    private final boolean removeRelations;
    private final int requestPageSize;

    private final Object syncObject = new Object();
    private Map<Tuple2<String, String>, String> allRelations;
    private final Set<Tuple2<String, String>> relationsAddedSinceLastSync = new HashSet<>();
    private final RepeatedTask syncLoop;
    private final Duration syncTimeout;

    public RelationControllerImpl(YpRawObjectService ypClient,
                                  GaugeRegistry registry,
                                  boolean addMissedRelations,
                                  boolean removeRelations,
                                  Duration cacheResetInterval,
                                  Duration initRetryInterval,
                                  Duration syncTimeout,
                                  int requestPageSize) {
        this.ypClient = ypClient;
        this.ypTransactionClient = new YpTransactionClientImpl(ypClient);
        this.addMissedRelations = addMissedRelations;
        this.removeRelations = removeRelations;
        this.requestPageSize = requestPageSize;
        this.syncTimeout = syncTimeout;

        init(initRetryInterval);
        addAllMetrics(registry);

        ScheduledExecutorService scheduledExecutorService =
                Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable,
                "relations"));

        syncLoop = new RepeatedTask(this::sync, cacheResetInterval, syncTimeout, scheduledExecutorService,
                Optional.of(registry), LOG, true);
        if (cacheResetInterval.isZero()) {
            LOG.warn("Relations reload is disabled by config option cache_reset_interval = 0");
        } else {
            scheduledExecutorService.schedule(syncLoop::start, cacheResetInterval.toMillis(), TimeUnit.MILLISECONDS);
        }

    }

    public void init(Duration initRetryInterval) {
        while (true) {
            try {
                sync().get(syncTimeout.toMillis(), TimeUnit.MILLISECONDS);
                break;
            } catch (Exception exception) {
                LOG.warn("Failed to load relations", exception);
            }
            try {
                TimeUnit.MILLISECONDS.sleep(initRetryInterval.toMillis());
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void addAllMetrics(GaugeRegistry registry) {
        registry.add(METRIC_RELATIONS, new GolovanableGauge<>(() -> metricRelationsCount, "axxx"));
        registry.add(METRIC_RELOAD_TIME_MS, new GolovanableGauge<>(() -> metricReloadTimeMilliseconds, "axxx"));
        registry.add(METRIC_ADD_DEPLOY_CHILD_COUNT, new GolovanableGauge<>(metricAddDeployChildCount::get, "dmmm"));
        registry.add(METRIC_SUCCESS_ADD_DEPLOY_CHILD_COUNT, new GolovanableGauge<>(metricSuccessAddDeployChildCount::get, "dmmm"));
        registry.add(METRIC_FAILED_ADD_DEPLOY_CHILD_COUNT, new GolovanableGauge<>(metricFailedAddDeployChildCount::get, "dmmm"));
        registry.add(METRIC_REMOVE_RELATION_COUNT, new GolovanableGauge<>(metricRemoveRelationCount::get, "dmmm"));
        registry.add(METRIC_FAILED_REMOVE_RELATION_COUNT, new GolovanableGauge<>(metricFailedRemoveRelationCount::get, "dmmm"));
        registry.add(METRIC_RELATIONS_RELOAD_COUNT, new GolovanableGauge<>(metricRelationsReloadCount::get, "dmmm"));
        registry.add(METRIC_FAILED_RELATIONS_RELOAD_COUNT, new GolovanableGauge<>(metricFailedRelationsReloadCount::get, "dmmm"));
        registry.add(METRIC_MISSED_RELATION_EVENTS_COUNT, new GolovanableGauge<>(metricMissedRelationEventsCount::get, "dmmm"));
    }

    private CompletableFuture<?> sync() {
        return loadAllRelations()
                .thenAcceptAsync(relations -> {
                    synchronized (syncObject) {
                        relationsAddedSinceLastSync.forEach(key -> relations.putIfAbsent(key, null));
                        relationsAddedSinceLastSync.clear();
                        allRelations = relations;
                        metricRelationsCount = allRelations.size();
                    }
                });
    }

    public void shutdown() {
        syncLoop.stop();
    }

    private CompletableFuture<Map<Tuple2<String, String>, String>> loadAllRelations() {
        metricRelationsReloadCount.incrementAndGet();
        long startTimeMillis = System.currentTimeMillis();
        LOG.info("Loading all relations from yp...");

        YpSelectStatement.Builder builder = YpSelectStatement.builder(YpObjectType.RELATION, YpPayloadFormat.YSON)
                .addSelector(Paths.ID)
                .addSelector("/meta/from_fqid")
                .addSelector("/meta/to_fqid")
                .setFilter("is_substr('|stage|', [/meta/from_fqid])");
        return YpRequestWithPaging.selectObjects(ypClient, requestPageSize, builder,
                        payloads -> {
                            String id = payloadToYson(payloads.get(0)).stringValue();
                            String fromFqid = payloadToYson(payloads.get(1)).stringValue();
                            String toFqid = payloadToYson(payloads.get(2)).stringValue();

                            return Tuple2.tuple(Tuple2.tuple(fromFqid, toFqid), id);
                        })
                .whenCompleteAsync((result, error) -> {
                    if (error != null) {
                        metricFailedRelationsReloadCount.incrementAndGet();
                        LOG.error("Failed to reload relations", error);
                    } else {
                        metricReloadTimeMilliseconds = System.currentTimeMillis() - startTimeMillis;
                        LOG.info("Loaded {} relations in {} ms", result.size(), metricReloadTimeMilliseconds);
                    }
                })
                .thenApply(objects -> objects.stream().collect(Collectors.toMap(t -> t._1, t -> t._2)));
    }

    @Override
    public boolean containsRelation(String stageFqid, String childFqid) {
        final Tuple2<String, String> relation = Tuple2.tuple(stageFqid, childFqid);
        synchronized (syncObject) {
            if (allRelations.containsKey(relation) || relationsAddedSinceLastSync.contains(relation)) {
                return true;
            }
        }

        metricMissedRelationEventsCount.incrementAndGet();
        LOG.warn("Found RS/MCRS {} with missed relation to stage {}", childFqid, stageFqid);

        if (addMissedRelations) {
            addRelation(stageFqid, childFqid);
            return true;
        }

        return false;
    }

    @Override
    public CompletableFuture<?> addRelation(String stageFqid, String childFqid) {
        LOG.info("Trying to add new relation: {} -> {}", stageFqid, childFqid);
        metricAddDeployChildCount.incrementAndGet();

        return ypTransactionClient.runWithTransaction(transaction ->
                        checkStageAndUpdateRelation(transaction, stageFqid, childFqid, RelationOperation.ADD))
                .whenCompleteAsync((result, error) -> {
                    final Tuple2<String, String> tuple = Tuple2.tuple(stageFqid, childFqid);
                    if (error == null) {
                        metricSuccessAddDeployChildCount.incrementAndGet();
                        LOG.info("Added relation: {} -> {}", stageFqid, childFqid);
                        synchronized (syncObject) {
                            relationsAddedSinceLastSync.add(tuple);
                            metricRelationsCount++;
                        }
                    } else {
                        metricFailedAddDeployChildCount.incrementAndGet();
                        if (isYpObjectAlreadyExistError(error)) {
                            LOG.warn(String.format("Relation already exist: %s -> %s", stageFqid, childFqid), error);
                            synchronized (syncObject) {
                                relationsAddedSinceLastSync.add(tuple);
                            }
                        } else {
                            LOG.error(String.format("Failed to add relation: %s -> %s", stageFqid, childFqid), error);
                        }
                    }
                });
    }

    @Override
    public CompletableFuture<?> removeRelation(String stageFqid, String childFqid) {
        if (!removeRelations) {
            return CompletableFuture.completedFuture(null);
        }
        LOG.info("Trying to remove relation: {} -> {}", stageFqid, childFqid);
        metricRemoveRelationCount.incrementAndGet();

        final Tuple2<String, String> relation = Tuple2.tuple(stageFqid, childFqid);
        return ypTransactionClient.runWithTransaction(transaction ->
                        checkStageAndUpdateRelation(transaction, stageFqid, childFqid, RelationOperation.REMOVE))
                .whenCompleteAsync((result, error) -> {
                    if (error == null) {
                        LOG.info("Removed relation: {} -> {}", stageFqid, childFqid);
                        synchronized (syncObject) {
                            allRelations.remove(relation);
                            relationsAddedSinceLastSync.remove(relation);
                            metricRelationsCount--;
                        }
                    } else {
                        metricFailedRemoveRelationCount.incrementAndGet();
                        LOG.error(String.format("Failed to remove relation: %s -> %s", stageFqid, childFqid), error);
                    }
                });
    }

    private static boolean isYpObjectAlreadyExistError(Throwable error) {
        return ExceptionUtils.tryExtractYpError(error)
                .filter(ypError -> ypError.getCode() == YpErrorCodes.DUPLICATE_OBJECT_ID).isPresent();
    }

    @VisibleForTesting
    public static String getStageId(String stageFqid) {
        final String[] splittedFqid = stageFqid.split("\\|");
        //expecting Fqid value in format like "yp|sas-test|stage|vzstage9|d7c28a6b-a59aedd2-25eb200a-301468c4"
        if (splittedFqid.length == YpFqidColumns.values().length &&
                splittedFqid[YpFqidColumns.SCHEMA.ordinal()].equals("yp") &&
                splittedFqid[YpFqidColumns.OBJECT_TYPE.ordinal()].equals("stage")) {
            return splittedFqid[YpFqidColumns.OBJECT_ID.ordinal()];
        }

        throw new RuntimeException("Unsupported fqid format: " + stageFqid);
    }

    private CompletableFuture<StageMeta> getStageMeta(YpTypedId typedStageId, YpTransaction transaction) {
        return ypClient.getObject(
                YpGetStatement.ysonBuilder(typedStageId)
                        .setTimestamp(transaction.getStartTimestamp())
                        .addSelector(Paths.META)
                        .build(),
                payloads -> {
                    Autogen.TStageMeta proto = (Autogen.TStageMeta) YTreeProtoUtils.unmarshal(
                            payloadToYson(payloads.get(0)), Autogen.TStageMeta.newBuilder());
                    return StageMeta.fromProto(proto);
                });
    }

    enum RelationOperation {
        ADD,
        REMOVE
    }

    private CompletableFuture<?> checkStageAndUpdateRelation(YpTransaction transaction, String stageFqid,
                                                             String childFqid,
                                                             RelationOperation operation) {

        final String stageId = getStageId(stageFqid);
        final YpTypedId typedStageId = new YpTypedId(stageId, YpObjectType.STAGE);

        return getStageMeta(typedStageId, transaction)
                .thenAccept(stageMeta -> {
                    final String fqid = stageMeta.getFqid();
                    if (!fqid.equals(stageFqid)) {
                        throw new RuntimeException(String.format("Stage with id '%s' was replaced. Expected '%s' but got '%s'",
                                stageId, stageFqid, fqid));
                    }
                })
                .thenCompose(ignore -> addOrRemoveRelation(transaction, childFqid, typedStageId, operation));
    }

    private CompletableFuture<Void> addOrRemoveRelation(YpTransaction transaction, String childFqid,
                                                        YpTypedId typedParentId,
                                                        RelationOperation operation) {
        String path = operation == RelationOperation.ADD ? "/control/add_deploy_child" : "/control/remove_deploy_child";
        return ypClient.updateObject(YpObjectUpdate.builder(typedParentId)
                .addSetUpdate(new YpSetUpdate(
                        path,
                        new YTreeBuilder()
                                .beginMap()
                                .key("fqid").value(childFqid)
                                .endMap()
                                .build(),
                        YsonUtils::toYsonPayload
                ))
                .build(), transaction);
    }

    @VisibleForTesting
    Map<Tuple2<String, String>, String> getAllRelations() {
        return allRelations;
    }
}
