import argparse
import logging
import time
import yp.data_model as data_model
import yt_proto.yt.core.yson.proto.protobuf_interop_pb2 as yson_protobuf_interop

from infra.dctl.src.consts import CLUSTER_CONFIGS
from infra.dctl.src.lib import yp_client
from yp.client import find_token
from yp.common import YpNoSuchObjectError
from yt.wrapper.errors import YtTabletTransactionLockConflict


logging.basicConfig(level=logging.DEBUG)


TYPE_MAP = {
    'stage': data_model.OT_STAGE,
    'project': data_model.OT_PROJECT,
    'group': data_model.OT_GROUP
}


def _gc_objects(client, object_type, get_objects_func, dry_run):
    objects = get_objects_func()
    while objects:
        for object in objects:
            object_id = object.meta.id
            object_name = data_model.EObjectType.DESCRIPTOR.values_by_number[object_type].GetOptions().Extensions[
                yson_protobuf_interop.enum_value_name
            ]

            logging.info("Removing garbage %s %s", object_name, object_id)
            print(f'Removing garbage {object_name} {object_id}')
            if dry_run:
                continue
            try:
                client.remove(object_type=object_type, object_id=object_id)
            except (YpNoSuchObjectError, YtTabletTransactionLockConflict):
                logging.warn("Unable to remove garbage %s %s", object_name, object_id)
        objects = get_objects_func()


def _gc_objects_by_label(client, object_type, gc_label_key, gc_label, start_timestamp_seconds, dry_run):
    query = f'[/labels/{gc_label_key}] = "{gc_label}" and [/meta/creation_time] < {int(start_timestamp_seconds * 1e6)}u'
    def get_garbage_objects():
        return client.list(object_type=object_type, user=None, query=query, limit=100)

    _gc_objects(client, object_type, get_garbage_objects, dry_run)


def _gc_objects_by_prefix(client, object_type, id_prefix, start_timestamp_seconds, dry_run):
    query = f'[/meta/id] >= "{id_prefix}" and [/meta/creation_time] < {int(start_timestamp_seconds * 1e6)}u'
    def get_garbage_objects():
        objects = client.list(object_type=object_type, user=None, query=query, limit=100)
        return list(filter(lambda x: x.meta.id.startswith(id_prefix), objects))

    _gc_objects(client, object_type, get_garbage_objects, dry_run)


def parse_args():
    parser = argparse.ArgumentParser(description='StageCtl IT GC')
    parser.add_argument('--gc-labels', dest='gc_labels', default=None,
                        help='Labels for filtering in form of gc_key=gc_value[,gc_key1=gc_value2]')
    parser.add_argument('--gc-ttl', dest='gc_ttl', default=None, required=True,
                        help='Maximum allowed garbage age (in minutes)')
    parser.add_argument('--gc-prefix', dest='gc_prefix', default=None,
                        help='Prefix for filtering objects')
    parser.add_argument('--user', dest='user', default=None, required=True,
                        help='User of token')
    parser.add_argument('--cluster', dest='cluster', default=None, required=True,
                        help='Cluster for cleaning')
    parser.add_argument('--objects', dest='gc_objects', default=None, required=True,
                        help='Object types in form of stage[,project]...')
    parser.add_argument('--dry-run', dest='dry_run', default=False, type=bool,
                        help='Dry run flag')
    return parser.parse_args()


def main():
    args = parse_args()

    start_timestamp_seconds = time.time() - int(args.gc_ttl) * 60
    yp_cluster = args.cluster
    gc_labels = args.gc_labels
    gc_prefix = args.gc_prefix
    gc_objects = args.gc_objects
    if yp_cluster not in CLUSTER_CONFIGS.keys():
        raise Exception(f'Unsupported cluster: {yp_cluster}')

    client = _create_yp_client_fixture(args.user, yp_cluster)
    for gc_object in gc_objects.split(','):
        if gc_object not in TYPE_MAP.keys():
            raise Exception(f'Unsupported object type: {gc_object}')
        object_type = TYPE_MAP[gc_object]
        if gc_labels is not None:
            for pair in gc_labels.split(','):
                gc_key, gc_value = pair.split('=')
                _gc_objects_by_label(client, object_type, gc_key, gc_value, start_timestamp_seconds, args.dry_run)
        if gc_prefix is not None:
            _gc_objects_by_prefix(client, object_type, gc_prefix, start_timestamp_seconds, args.dry_run)
    client.close()


def _create_yp_client_fixture(user, cluster):
    logging.info('Creating client for cluster %s', cluster)
    client = yp_client.YpClient(cluster, find_token(), user)
    return client


if __name__ == '__main__':
    main()
