import requests

from juggler_sdk import Check, FlapOptions, NotificationOptions


YASM_API_URL = 'https://yasm.yandex-team.ru/srvambry/'

JUGGLER_HOSTNAME = 'infra'
JUGGLER_NAMESPACE = 'infra'
DISABLE_ALERTS_UPDATING = False
ALERT_SUFFIX = ''
ALERT_DESCRIPTION = None
USER = 'nanny-robot'
DEFAULT_TAG = None
EDITORS = []


def setup(juggler_hostname=None, juggler_namespace=None, disable_alerts_updating=None, user=None, default_tag=None,
          editors=None, alert_suffix=None, alert_description=None):
    global JUGGLER_NAMESPACE, JUGGLER_HOSTNAME, DISABLE_ALERTS_UPDATING, USER, DEFAULT_TAG, EDITORS, ALERT_SUFFIX, ALERT_DESCRIPTION
    if juggler_hostname is not None:
        JUGGLER_HOSTNAME = juggler_hostname
    if juggler_namespace is not None:
        JUGGLER_NAMESPACE = juggler_namespace
    if disable_alerts_updating is not None:
        DISABLE_ALERTS_UPDATING = disable_alerts_updating
    if user is not None:
        USER = user
    if default_tag is not None:
        DEFAULT_TAG = default_tag
    if editors is not None:
        EDITORS = editors
    if alert_suffix is not None:
        ALERT_SUFFIX = alert_suffix
    if alert_description is not None:
        ALERT_DESCRIPTION = alert_description


class JugglerCheck(Check):
    DEFAULTS = {
        'aggregator_kwargs': {
            'nodata_mode': 'force_crit',
            'unreach_mode': 'force_ok',
            'unreach_service': [{'check': 'yasm_alert:virtual-meta'}],
        },
        'refresh_time': 5,
    }

    def __init__(self, **kwargs):
        assert 'host' in kwargs
        assert 'service' in kwargs
        for key, default in self.DEFAULTS.items():
            kwargs.setdefault(key, default)

        super(JugglerCheck, self).__init__(**kwargs)


class Signal(object):
    def __init__(self, name, host='ASEARCH', title=None, tag=None, continuous=False, color=None):
        self.name = name
        self.host = host
        self.tag = tag or DEFAULT_TAG
        self.title = title or name
        self.continuous = continuous
        self.color = color

    def to_json(self):
        rv = {}
        for attr in ('name', 'host', 'tag', 'title', 'continuous', 'color'):
            if getattr(self, attr) is not None:
                rv[attr] = getattr(self, attr)
        return rv


class Chart(object):
    type = None
    width = 1
    height = 1

    def to_json(self):
        raise NotImplementedError


class Graphic(Chart):
    type = 'graphic'
    width = 2

    def __init__(self, signals, name=None, id=None, title=None, width=None, height=None,
                 range_=60 * 25, stacked=False, normalize=False):
        """
        :type signals: list[Signal]
        """
        assert len(signals) > 0

        self.signals = signals
        self.name = name or signals[0].name
        self.id = id or self.name
        self.title = title or signals[0].title + '...' * (len(signals) > 1)
        self.range = range_
        self.normalize = normalize
        self.stacked = stacked
        if width:
            self.width = width
        if height:
            self.height = height

    def to_json(self):
        rv = {
            'signals': [signal.to_json() for signal in self.signals],
        }
        for attr in ('name', 'id', 'title', 'width', 'height', 'type', 'range', 'stacked', 'normalize'):
            if getattr(self, attr) is not None:
                rv[attr] = getattr(self, attr)
        return rv

    def append(self, signals):
        if isinstance(signals, list):
            self.signals.extend(signals)
        else:
            self.signals.append(signals)


class Text(Chart):
    type = 'text'
    width = 4

    def __init__(self, text='', width=None, height=None, color=None, bgColor=None):
        self.text = text
        self.color = color
        self.bgColor = bgColor
        if width:
            self.width = width
        if height:
            self.height = height

    def to_json(self):
        rv = {}
        for attr in ('text', 'width', 'height', 'color', 'bgColor', 'type'):
            if getattr(self, attr) is not None:
                rv[attr] = getattr(self, attr)
        return rv


class Alert(Chart):
    type = 'alert'

    def __init__(self, name, signal=None, tags=None, crit=None, warn=None, title=None, value_modify=None,
                 trend=None, warn_perc=None, crit_perc=None, interval=None, juggler_check=None, width=None, height=None,
                 description=None):
        """
        :type signal: Signal
        """
        self.name = name + ALERT_SUFFIX
        self.signal = signal
        self.tags = tags
        self.title = title or name
        self.crit = crit
        self.warn = warn

        self.trend = trend
        self.warn_perc = warn_perc
        self.crit_perc = crit_perc
        self.interval = interval

        self.value_modify = value_modify

        self.juggler_check = juggler_check
        if width:
            self.width = width
        if height:
            self.height = height

        self.description = description or ALERT_DESCRIPTION

    def with_juggler_check(self, **kwargs):
        kwargs.setdefault('host', JUGGLER_HOSTNAME)
        kwargs.setdefault('service', self.name)
        kwargs.setdefault('namespace', JUGGLER_NAMESPACE)
        self.juggler_check = JugglerCheck(**kwargs)
        return self

    def update(self):
        if DISABLE_ALERTS_UPDATING:
            return self
        if self.exists():
            url = YASM_API_URL + 'alerts/update?name={}'.format(self.name)
        else:
            url = YASM_API_URL + 'alerts/create'
        data = self.to_alert()
        r = requests.post(url, json=data)
        try:
            r.raise_for_status()
        except:
            print(r.text)
            raise

        print('Alert {} updated'.format(self.name))
        return self

    def exists(self):
        url = YASM_API_URL + 'alerts/get'
        params = {
            'name': self.name
        }
        r = requests.get(url, params=params)
        if r.status_code == 404:
            return False
        r.raise_for_status()
        return True

    def to_json(self):
        rv = dict()
        for attr in ('name', 'width', 'height', 'type', 'title'):
            if getattr(self, attr) is not None:
                rv[attr] = getattr(self, attr)
        return rv

    def to_alert(self):
        assert self.signal

        tags = {}
        for key, value in (tag.split('=') for tag in self.signal.tag.split(';')):
            tags[key] = value
        rv = {
            'name': self.name,
            'signal': self.signal.name,
            'tags': tags,
            'mgroups': [self.signal.host]
        }

        if self.trend:
            attrs = ('trend', 'warn_perc', 'crit_perc', 'interval')
        else:
            attrs = ('warn', 'crit')
        for attr in attrs:
            rv[attr] = getattr(self, attr)

        if self.value_modify:
            assert self.value_modify['type'] in ('aver', 'max', 'min', 'summ')
            rv['value_modify'] = self.value_modify

        if self.juggler_check:
            rv['juggler_check'] = self.juggler_check.to_dict()

        if self.description:
            rv['description'] = self.description

        return rv


class Panel(object):
    def __init__(self, key, charts=None, title=None, description=None, editors=None):
        """
        :type charts: list[Chart]
        """
        self.key = key
        self.charts = charts or []
        self.title = title
        self.description = description
        self.editors = editors or EDITORS

        self._validate_chart_names()

    def _validate_chart_names(self):
        seen_names = set()
        for chart in self.charts:
            if isinstance(chart, Graphic):
                if chart.name in seen_names:
                    raise ValueError(u'charts with duplicate name found: "{}"'.format(chart.name))
                seen_names.add(chart.name)

    def append(self, charts):
        if isinstance(charts, list):
            self.charts.extend(charts)
        else:
            self.charts.append(charts)
        self._validate_chart_names()

    def to_json(self):
        rv = {
            'type': 'panel',
            'charts': []
        }
        for attr in ('title', 'description', 'editors'):
            if getattr(self, attr):
                rv[attr] = getattr(self, attr)
        for chart in self.charts:
            rv['charts'].append(chart.to_json())
        return rv

    def upsert(self):
        assert len(self.charts) > 0

        url = YASM_API_URL + 'upsert'
        data = {
            'keys': {
                'user': USER,
                'key': self.key
            },
            'values': self.to_json()
        }
        r = requests.post(url, json=data)
        r.raise_for_status()

        print('Panel {} updated'.format(self.key))
