# coding: utf-8
from collections import defaultdict

import copy
import gevent
from infra.swatlib.httpclient import HttpClient, HttpClientException

import inject


class AbcError(HttpClientException):
    pass


class IAbcClient(object):
    @classmethod
    def instance(cls):
        """:rtype: AbcClient"""
        return inject.instance(cls)

    def list_service_members(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?service=service_id

        Returns a list of service members.

        :type service_id: int
        :rtype: list[unicode]
        """

    def is_service_member(self, login, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?person__login=login

        Returns True if user is a service member, False otherwise

        :type login: unicode
        :type service_id: int
        :rtype: bool
        """

    def list_service_member_role_slugs(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?service=service_id

        Returns a list of service member roles.

        :type service_id: int
        :rtype: list[unicode]
        """

    def get_service_member_roles(self, service_id, login=None):
        """
        Returns list of roles for given service id and login.

        :type service_id: int
        :type login: unicode
        :rtype: list[dict]
        """

    def get_service_slug(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/service_id/

        Returns the service slug

        :type service_id: int
        :rtype: unicode
        """

    def get_service(self, service_id, fields=None):
        """
        https://abc-back.yandex-team.ru/api/v4/services/service_id/

        Returns the service by id

        :type service_id: int
        :type fields: Optional[list[str]]
        :rtype: dict
        """

    def multicomplete(self, spec):
        """
        https://abc-back.yandex-team.ru/multic

        :type spec: dict
        :rtype: dict
        """

    def list_services(self, spec):
        """
        https://abc-back.yandex-team.ru/api/v4/services/

        :type spec: dict
        :rtype: dict
        """

    def list_members(self, spec):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/

        :type spec: dict
        :rtype: dict
        """

    def list_members_iter(self, spec):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/

        :type spec: dict
        :rtype: dict
        """

    def list_user_services(self, login):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?person__login=login

        Returns a mapping service_id:role_ids for given user

        :type login: unicode
        :rtype: dict[str, list[str]]
        """


class AbcClient(IAbcClient):
    ABC_API_URL = 'https://abc-back.yandex-team.ru/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_RESPONSE_LIMIT = 200
    DEFAULT_VERIFY_SSL = True
    DEFAULT_CACHE_SLUGS = False
    DEFAULT_POPULATE_SLUGS_IN_BACKGROUND = False

    @classmethod
    def from_config(cls, d):
        return cls(abc_url=d.get('api_url'),
                   oauth_token=d.get('oauth_token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   max_retries=d.get('max_retries'),
                   cache_slugs=d.get('cache_slugs'),
                   populate_slugs_in_background=d.get('populate_slugs_in_background'))

    def __init__(self, abc_url=None, oauth_token=None, req_timeout=None, verify_ssl=None, max_retries=None,
                 cache_slugs=None, populate_slugs_in_background=False):
        abc_url = abc_url or self.ABC_API_URL
        token = oauth_token
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        cache_slugs = self.DEFAULT_CACHE_SLUGS if cache_slugs is None else cache_slugs
        self._slugs_cache = {} if cache_slugs else None
        if populate_slugs_in_background is None:
            populate_slugs_in_background = self.DEFAULT_POPULATE_SLUGS_IN_BACKGROUND
        self._client = HttpClient(client_name='abc',
                                  exc_cls=AbcError,
                                  base_url=abc_url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries)
        if populate_slugs_in_background and not cache_slugs:
            raise ValueError("populate_slugs_in_background cannot be True while cache_slugs is not")
        if populate_slugs_in_background:
            gevent.spawn(self._populate_slugs)

    def list_service_members(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?service=service_id

        Returns a list of service members.

        :type service_id: int
        :rtype: list[unicode]
        """
        result = []
        for resp in self.list_members_iter(spec={'service': service_id, 'fields': 'person'}):
            for item in resp['results']:
                result.append(item['person']['login'])
        return result

    def is_service_member(self, login, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?person__login=login

        Returns True if user is a service member, False otherwise

        :type service_id: int
        :type login: unicode
        :rtype: bool
        """
        for resp in self.list_members_iter(spec={'person__login': login, 'fields': 'service'}):
            for item in resp['results']:
                if item['service']['id'] == service_id:
                    return True
        return False

    def get_service_member_roles(self, service_id, login=None):
        """
        Returns list of roles for given service id and login.

        :type service_id: int
        :type login: unicode
        :rtype: list[dict]
        """
        spec = {
            'service': service_id,
            'fields': 'role',
        }
        if login is not None:
            spec['person__login'] = login
        roles = []
        for resp in self.list_members_iter(spec=spec):
            for item in resp['results']:
                roles.append(item['role'])
        return roles

    def list_user_services(self, login):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?person__login=login

        Returns a mapping service_id:list_of_role_ids for given user

        :type login: unicode
        :rtype: dict[str, list[str]]
        """
        all_services = defaultdict(list)
        for resp in self.list_members_iter(spec={'person__login': login, 'fields': 'service,role'}):
            for item in resp['results']:
                all_services[str(item['service']['id'])].append(str(item['role']['id']))
        return all_services

    def list_service_member_role_slugs(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?service=service_id

        Returns a list of service member roles.

        :type service_id: int
        :rtype: list[unicode]
        """
        result = []
        for resp in self.list_members_iter(spec={'service': service_id, 'fields': 'role'}):
            for item in resp['results']:
                result.append(item['role']['scope']['slug'])
        return result

    def get_service_slug(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/service_id/

        Returns the service slug

        :type service_id: int
        :rtype: unicode
        """
        if self._slugs_cache is not None and service_id in self._slugs_cache:
            return self._slugs_cache[service_id]

        slug = self.get_service(service_id, fields=['slug'])['slug']
        if self._slugs_cache is not None:
            self._slugs_cache[service_id] = slug

        return slug

    def get_service(self, service_id, fields=None):
        """
        https://abc-back.yandex-team.ru/api/v4/services/service_id/

        Returns the service by id

        :type service_id: int
        :type fields: Optional[list[str]]
        :rtype: dict
        """
        params = None if not fields else {'fields': ','.join(fields)}
        return self._client.get('/api/v4/services/{}/'.format(service_id), params=params)

    def get_schedule_service_id(self, schedule_id):
        """
        https://abc-back.yandex-team.ru/api/v4/duty/schedules/schedule_id/

        Returns the ABC service id of schedule

        :type schedule_id: int
        :rtype: int
        """
        return self.get_schedule(schedule_id, fields=['service.id'])['service']['id']

    def get_schedule(self, schedule_id, fields=None):
        """
        https://abc-back.yandex-team.ru/api/v4/duty/schedules/schedule_id/

        Returns the schedule by id

        :type schedule_id: int
        :type fields: Optional[list[str]]
        :rtype: dict
        """
        params = {} if not fields else {'fields': ','.join(fields)}
        params['id'] = schedule_id
        params['with_watcher'] = 1
        schedules = self._client.get('/api/v4/duty/schedules/', params=params)['results']
        if len(schedules) == 0:
            raise AbcError('Schedule {} not found'.format(schedule_id))
        return schedules[0]

    def multicomplete(self, spec):
        return self._client.get('/multic/', params=spec)

    def list_services(self, spec):
        return self._client.get('/api/v4/services/', params=self._set_response_limit(spec))

    def list_members(self, spec):
        return self._client.get('/api/v4/services/members/', params=self._set_response_limit(spec))

    def list_services_iter(self, spec):
        spec = self._set_response_limit(spec)
        resp = self.list_services(spec)
        for service in resp['results']:
            yield service

        next_url = resp.get('next')
        while next_url:
            resp = self._client.get(next_url, params=spec)
            for service in resp['results']:
                yield service
            next_url = resp.get('next')

    def list_members_iter(self, spec):
        spec = self._set_response_limit(spec)
        resp = self.list_members(spec)
        yield resp
        next_url = resp.get('next')
        while next_url:
            resp = self._client.get(next_url, params=spec)
            yield resp
            next_url = resp.get('next')

    def _set_response_limit(self, spec):
        if 'page_size' in spec:
            return spec
        new_spec = copy.deepcopy(spec)
        new_spec['page_size'] = self.DEFAULT_RESPONSE_LIMIT
        return new_spec

    def _populate_slugs(self):
        spec = {'fields': 'id,slug'}
        for idx, service in enumerate(self.list_services_iter(spec)):
            self._slugs_cache[service['id']] = service['slug']
            if idx % self.DEFAULT_RESPONSE_LIMIT == 0:
                gevent.sleep()
