"""
Simple handmade blackbox connector which uses gevent and requests.
Official version is sync and seemed too heavy (uses only standard library).
This one uses json format only and can call methods:
    * sessionid
"""
import logging

import object_validator
import requests
import socket
from six.moves.urllib import parse as urlparse
from requests.exceptions import RequestException
import six


log = logging.getLogger('swatlib.auth.blackbox')


try:
    import ujson as json
except ImportError:
    try:
        import json
    except ImportError:
        import simplejson as json
json_decode = json.loads


class BlackboxError(Exception):
    pass


# handy field name aliases
FIELD_LOGIN = 'accounts.login.uid'
FIELD_FIO = 'account_info.fio.uid'
FIELD_EMAIL = 'account_info.email.uid'

BLACKBOX_URL = 'http://blackbox.yandex-team.ru/blackbox'
BLACKBOX_AUTH_URL = 'http://passport.yandex-team.ru/passport?retpath={0}'
BLACKBOX_TEST_AUTH_URL = 'http://passport-test.yandex.ru/passport?retpath={0}'
BLACKBOX_TEST_URL = 'http://pass-test.yandex.ru/blackbox'


def smart_str(s):
    if not isinstance(s, six.string_types):
        return str(s)
    elif isinstance(s, six.text_type):
        return s.encode('utf-8', 'replace')
    else:
        return s


def urlencode(params):
    """
    Django copy-paste.
    A version of Python's urllib.urlencode() function that can operate on
    unicode strings. The parameters are first case to UTF-8 encoded strings and
    then encoded as per normal.
    """
    if hasattr(params, 'items'):
        params = params.items()
    return urlparse.urlencode(
        [(smart_str(k),
          isinstance(v, (list, tuple)) and [smart_str(i) for i in v] or smart_str(v))
         for k, v in params])


def join_url_params(url, params):
    # === sanitize ===
    dbfields = params.get('dbfields')
    if dbfields is not None:
        params['dbfields'] = ','.join(dbfields)
    ipv6_prefix = '::ffff:'
    userip = params.get('userip')
    if userip is not None and userip.startswith(ipv6_prefix):
        params['userip'] = userip[len(ipv6_prefix):]
    # === join ===
    params = urlparse.urlencode(params)
    parts = list(urlparse.urlparse(url))
    path = parts[2]
    if not path.startswith('/'):
        path = '/{}'.format(path)
    path = '{0}?{1}'.format(path, params)
    parts[2] = path
    return urlparse.urlunparse(parts)


def _http_get(url, params, timeout=None, headers=None, session=None):
    url = join_url_params(url, params)
    try:
        if session is None:
            session = requests
        resp = session.get(url, headers=headers, timeout=timeout)
    except socket.error as e:
        raise BlackboxError('socket error: {0}'.format(e.strerror))
    except RequestException as e:
        raise BlackboxError('connection error: {0}'.format(str(e)))
    # according to blackbox documentation service always returns status 200
    # any other status code is treated as error
    if resp.status_code != 200:
        raise BlackboxError('bad http status: {0}'.format(resp.status_code))
    content_type = resp.headers.get('content-type', 'text/xml').split(';')[0]
    return resp.text, content_type


def _blackbox_json_call(url, params, timeout=None, headers=None, session=None):
    params['format'] = 'json'
    data, content_type = _http_get(url, params, timeout, headers=headers, session=session)
    if content_type != 'application/json':
        raise BlackboxError("received content type '{0}', "
                            "was waiting for JSON".format(content_type))
    return json_decode(data)


def validate_session_id(sessionid, userip, host,
                        fields=None, timeout=None, url=BLACKBOX_URL, session=None,
                        service_ticket=None):
    """
    Check provided sessionid for validity.
    Futher reading:
        http://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml

    Possible results:
        * socket.error in case of TCP failure
        * gevent.Timeout in case of timeout
        * BlackboxError in case of:
            * bad HTTP code
            * ACL/format error
            * internal error
        * success
    """
    params = {
        'method': 'sessionid',
        'sessionid': sessionid,
        'userip': userip,
        'host': host,
        'dbfields': fields
    }
    if service_ticket is not None:
        headers = {'X-Ya-Service-Ticket': service_ticket}
    else:
        headers = None
    result = _blackbox_json_call(url, params, timeout, session=session, headers=headers)
    error = result['error']
    if error != "OK":
        raise BlackboxError(error)
    status = result['status']['value']
    age = result.get('age', 0)
    valid = (status == 'VALID' or status == 'NEED_RESET')
    redirect = (status == 'NEED_RESET' or (status == 'NOAUTH' and age < 2 * 60 * 60))
    fields = result['dbfields'] if valid else None
    return valid, redirect, fields


# Schema: http://doc.yandex-team.ru/blackbox/reference/method-oauth-response-json.xml
# Very basic validation. In case of error - status can be absent, thus optional.
OAUTH_RESPONSE_SCHEMA = object_validator.DictScheme(
    {
        "status": object_validator.DictScheme(
            {
                "value": object_validator.String(min_length=1),
            },
            ignore_unknown=True,
            optional=True,
        ),
        "error": object_validator.String(min_length=1),
        "oauth": object_validator.DictScheme(
            {
                "client_id": object_validator.String(),
                "scope": object_validator.String(),
            },
            ignore_unknown=True,
            optional=True,
        ),
    },
    ignore_unknown=True,
)


def validate_oauth_token(userip, oauth_token=None, authorization_header=None, fields=None,
                         timeout=None, url=BLACKBOX_URL, session=None, service_ticket=None):
    """
    Wrapper for http://doc.yandex-team.ru/blackbox/reference/method-oauth.xml
    """
    params = {
        'method': 'oauth',
        'userip': userip,
        'dbfields': fields
    }
    headers = {}

    if authorization_header:
        headers['Authorization'] = authorization_header
    elif oauth_token:
        params['oauth_token'] = oauth_token
    else:
        raise RuntimeError('either oauth_token or authorization_header must be specified')

    if service_ticket is not None:
        headers['X-Ya-Service-Ticket'] = service_ticket

    result = _blackbox_json_call(url, params, timeout, headers=headers, session=session)

    try:
        OAUTH_RESPONSE_SCHEMA.validate(result)
    except object_validator.ValidationError as e:
        raise BlackboxError('Invalid blackbox response: {}'.format(e.get_message()))
    if result['error'] != 'OK':
        log.info('Could not authenticate request from "%s". Status: "%s", Error: "%s"', userip,
                 result.get('status'), result.get('error'))
        return False, None, None, None, result['error']
    try:
        status = result['status']['value']
        valid = status == 'VALID'
        fields = result['dbfields'] if valid else None
        client_id = result['oauth']['client_id'] if valid else None
        scope = result['oauth']['scope'] if valid else None
        error = result['error'] if not valid else None
    except KeyError:
        raise BlackboxError('Got invalid blackbox response: {}'.format(result))
    if fields and not fields.get(FIELD_LOGIN):
        log.info('Could not authenticate request from "%s". Status: "%s", Error: "%s"', userip,
                 result.get('status'), result.get('error'))
    return valid, fields, client_id, scope, error


class BlackboxClient(object):
    def __init__(self, url):
        self._url = url.rstrip('/') + '/blackbox/'
        self._session = requests.Session()

    def get_user_ticket(self, service_ticket, user_ip, host, session_id=None, oauth_token=None):
        if session_id is None and oauth_token is None:
            raise ValueError('"session_id" or "oauth_token" argument should be provided for Blackbox authentication')
        if session_id is not None and oauth_token is not None:
            raise ValueError('Only one of "session_id" and "oauth_token" arguments should be provided')
        params = {
            'get_user_ticket': 'yes',
            'userip': user_ip,
            'host': host,
        }
        if session_id is not None:
            params.update({
                'method': 'sessionid',
                'sessionid': session_id,
            })
        elif oauth_token is not None:
            params.update({
                'method': 'oauth',
                'oauth_token': oauth_token,
            })
        else:
            raise ValueError('Unknown authentication method, should be "sessionid" or "oauth_token"')
        data = _blackbox_json_call(
            url=self._url,
            params=params,
            headers={'X-Ya-Service-Ticket': service_ticket},
            session=self._session)
        if 'user_ticket' not in data:
            raise BlackboxError('No user ticket in response: {}'.format(data))
        return data['user_ticket']
