# coding: utf-8
"""
Storable items for authentication and authorization mechanisms.
"""
import hashlib
import logging
import json

from enum import Enum
from mongoengine.document import EmbeddedDocument, Document
from mongoengine.fields import BaseField, StringField, ListField, EmbeddedDocumentField, DictField


log = logging.getLogger(__name__)


class EnumField(BaseField):
    """Enumerator field.

    :type enum: :class:`enum.Enum`
    """

    def __init__(self, enum, *args, **kwargs):
        self.enum = enum
        kwargs['choices'] = [choice for choice in enum]
        super(EnumField, self).__init__(*args, **kwargs)

    def to_python(self, value):
        """Convert a MongoDB-compatible type to a Python type."""
        if value in self.enum:
            return value
        return self.enum[value]

    def to_mongo(self, enum):
        """Convert a Python type to a MongoDB-compatible type."""
        return enum.name


class StaffMembersCache(Document):
    """Persistent cache for staff group members."""
    group_id = StringField(primary_key=True)
    member_logins = ListField(StringField())

    @classmethod
    def set(cls, group_id, member_logins):
        obj = cls(
            group_id=str(group_id),
            member_logins=member_logins
        )
        obj.save()

    @classmethod
    def get(cls, group_id):
        obj = cls.objects(group_id=str(group_id)).first()
        return obj and list(obj.member_logins)


class StaffUserKey(EmbeddedDocument):
    key = StringField()
    fingerprint = StringField()


class StaffGroupIdsCache(Document):
    """Persistent cache for user group ids, keys and official"""
    login = StringField(primary_key=True)
    group_ids = ListField(StringField())
    keys = ListField(EmbeddedDocumentField(StaffUserKey))
    official = DictField()
    signature = StringField()

    @staticmethod
    def calc_signature(group_ids, keys, official):
        h = hashlib.md5()
        for i in sorted(group_ids):
            h.update(i)
        for i in sorted(k['key'] for k in keys):
            h.update(i.encode('utf-8'))
        official_str = json.dumps(official, sort_keys=True)
        h.update(official_str)
        return h.hexdigest()

    def save(self, *args, **kwargs):
        self.signature = self.calc_signature(
            self.group_ids or [],
            self.keys or [],
            self.official or {}
        )
        super(StaffGroupIdsCache, self).save(*args, **kwargs)

    @classmethod
    def set_group_ids(cls, login, group_ids):
        obj = cls.objects(login=login).first() or cls(login=login)
        obj.group_ids = group_ids
        obj.save()

    @classmethod
    def set_user_keys(cls, login, keys):
        obj = cls.objects(login=login).first() or cls(login=login)
        obj.keys = keys
        obj.save()

    @classmethod
    def set_official(cls, login, official):
        obj = cls.objects(login=login).first() or cls(login=login)
        obj.official = official
        obj.save()

    @classmethod
    def get_group_ids(cls, login):
        obj = cls.objects(login=login).first()
        return obj and list(obj.group_ids)

    @classmethod
    def get_user_keys(cls, login):
        obj = cls.objects(login=login).first()
        return obj and list(obj.keys)

    @classmethod
    def get_official(cls, login):
        obj = cls.objects(login=login).first()
        return obj and obj.official


class StaffGroupNamesCache(Document):
    """Persistent cache for user group names."""
    login = StringField(primary_key=True)
    group_ids = ListField(StringField())

    @classmethod
    def set(cls, login, group_ids):
        obj = cls(
            login=login,
            group_ids=group_ids
        )
        obj.save()

    @classmethod
    def get(cls, login):
        obj = cls.objects(login=login).first()
        return obj and list(obj.group_ids)


class AccessRight(Enum):
    Read = 'read'
    Write = 'write'


class Role(EmbeddedDocument):
    module = StringField(min_length=1, max_length=64, required=True)
    access_right = EnumField(AccessRight, required=True)


class User(Document):
    id = StringField(min_length=1, max_length=64, primary_key=True)
    roles = ListField(EmbeddedDocumentField(Role))


class Group(Document):
    id = StringField(min_length=1, max_length=64, primary_key=True)
    roles = ListField(EmbeddedDocumentField(Role))
