# -*- coding: utf-8 -*-
"""
gencfg (http://api.gencfg.yandex-team.ru/) client.
"""
import json
import socket

import inject
from requests.adapters import HTTPAdapter
from requests.exceptions import ConnectionError
from sepelib.util.retry import RetrySleeper, RetryWithTimeout
from sepelib.yandex.alemate import ContainerResourceLimits

from infra.swatlib.metrics import InstrumentedSession


class GencfgInstance(object):
    __slots__ = ['hostname', 'port', 'tags', 'power', 'dc', 'location', 'domain', 'ipv4addr', 'ipv6addr', 'shard_name',
                 'limits', 'queue', 'rack']

    def __init__(self, hostname, port,
                 tags=None, power=None, dc=None, location=None, domain=None,
                 ipv4addr=None, ipv6addr=None, shard_name=None, limits=None,
                 queue=None, rack=None, **_):
        self.hostname = hostname
        self.port = port
        self.tags = tags
        self.power = power
        self.dc = dc
        self.location = location
        self.domain = domain
        self.ipv4addr = ipv4addr
        self.ipv6addr = ipv6addr
        self.shard_name = shard_name
        self.limits = limits or ContainerResourceLimits()
        self.queue = queue
        self.rack = rack

    @classmethod
    def from_gencfg_response(cls, params):
        limits = ContainerResourceLimits.from_gencfg_response(params.pop('porto_limits', {}))
        return cls(limits=limits, **params)

    def __eq__(self, other):
        return (
            self.hostname == other.hostname and
            self.port == other.port and
            self.tags == other.tags and
            self.power == other.power and
            self.dc == other.dc and
            self.location == other.location and
            self.domain == other.domain and
            self.ipv4addr == other.ipv4addr and
            self.ipv6addr == other.ipv6addr and
            self.shard_name == other.shard_name
        )

    def __repr__(self):
        kwargs_str = []
        for attr in self.__slots__:
            kwargs_str.append('{}={!r}'.format(attr, getattr(self, attr, None)))
        return 'GencfgInstance({})'.format(kwargs_str)

    def to_dict(self):
        return {
            "hostname": self.hostname,
            "port": self.port,
            "tags": self.tags,
            "power": self.power,
            "dc": self.dc,
            "location": self.location,
            "domain": self.domain,
            "ipv4addr": self.ipv4addr,
            "ipv6addr": self.ipv6addr,
            "shard_name": self.shard_name,
            "limits": self.limits.to_dict(),
            "queue": self.queue,
            "rack": self.rack,
        }

    @classmethod
    def from_dict(cls, params):
        limits = ContainerResourceLimits.from_dict(params.pop('limits', {}))
        return cls(limits=limits, **params)


class OrtogonalTagLocation(object):
    __slots__ = ['geo', 'ctype', 'itype', 'prj', 'metaprj']

    def __init__(self, geo, ctype, itype, prj, metaprj):
        """
        :param str geo: msk|sas|man or ALL
        :param str ctype: prod|prestable ...
        :param str itype: golovan itype
        :param list[str] prj: [ "sgcluster-nanny" ]
        :param str metaprj:  "internal"
        """
        self.geo = geo
        self.ctype = ctype
        self.itype = itype
        self.prj = prj
        self.metaprj = metaprj


class AbcService(object):
    __slots__ = ['id', 'role_policy', 'role_ids']

    def __init__(self, service_id, role_policy, role_ids):
        """
        :type service_id: str
        :type role_policy: str
        :type role_ids: set[str]
        """
        self.id = service_id
        self.role_policy = role_policy
        self.role_ids = role_ids

    def allows_roles(self, roles):
        """
        :type roles: list[str]
        :rtype bool
        """
        if self.role_policy == 'ALL':
            return True
        if self.role_ids.intersection(roles):
            return True
        return False


class GencfgGroupCard(object):
    def __init__(self, group, owners, owners_abc_services, tag_location, extra_tags, network_macro, volumes,
                 virtual_services=None):
        """
        :param str group: MSK_SG_NANNY
        :param list[str] owners: [ "nekto0n" ]
        :param dict[str, AbcService] owners_abc_services:
        :param OrtogonalTagLocation tag_location:
        :param list[str] extra_tags:
        :type network_macro: unicode
        :type volumes: list[dict]
        """
        self.group = group
        self.owners = owners
        self.owners_abc_services = owners_abc_services
        self.tag_location = tag_location
        self.extra_tags = extra_tags
        self.network_macro = network_macro
        self.volumes = volumes
        self.virtual_services = virtual_services or []

    @classmethod
    def from_gencfg_response(cls, d):
        group = d['name']
        tags = d.pop('tags')
        tags['geo'] = group.split('_', 1)[0].lower()
        extra_tags = tags.pop('itag', [])

        if not isinstance(tags['prj'], list):
            raise ValueError("Group attribute 'prj' is not list")

        tag_location = OrtogonalTagLocation(
            geo=tags['geo'],
            itype=tags['itype'],
            ctype=tags['ctype'],
            prj=tags['prj'],
            metaprj=tags['metaprj'],
        )

        owners_abc_services = {}
        for s in d.get('owners_abc_roles', []):
            s_id = str(s['service_id'])
            owners_abc_services[s_id] = AbcService(service_id=s_id,
                                                   role_policy=s['role_policy'],
                                                   role_ids={str(r_id) for r_id in s['role_ids']})

        props = d.get('properties', {})
        reqs = d.get('reqs', {})

        return cls(
            group=group,
            owners=d['owners'],
            owners_abc_services=owners_abc_services,
            tag_location=tag_location,
            extra_tags=extra_tags,
            network_macro=props.get('hbf_parent_macros', ''),
            volumes=reqs.get('volumes', []),
            virtual_services=props.get('mtn', {}).get('tunnels', {}).get('hbf_slb_name', [])
        )


class IGencfgClient(object):
    """
    Interface to be used in dependency injection.
    """

    def list_group_instances(self, group, gencfg_version=None):
        pass

    def list_group_instances_dicts(self, group, gencfg_version=None):
        """
        :type group: str | unicode
        :type gencfg_version: str | unicode

        :rtype: list[dict]
        """
        raise NotImplementedError

    def get_group_card(self, group, gencfg_version=None):
        """
        Get gencfg instance group card

        :param str group: gencfg group
        :param str gencfg_version: topology tag

        $ GET http://api.gencfg.yandex-team.ru/trunk/groups/MSK_SG_NANNY/card | json_pp
        {
           "owners" : [
              "nekto0n"
           ],
           "debug_info" : {
              "backend_host" : "man1-7286.search.yandex.net",
              "backend_port" : 7300,
              "branch" : "trunk",
              "backend_type" : "trunk"
           },
           "name" : "MSK_SG_NANNY",
           "tags" : {
              "itag" : [],
              "prj" : [
                 "sgcluster-nanny"
              ],
              "ctype" : "prod",
              "itype" : "nanny",
              "metaprj" : "internal"
           }
        }
        :rtype: GencfgGroupCard
        """
        raise NotImplementedError

    @classmethod
    def instance(cls):
        """
        :rtype: IGencfgClient
        """
        return inject.instance(cls)


class GencfgClient(IGencfgClient):
    _DEFAULT_BASE_URL = 'http://api.gencfg.yandex-team.ru/'
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3
    _RETRY_ERRORS = (ConnectionError, socket.error)

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, url=None, req_timeout=None, attempts=None):
        self._base_url = url or self._DEFAULT_BASE_URL
        self._req_timeout = req_timeout if req_timeout is not None else self._DEFAULT_REQ_TIMEOUT
        if self._req_timeout < 1.0:
            raise ValueError('req_timeout must be >= 1.0, got {}'.format(self._req_timeout))
        self._attempts = int(attempts if attempts is not None else self._DEFAULT_ATTEMPTS)
        if self._attempts < 1:
            raise ValueError('attempts must be >= 1, got {}'.format(self._attempts))
        self._session = InstrumentedSession('gencfg')
        self._session.mount(self._base_url, self._make_http_adapter())

    def _make_http_adapter(self):
        """
        Make custom http adapter.
        We need maximum pooled connection to be less than attempts.
        Thus we work around situation when we have actually dead connections at hand.
        We achieve this by exhausting connection pool if connections are dead.
        E.g. they can be dropped by ipvs or/and http balancer due to inactivity.

        Proper ways to fix can be:
          * Use SO_KEEPALIVE to minimize having broken connections in the pool.
          * Throw connections coming from pool if they were idle for too long.
        But these ways are hard to implement, there aren't enough hooks in requests library.
        This fix should do it. See SEPE-8393 for details.
        """
        max_conn = max(1, self._attempts // 2)  # We need at least 1 connection
        return HTTPAdapter(pool_connections=max_conn, pool_maxsize=max_conn)

    def _call_remote(self, func, *args, **kwargs):
        r = RetryWithTimeout(self._req_timeout,
                             RetrySleeper(max_tries=self._attempts, max_delay=30))
        return r(func, *args, **kwargs)

    def _request_group_instances(self, group, gencfg_version=None):
        """
        Requests gencfg group instances in gencfg and returns response

        :type group: str | unicode
        :type gencfg_version: str | unicode
        :rtype: requests.models.Response
        """
        gencfg_version = gencfg_version or 'trunk'
        url = '{base_url}{version}/searcherlookup/groups/{group}/instances'.format(
            base_url=self._base_url,
            version=gencfg_version,
            group=group)
        response = self._call_remote(self._session.get, url, timeout=self._req_timeout)
        response.raise_for_status()
        return response

    def list_group_instances_json(self, group, gencfg_version=None):
        """
        List all instances having specified :param group: tag, returns raw gencfg API response.

        :param gencfg_version: string representing gencfg release in format "tags/stable-82-r2" or "trunk"
        :type group: str | unicode
        :type gencfg_version: str | unicode

        :rtype: str
        """
        response = self._request_group_instances(group, gencfg_version)
        return response.content

    def list_group_instances_dicts(self, group, gencfg_version=None):
        """
        List all instances having specified :param group: tag, returns parsed gencfg API response.

        :param gencfg_version: string representing gencfg release in format "tags/stable-82-r2" or "trunk"
        :type group: str | unicode
        :type gencfg_version: str | unicode

        :rtype: list[dict]
        """
        response = self._request_group_instances(group, gencfg_version)
        return response.json()['instances']

    def list_group_instances(self, group, gencfg_version=None):
        """
        List all instances having specified :param group: tag.

        :param gencfg_version: string representing gencfg release in format "tags/stable-82-r2" or "trunk"
        :type group: str | unicode
        :type gencfg_version: str | unicode

        :rtype: list[GencfgInstance]
        """
        response = self._request_group_instances(group=group, gencfg_version=gencfg_version)
        js = response.json()
        if isinstance(js, list):  # Old format
            instances = js
        else:
            instances = js['instances']  # Modern format
        return [GencfgInstance.from_gencfg_response(i) for i in instances]

    def list_intlookup_instances(self, intlookup_name, instance_type=None, gencfg_version=None):
        """
        List intlookup instances.

        :param str intlookup_name: existing intlookup name, e.g. intlookup-msk-imgs.py
        :param str instance_type: base or int for specific type, None for all instances
        :param str gencfg_version: str specific tag, None for trunk
        """
        gencfg_version = gencfg_version or 'trunk'
        url = '{base_url}{version}/intlookups/{intlookup}/instances?instance_type={instance_type}'.format(
            base_url=self._base_url,
            version=gencfg_version,
            intlookup=intlookup_name,
            instance_type=instance_type or '')
        response = self._call_remote(self._session.get, url, timeout=self._req_timeout)
        response.raise_for_status()
        return [GencfgInstance(**i) for i in response.json()['instances']]

    def describe_gencfg_version(self, gencfg_version=None):
        """Get gencfg version description"""
        gencfg_version = gencfg_version or 'trunk'
        url = '{base_url}{version}/description'.format(
            base_url=self._base_url,
            version=gencfg_version
        )
        response = self._call_remote(self._session.get, url, timeout=self._req_timeout)
        response.raise_for_status()
        return response.json()['description']

    def get_hosts_info(self, hosts_fqdn, gencfg_version=None):
        """
        Get info about host.

        :type hosts_fqdn: list[str | unicode]
        :type gencfg_version: str | unicode
        :rtype dict:
        """

        # SEPE-13460
        # мы сейчас вынуждены ходить за описаниями хостов делая по запросу на каждый хост
        # (т.к. пользователи нам дают для части хостов левые имена, о которых gencfg ничего не знает).
        # а если gencfg не знает хотя бы один из требуемых хостов, она возвращает 500.
        # А можно в случае, если есть неизвестные хосты, возвращать 200 и инфу хотя бы по известным хостам?

        gencfg_version = gencfg_version or 'trunk'
        url = '{base_url}{version}/hosts_data'.format(base_url=self._base_url,
                                                      version=gencfg_version)
        response = self._call_remote(self._session.post, url, timeout=self._req_timeout,
                                     data=json.dumps({'hosts': hosts_fqdn}),
                                     headers={'Content-Type': 'application/json'})
        response.raise_for_status()
        return response.json()['hosts_data']

    def list_hosts(self, gencfg_version=None):
        """
        Returns all known by gencfg hosts

        Example GenCfg reply:

        {
           "host_names" : [
              "100-43-90-6.yandex.com",
              "100-43-90-7.yandex.com",
               ...
              "zworker-s1.yandex.net",
              "zworker-test.yandex.net"
           ],
           "debug_info" : {
              "branch" : "trunk",
              "backend_host" : "ws39-778.yandex.ru",
              "backend_port" : 7200,
              "backend_type" : "trunk"
           },
           "std header" : {
              "current_branch" : "trunk",
              "auth_user" : "alximik",
              "ids_alphabet" : "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_."
           }
        }

        :type gencfg_version: str | unicode
        :rtype list[str|unicode]:
        """

        gencfg_version = gencfg_version or 'trunk'
        url = '{base_url}{version}/hosts'.format(base_url=self._base_url, version=gencfg_version)
        response = self._call_remote(self._session.get, url, timeout=self._req_timeout)

        response.raise_for_status()

        return response.json()['host_names']

    def get_group_card(self, group, gencfg_version=None):
        """
        Get gencfg instance group card

        :param str group: gencfg group
        :param str gencfg_version: topology tag

        $ GET http://api.gencfg.yandex-team.ru/trunk/groups/MSK_SG_NANNY/card | json_pp
        {
           "owners" : [
              "nekto0n"
           ],
           "debug_info" : {
              "backend_host" : "man1-7286.search.yandex.net",
              "backend_port" : 7300,
              "branch" : "trunk",
              "backend_type" : "trunk"
           },
           "name" : "MSK_SG_NANNY",
           "tags" : {
              "itag" : [],
              "prj" : [
                 "sgcluster-nanny"
              ],
              "ctype" : "prod",
              "itype" : "nanny",
              "metaprj" : "internal"
           }
        }
        :rtype: GencfgGroupCard
        """

        gencfg_version = gencfg_version or 'trunk'
        url = '{base_url}{version}/groups/{group}/card'.format(
            base_url=self._base_url,
            version=gencfg_version,
            group=group)
        response = self._call_remote(self._session.get, url, timeout=self._req_timeout)
        response.raise_for_status()
        js = response.json()
        return GencfgGroupCard.from_gencfg_response(js)
