# coding: utf-8
import collections
from six.moves.urllib import parse as urlparse

import requests
from flask import redirect


class IOAuth(object):
    pass


class OAuthException(Exception):
    pass


Token = collections.namedtuple('Token', ['content', 'expires_in'])


class TokenChecker(object):
    def __init__(self, client_id, scopes):
        self.client_id = client_id
        self.scopes = frozenset(scopes)

    def check(self, client_id, scopes_str=''):
        if self.client_id and self.client_id == client_id:
            return True
        # From blackbox it comes as ""direct:api-1month metrika:write"
        scopes = scopes_str.split(' ')
        for scope in scopes:
            if scope in self.scopes:
                return True
        raise OAuthException('Token does not have required client_id or scope')


class OAuth(IOAuth):
    DEFAULT_URL = 'https://oauth.yandex-team.ru/'

    @classmethod
    def from_config(cls, d):
        return cls(url=d.get('url'),
                   client_id=d['client_id'],
                   client_secret=d['client_secret'],
                   scopes=d.get('scopes'))

    def __init__(self, client_id, client_secret, url=DEFAULT_URL, scopes=None):
        self.url = url
        self.client_secret = client_secret
        self.client_id = client_id
        if scopes is None:
            scopes = []
        if scopes and not isinstance(scopes, list):
            scopes = scopes.split(' ')
        self.scopes = scopes
        self.checker = TokenChecker(self.client_id, self.scopes)

    def redirect_to_authorize_page(self):
        url = urlparse.urljoin(self.url, '/authorize') + '?response_type=code&client_id={}'.format(self.client_id)
        return redirect(url)

    def get_token_by_code(self, code):
        """Deprecated. Use get_token_by_authorization_code instead."""
        return self.get_token_by_authorization_code(code).content

    def get_token_by_authorization_code(self, code):
        """Returns :class:`Token` by authorization code
        (see https://doc.yandex-team.ru/oauth/dg-internal/reference/auto-code-client.xml).
        """
        url = urlparse.urljoin(self.url, '/token')
        data = {
            'grant_type': 'authorization_code',
            'code': code,
            'client_id': self.client_id,
            'client_secret': self.client_secret,
        }
        headers = {
            'Host': urlparse.urlparse(self.url).hostname,
            'Content-type': 'application/x-www-form-urlencoded',
        }
        resp = requests.post(url, data=data, headers=headers).json()
        token = resp.get('access_token')
        if not token:
            raise OAuthException('{}: {}'.format(resp.get('error'), resp.get('error_description')))
        return Token(content=token, expires_in=resp.get('expires_in'))

    def get_token_by_x_token(self, x_token, user_ip):
        """Returns :class:`Token` by X-Token
        (see https://doc.yandex-team.ru/oauth/dg-internal/internal-tokens/xtoken.xml).
        """
        url = urlparse.urljoin(self.url, '/token')
        data = {
            'grant_type': 'x-token',
            'access_token': x_token,
            'client_id': self.client_id,
            'client_secret': self.client_secret,
        }
        headers = {
            'Host': urlparse.urlparse(self.url).hostname,
            'Content-type': 'application/x-www-form-urlencoded',
            'X-Forwarded-For': user_ip,
        }
        resp = requests.post(url, data=data, headers=headers).json()
        token = resp.get('access_token')
        if not token:
            raise OAuthException('{}: {}'.format(resp.get('error'), resp.get('error_description')))
        return Token(content=token, expires_in=resp.get('expires_in'))

    def get_user_login_by_authorization_header(self, passport_client, authorization_header, user_ip):
        r = passport_client.check_oauth_token(
            authorization_header=authorization_header,
            user_ip=user_ip
        )
        if r.error:
            raise OAuthException('blackbox error: {}'.format(r))
        self.checker.check(r.client_id, r.scope)
        return r.login
