"""
Passport client which tries to reuse connections.
"""
from collections import namedtuple
from logging import getLogger

from infra.swatlib.metrics import InstrumentedSession
from infra.swatlib.auth.blackbox import (validate_session_id,
                                         validate_oauth_token,
                                         FIELD_LOGIN,
                                         BLACKBOX_URL,
                                         BLACKBOX_AUTH_URL)


log = getLogger(__name__)

PassportCheckResult = namedtuple('PassportCheckResult', ['login', 'redirect_url'])


class OAuthCheckResult(object):
    __slots__ = ['login', 'client_id', 'scope', 'error']

    def __init__(self, login, client_id, scope=None, error=None):
        self.login = login
        self.client_id = client_id
        self.scope = scope
        self.error = error

    def __str__(self):
        return "OAuthCheckResult(login='{}',client_id='{}',scope='{}',error='{}')".format(
            self.login, self.client_id, self.scope, self.error,
        )


class IPassportClient(object):
    """
    Interface to be used in dependency injection.
    """
    pass


class PassportClient(IPassportClient):
    _DEFAULT_REQ_TIMEOUT = 10

    @classmethod
    def from_config(cls, d):
        return cls(blackbox_url=d.get('blackbox_url'),
                   blackbox_auth_url=d.get('blackbox_auth_url'),
                   req_timeout=d.get('req_timeout'),
                   force_redirect_to_https=d.get('force_redirect_to_https'))

    @staticmethod
    def _get_session():
        # We do not have retries at the moment, so in order to avoid
        # having keep alive connections being dropped on IPVS while idle,
        # let us create new session every time.
        return InstrumentedSession('passport')

    def __init__(self, blackbox_url=BLACKBOX_URL,
                 blackbox_auth_url=BLACKBOX_AUTH_URL,
                 blackbox_tvm_id=None,
                 req_timeout=None,
                 force_redirect_to_https=False):
        self._blackbox_url = blackbox_url
        self._blackbox_auth_url = blackbox_auth_url
        self._blackbox_tvm_id = blackbox_tvm_id
        self._req_timeout = req_timeout if req_timeout is not None else self._DEFAULT_REQ_TIMEOUT
        self._force_redirect_to_https = force_redirect_to_https
        if self._req_timeout < 1.0:
            raise ValueError('req_timeout must be >= 1.0, got {}'.format(self._req_timeout))

    def get_service_ticket_to_blackbox(self):
        """
        Get TVM service ticket to authenticate in Blackbox.
        """
        # In PassportClient we don't use TVM service ticket to authenticate in
        # Blackbox.
        return None

    def check_passport_cookie(self, cookies, host, user_ip, request_url):
        """
        Check passport cookie.
        :param cookies: dict-like object containing cookies,
                        e.g. :attr:`flask.request.cookies`)
        :param host: requested host, e.g. :attr:`flask.request.host`
        :param user_ip: user ip, e.g. the first element of :attr:`flask.request.access_route`
        :param request_url: URL to redirect a user to after successful authentication
        :rtype: PassportCheckResult
        """
        session_id = cookies.get('Session_id')
        if not session_id:
            if self._force_redirect_to_https and request_url.startswith('http://'):
                request_url = request_url.replace('http://', 'https://', 1)
            return PassportCheckResult(login=None, redirect_url=self._blackbox_auth_url.format(request_url))
        if ':' in host:
            host = host.split(':')[0]

        with self._get_session() as s:
            valid, need_redirect, dbfields = validate_session_id(session_id,
                                                                 user_ip, host,
                                                                 [FIELD_LOGIN],
                                                                 timeout=self._req_timeout,
                                                                 url=self._blackbox_url,
                                                                 session=s,
                                                                 service_ticket=self.get_service_ticket_to_blackbox())
        if not valid or need_redirect:
            return PassportCheckResult(login=None, redirect_url=self._blackbox_auth_url.format(request_url))
        if need_redirect:
            redirect_url = self._blackbox_auth_url.format(request_url)
        else:
            redirect_url = None
        # put login to lowercase
        # otherwise users like 'nARN' can get in trouble
        login = dbfields[FIELD_LOGIN].lower()
        log.info("authenticated via blackbox: login='{0}' ip='{1}'".format(login, user_ip))
        return PassportCheckResult(login=login, redirect_url=redirect_url)

    def check_oauth_token(self, user_ip, oauth_token=None, authorization_header=None):
        """
        Check OAuth token.
        :param oauth_token: OAuth token or entire content of Authorization header
        :param user_ip: user ip, e.g. the first element of :attr:`flask.request.access_route`
        :rtype: OAuthCheckResult
        """
        assert (oauth_token or authorization_header) and not (oauth_token and authorization_header)
        with self._get_session() as s:
            valid, fields, client_id, scope, error = validate_oauth_token(
                oauth_token=oauth_token,
                authorization_header=authorization_header,
                userip=user_ip,
                fields=[FIELD_LOGIN],
                timeout=self._req_timeout,
                url=self._blackbox_url,
                session=s,
                service_ticket=self.get_service_ticket_to_blackbox())
        if valid:
            return OAuthCheckResult(login=fields[FIELD_LOGIN],
                                    client_id=client_id,
                                    scope=scope,
                                    error=None)
        else:
            return OAuthCheckResult(login=None, client_id=client_id, error=error)

    # to be consistent with other services
    def start(self):
        pass

    def stop(self):
        pass


class TvmPassportClient(PassportClient):

    def __init__(self, blackbox_tvm_id, tvm_client,
                 blackbox_url=BLACKBOX_URL,
                 blackbox_auth_url=BLACKBOX_AUTH_URL,
                 req_timeout=None,
                 force_redirect_to_https=False):
        super(TvmPassportClient, self).__init__(blackbox_url=blackbox_url,
                                                blackbox_auth_url=blackbox_auth_url,
                                                req_timeout=req_timeout,
                                                force_redirect_to_https=force_redirect_to_https)
        self._tvm_client = tvm_client
        self._blackbox_tvm_id = blackbox_tvm_id

    def get_service_ticket_to_blackbox(self):
        return self._tvm_client.ticket_to(self._blackbox_tvm_id)

    @classmethod
    def from_config(cls, d, tvm_client):
        return cls(blackbox_tvm_id=d.get('tvm_client_id'),
                   blackbox_url=d.get('blackbox_url'),
                   blackbox_auth_url=d.get('blackbox_auth_url'),
                   req_timeout=d.get('req_timeout'),
                   force_redirect_to_https=d.get('force_redirect_to_https'),
                   tvm_client=tvm_client)
