# coding: utf-8
"""
Port of the sepelib.flask.auth.staff with additional methods
"""
import six
import logging
import cachetools
import requests

from infra.swatlib.httpclient import HttpClient, HttpClientException
from infra.swatlib.auth.dao import StaffGroupIdsCache, StaffGroupNamesCache, StaffMembersCache
from infra.swatlib.auth.tvm import TvmClient
from sepelib.core import config


class StaffError(HttpClientException):
    pass


class CachingStaffError(Exception):
    def __init__(self, error, cached):
        super(CachingStaffError, self).__init__(error)
        self.error = error
        self.cached = cached


class IStaffClient(object):
    """
    Interface to be used in dependency injection.
    """


class ICachingStaffClient(object):
    """
    Interface to be used in dependency injection.
    """


class StaffClient(object):
    STAFF_API_URL = 'https://staff-api.yandex-team.ru/v3/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_RESPONSE_LIMIT = 200
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(staff_url=d.get('api_url'),
                   oauth_token=d.get('oauth_token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   limit=d.get('response_entities_limit'),
                   max_retries=d.get('max_retries'),
                   connection_timeout=d.get('connection_timeout'),
                   tvm_service_id=d.get('tvm_service_id')
                   )

    def __init__(self, staff_url=None, oauth_token=None, req_timeout=None, verify_ssl=None,
                 limit=None, max_retries=None, connection_timeout=None, tvm_service_id=None):
        staff_url = staff_url or self.STAFF_API_URL
        token = oauth_token
        self._tvm_service_id = tvm_service_id
        if token and self._tvm_service_id:
            raise ValueError("Params 'token' and 'tvm_service_id' can't be set together")
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = HttpClient(client_name='staff',
                                  exc_cls=StaffError,
                                  base_url=staff_url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries,
                                  connection_timeout=connection_timeout)
        self._limit = self.DEFAULT_RESPONSE_LIMIT if limit is None else limit
        if self._tvm_service_id:
            self._tvm_client = TvmClient(client_id=config.get_value('tvm.client_id'),
                                         secret=config.get_value('tvm.secret'),
                                         api_url=config.get_value('tvm.url'))

    def _list(self, rel_url, spec, fields, one, limit):
        params = dict((key, value if not isinstance(value, list) else ','.join(map(str, value)))
                      for key, value in six.iteritems(spec))
        params['_fields'] = ','.join(fields)
        if one:
            params['_one'] = '1'
        if limit is None:
            params['_limit'] = self._limit
        else:
            params['_limit'] = limit
        headers = {}
        if self._tvm_service_id:
            tvm_ticket = self._tvm_client.ticket_to(self._tvm_service_id)
            headers = {"X-Ya-Service-Ticket": tvm_ticket}
        return self._client.get(rel_url, params=params, headers=headers)

    def list_persons(self, spec, fields=('id', 'name', 'url'), one=False, limit=None):
        """
        https://staff-api.yandex-team.ru/v3/persons?_doc=1
        """
        return self._list('persons', spec, fields, one, limit)

    def list_groups(self, spec, fields=('id', 'name'), one=False, limit=None):
        """
        https://staff-api.yandex-team.ru/v3/groups?_doc=1
        """
        return self._list('groups', spec, fields, one, limit)

    def list_groupmembership(self, spec, fields=('person',), one=False, limit=None):
        """
        https://staff-api.yandex-team.ru/v3/groupmembership?_doc=1
        """
        return self._list('groupmembership', spec, fields, one, limit)


class CachingStaffClient(object):
    _USE_CACHE_ERRORS = (requests.HTTPError, requests.ConnectionError, requests.Timeout, StaffError)

    def __init__(self, staff_client):
        self._staff_client = staff_client
        self._log = logging.getLogger(__name__)
        self._official_cache = cachetools.LRUCache(maxsize=20000)

    @classmethod
    def from_config(cls, d):
        return cls(staff_client=StaffClient.from_config(d))

    @staticmethod
    def get_cached_group_ids(login):
        return StaffGroupIdsCache.get_group_ids(login)

    @staticmethod
    def set_cached_group_ids(login, group_ids):
        StaffGroupIdsCache.set_group_ids(login, group_ids)

    def get_group_ids_from_staff(self, login):
        group_ids = get_group_ids(self._staff_client, login)
        self.set_cached_group_ids(login, group_ids)
        return group_ids

    def get_group_ids(self, login):
        try:
            group_ids = get_group_ids(self._staff_client, login)
        except self._USE_CACHE_ERRORS as e:
            cached_group_ids = self.get_cached_group_ids(login)
            raise CachingStaffError(error=e, cached=cached_group_ids)
        else:
            self.set_cached_group_ids(login, group_ids)
            return group_ids

    def get_user_keys(self, login):
        fields = ('keys.key', 'keys.fingerprint')
        try:
            p = self._staff_client.list_persons({'login': login}, fields=fields, one=True)
        except self._USE_CACHE_ERRORS as e:
            keys = StaffGroupIdsCache.get_user_keys(login)
            raise CachingStaffError(error=e, cached=keys)
        else:
            keys = p['keys']
            # StaffGroupIdsCache.set_user_keys(login, keys)
            return keys

    def get_user_official(self, login):
        # local cache
        official = self._official_cache.get(login)
        if official:
            return official, 'local'

        # db cache
        official = StaffGroupIdsCache.get_official(login)
        if official:
            self._official_cache[login] = official
            return self._official_cache[login], 'db'

        # try get from staff
        fields = ('login', 'official')
        try:
            p = self._staff_client.list_persons({'login': login}, fields=fields, one=True)
        except self._USE_CACHE_ERRORS as e:
            self._log.warn('return user official from staff error: %s', e)
            raise CachingStaffError(error=e, cached=None)
        else:
            official = p['official']
            StaffGroupIdsCache.set_official(login, official)
            self._official_cache[login] = official
            return self._official_cache[login], 'staff'

    def get_users_dismissed_statuses(self, logins):
        """

        :type logins: list[six.text_type]
        :rtype: dict[six.text_type, bool]
        """
        fields = ('login', 'official.is_dismissed')
        try:
            p = self._staff_client.list_persons({'login': ",".join(logins)}, fields=fields, limit=len(logins))
        except self._USE_CACHE_ERRORS:
            return {}
        else:
            return {user['login']: user['official']['is_dismissed'] for user in p['result']}

    def get_group_names(self, login):
        try:
            group_names = get_group_names(self._staff_client, login)
        except self._USE_CACHE_ERRORS as e:
            cached_group_names = StaffGroupNamesCache.get(login)
            raise CachingStaffError(error=e, cached=cached_group_names)
        else:
            StaffGroupNamesCache.set(login, group_names)
            return group_names

    def get_group_member_logins(self, group_id):
        try:
            member_logins = get_group_member_logins(self._staff_client, group_id)
        except self._USE_CACHE_ERRORS as e:
            member_logins = StaffMembersCache.get(group_id)
            raise CachingStaffError(error=e, cached=member_logins)
        else:
            StaffMembersCache.set(group_id, member_logins)
            return member_logins


def get_group_member_logins(staff_client, group_id, skip_robots=False):
    """
    Returns a list of group members.
    :type staff_client: StaffClient
    :type group_id: unicode
    :rtype: list[unicode]
    """
    member_filter = {
        'group.id': group_id,
        'person.official.is_dismissed': False
    }
    if skip_robots:
        member_filter['person.official.is_robot'] = False

    staff_resp = staff_client.list_groupmembership(member_filter, fields=('person.login',))
    logins = []
    for item in staff_resp['result']:
        logins.append(item['person']['login'])
    return logins


def get_group_ids(staff_client, login):
    return get_group_details(staff_client, login, 'id')


def get_group_names(staff_client, login):
    # 'name' is actually a description, so use 'url' instead
    return get_group_details(staff_client, login, 'url')


def get_group_details(staff_client, login, field):
    """
    Returns a list of a single group field for groups in which the provided login is a member.
    :type staff_client: StaffClient
    :type login: unicode
    :type field: unicode
    :rtype: list[unicode]
    """
    spec = {
        'person.login': login,
        'group.type': 'department,service,servicerole',
        'group.is_deleted': 'false',
    }
    staff_resp = staff_client.list_groupmembership(spec=spec, fields=('group',))
    groups = staff_resp['result']
    rv = []
    for item in groups:
        # Every item in result list is a dictionary with "group" key
        group = item['group']
        t = group['type']
        if t == 'department':
            rv.append(six.text_type(group[field]))
            for g in group['ancestors']:
                rv.append(six.text_type(g[field]))
        elif t == 'service':
            rv.append(six.text_type(group[field]))
        elif t == 'servicerole':
            rv.append(six.text_type(group[field]))
    return rv
