# Based on https://github.yandex-team.ru/e-sidorov/grpctvm/blob/master/grpc/tvm.py
from __future__ import unicode_literals

import ticket_parser2 as tp2
import time
from datetime import datetime, timedelta
from infra.swatlib.httpclient import HttpClient, HttpClientException
from ticket_parser2.api.v1 import ServiceContext, exceptions


class ITvmClient(object):
    """
    Interface to be used in dependency injection.
    """
    pass


class TvmClient(object):
    def __init__(self, client_id, secret, api_url, base_client=None):
        self._client_id = client_id
        self._secret = secret
        self._tickets_cache = dict()
        self._client = HttpClient(client_name='tvm',
                                  exc_cls=HttpClientException,
                                  base_url=api_url)
        self._keys_received = None
        self._tvm_keys = None
        self._service_context = None
        self._base_client = base_client

    def _get_tvm_keys(self):
        keys_resp = self._client.get(rel_url='/2/keys?lib_version={version}'.format(version=tp2.__version__),
                                     return_json=False)
        if keys_resp.status_code != 200:
            raise exceptions.TvmException("Couldn't fetch public TVM keys")
        self._tvm_keys = keys_resp.content
        self._keys_received = datetime.now()

    def _refresh_tvm_keys(self):
        if (datetime.now() - self._keys_received) >= timedelta(hours=24):
            self._get_tvm_keys()
            self._service_context.reset_keys(self._tvm_keys)

    # if you need more dst, you have to request tickets in one batch request
    # take a look at API documentation at https://wiki.yandex-team.ru/passport/tvm2/api/
    def ticket_to(self, dst):
        if self._tvm_keys is None:
            self._get_tvm_keys()
            self._service_context = ServiceContext(int(self._client_id), self._secret, self._tvm_keys)
        dst_str = str(dst)
        self._refresh_tvm_keys()
        ticket = self._tickets_cache.get(dst_str, None)
        if ticket is not None:
            try:
                self._service_context.check(ticket)
                return ticket
            except exceptions.TicketParsingException:
                # stale ticket in cache - will be updated
                pass

        ts = int(time.time())
        data = self._client.post(rel_url='/2/ticket/',
                                 data={
                                     'grant_type': 'client_credentials',
                                     'src': int(self._client_id),
                                     'dst': int(dst),
                                     'ts': ts,
                                     'sign': self._service_context.sign(ts, dst_str, '')
                                 })
        if dst_str not in data:
            raise exceptions.TvmException("Failed to get TVM service ticket to YaVault: {}".format(data))
        ticket = data[dst_str]['ticket']
        self._tickets_cache[dst_str] = ticket
        return ticket

    def check_service_ticket(self, ticket):
        if self._base_client is None:
            raise exceptions.TvmException('No base client provided')
        return self._base_client.check_service_ticket(ticket)
