import logging
import sys
import traceback

import click
from sepelib.core import config
from sepelib.util import log as logutil

log = logging.getLogger('main')

SETUP_COMPLETED = False
DEFAULT_CONFIG_PATH = './cfg_default.yml'


class Group(click.Group):
    def add_command(self, cmd, name=None):
        _invoke = cmd.invoke

        def invoke(ctx):
            run_setup(cfg=click.get_current_context().params.get('config', None))
            _invoke(ctx)

        cmd.invoke = invoke
        super(Group, self).add_command(cmd, name)


def group(name=None, **attrs):
    """Creates a new :class:`Group` with a function as callback.  This
    works otherwise the same as :func:`command` just that the `cls`
    parameter is set to :class:`SwatGroup`.
    """
    attrs.setdefault('cls', Group)
    return click.command(name, **attrs)


def setup_exception_logging():
    """
    Sets up handler, which additionally logs errors in "main" handler to stderr.
    """
    h = logging.StreamHandler(stream=sys.stderr)
    h.setLevel(logging.ERROR)
    log.addHandler(h)


def setup_logging(logcfg, console_handler, console=False):
    """
    Setup service logs according to config
    """
    # disable requests module logging
    logging.getLogger('requests').setLevel(logging.DEBUG)

    if not sys.stdout.isatty() and not console:
        handler = logutil.create_handler_from_config(logcfg)
        logger = logutil.setup_logging_to_file(handler, redirect_stderr=False, redirect_stdout=False)
        setup_exception_logging()
        # Don't forget to remove stdout handler
        logging.getLogger().removeHandler(console_handler)
    else:
        logger = console_handler
    logger.setLevel(logcfg.get('loglevel', 'INFO'))


def run_setup(cfg):
    global SETUP_COMPLETED
    if SETUP_COMPLETED:
        return
    SETUP_COMPLETED = True
    try:
        # init log to stdout to log errors if we failed to parse config etc.
        console_handler = logutil.setup_logging_to_stdout()
        if cfg:
            config.load(cfg)
        else:
            config.load(DEFAULT_CONFIG_PATH)
        logcfg = config.get_value('log')
        setup_logging(logcfg, console_handler)
        if cfg:
            log.debug("using '{}' as config".format(cfg))
        else:
            log.debug('using default config')
    except SystemExit:
        raise
    except KeyboardInterrupt:
        log.info("exiting on user request")


def error(msg, exception=None):
    click.echo(click.style(msg, fg='red'))
    if exception is not None:
        traceback.print_exc(exception)


def success(msg):
    click.echo(click.style(msg, fg='green'))
