from __future__ import print_function

import errno
import logging
import socket
import sys

from sepelib.util import log as logutil
from sepelib.util import exc
from sepelib.core import config

from . import logutil as nlogutil


log = logging.getLogger('main')

DEFAULT_CONFIG_PATH = './cfg_default.yml'


def setup_exception_logging():
    """
    Sets up handler, which additionally logs errors in "main" handler to stderr.
    """
    h = logging.StreamHandler(stream=sys.stderr)
    h.setLevel(logging.ERROR)
    log.addHandler(h)


def setup_logging(logcfg, console_handler, console=False, fmt=None, filter_=None):
    """
    Setup service logs according to config
    """
    # disable requests module logging
    logging.getLogger('requests').setLevel(logging.DEBUG)

    if not sys.stdout.isatty() and not console:
        handler = logutil.create_handler_from_config(logcfg)
        if filter_ is not None:
            handler.addFilter(filter_)
        kwargs = {}
        if fmt is not None:
            kwargs['fmt'] = fmt
        logger = logutil.setup_logging_to_file(handler, redirect_stderr=False, redirect_stdout=False, **kwargs)
        setup_exception_logging()
        # Don't forget to remove stdout handler
        logging.getLogger().removeHandler(console_handler)
    else:
        logger = console_handler
        if filter_ is not None:
            logger.addFilter(filter_)
    # python2.6 doesn't automatically converts level name to int
    # so we do it manually
    loglevel = logging.getLevelName(logcfg.get('loglevel', 'INFO'))
    logger.setLevel(loglevel)


def setup_wsgi(args_parser, default_cfg=DEFAULT_CONFIG_PATH, argv=sys.argv[1:]):
    console_handler = logutil.setup_logging_to_stdout()
    # process command line arguments
    args = parse_args(args_parser, argv)
    # override some config settings
    if args.cfg:
        config.load(args.cfg, config_context=args.config_context)
    else:
        config.load(default_cfg, config_context=args.config_context)

    logcfg = config.get_value('log')
    filepath = logcfg.get('filepath')
    if filepath:
        logcfg['filepath'] = nlogutil.add_logfile_prefix(filepath, 'wsgi')
    setup_logging(logcfg, console_handler, console=args.console)
    # log what config we're using
    # couldn't do that earlier
    # because log has been just set up
    if args.cfg:
        log.info("using '{0}' as config".format(args.cfg))
    else:
        log.info("using hardwired config")


def parse_args(args_parser, argv):
    config.augment_args_parser(args_parser)
    return args_parser.parse_args(argv)


def get_config_context(args, env_prefix=None):
    if env_prefix is None:
        return args.config_context

    env_context = config.get_context_from_env(prefix=env_prefix)
    env_count = len(env_context)
    env_context.update(args.config_context)
    if env_count + len(args.config_context) != len(env_context):
        raise ValueError('Config env context clashes with input args context')
    return env_context


def make_instance_id(app_name):
    rv = '{}@{}'.format(app_name, socket.gethostname())
    port = config.get_value('web.http.port', default=None)
    if port is not None:
        rv += ':{}'.format(port)
    return rv


def setup_logging_to_stdout(fmt=None):
    console_handler = logutil.setup_logging_to_stdout()

    if fmt:
        # romanovich@: this is a temporary solution to avoid patching and releasing sepelib.
        # A better solution would be to add an "fmt" argument to `logutil.setup_logging_to_stdout`.
        formatter = logutil.utils.Formatter(fmt)
        console_handler.setFormatter(formatter)

    return console_handler


def main(args_parser, app_cls, env_prefix=None, default_cfg=DEFAULT_CONFIG_PATH, argv=sys.argv[1:],
         log_fmt=None, log_filter=None, config_loader=config.load):
    application = None
    try:
        # init log to stdout to log errors
        # if we failed to parse config etc.
        console_handler = setup_logging_to_stdout(fmt=log_fmt)

        # process command line arguments
        args = parse_args(args_parser, argv)
        config_context = get_config_context(args, env_prefix)
        # override some config settings
        if args.cfg:
            config_loader(args.cfg, config_context=config_context)
        else:
            config_loader(default_cfg, config_context=config_context)

        logcfg = config.get_value('log')
        filepath = logcfg.get('filepath')
        if filepath:
            logcfg['filepath'] = nlogutil.add_logfile_prefix(filepath, app_cls.name)
        setup_logging(logcfg, console_handler, console=args.console,
                      fmt=log_fmt, filter_=log_filter)
        # log what config we're using
        # couldn't do that earlier
        # because log has been just set up
        if args.cfg:
            log.info("using '{0}' as config".format(args.cfg))
        else:
            log.info("using hardwired config")
        # create and run mastermind
        application = app_cls(make_instance_id(app_cls.name))
        application.run()
    # some handy exception handling
    # to show nicely formatted and detailed enough
    # error messages to user
    except SystemExit:
        raise
    except KeyboardInterrupt:
        log.info("exiting on user request")
        exit_code = 0
    except socket.error as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on socket error: code='{}' msg='{}'".format(errorcode, e.strerror))
        exit_code = 1
    # IOError has 'filename' attribute so we handle it
    # separately from Exception
    except IOError as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on io error: file='{}' code='{}' msg='{}'".format(
            e.filename, errorcode, e.strerror))
        exit_code = 1
    except Exception as e:
        # BaseException.message is deprecated so we manually
        # construct nice looking (hopefully) string to log
        msg = exc.format_exc("exiting on error", e)
        log.exception(msg)
        # ...and again in case logging was misconfigured during start and does not actually log anything:
        print(msg, file=sys.stderr)
        exit_code = 1
    else:
        exit_code = 0
    finally:
        if application is not None:
            application.stop()
    raise SystemExit(exit_code)
