from __future__ import unicode_literals

import logging

import gevent
import gevent.event

from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent import geventutil


class CtlManager(greenthread.GreenThread):
    """
    Manages controllers.

    I.e. starts/stops controllers when corresponding entities arrive/depart to/from storage.
    """

    def __init__(self, storage, factory):
        """
        :type storage: infra.swatlib.storage.interfaces.IWatchableCachingStorage
        :type factory: infra.swatlib.ctl.interfaces.ICtlFactory
        """
        super(CtlManager, self).__init__()
        self.storage = storage
        self.gs = {}  # List of active controller greenlets (id => greenlet)
        self.log = logging.getLogger('ctl-manager("{}")'.format(factory.get_name()))
        self.factory = factory
        self._stopped = gevent.event.Event()

    def _restart_controller(self, ctl_id):
        if self._stopped.is_set():
            return
        gevent.sleep(5)
        self.factory.start_controller(ctl_id)

    def _start_controller(self, ctl_id):
        self.log.info('Starting controller for "{}"...'.format(ctl_id))
        ctl = self.factory.start_controller(ctl_id)
        ctl.link_exception(lambda _: self._restart_controller(ctl_id))
        self.gs[ctl_id] = ctl
        self.log.info('Started controller for "{}"'.format(ctl_id))
        gevent.sleep(0.01)

    def _stop_controller(self, ctl_id):
        g = self.gs.pop(ctl_id, None)
        if g is None:
            return
        self.log.info('Stopping controller for "{}"...'.format(ctl_id))
        geventutil.force_kill_greenlet(g)
        self.log.info('Stopped controller for "{}"'.format(ctl_id))

    def run(self):
        try:
            w = self.storage.watch_list()
            while 1:
                added, removed = next(w)
                for ctl_id in added:
                    self._start_controller(ctl_id)
                for ctl_id in removed:
                    self._stop_controller(ctl_id)
        finally:
            self._stopped.set()
            self.log.info("Stopping all controllers...")
            for g in self.gs.values():
                geventutil.force_kill_greenlet(g)
            self.gs.clear()
            self.log.info("Stopped all controllers")
