from __future__ import unicode_literals
import functools
import sys
from logging import getLogger

import flask
import gevent

from sepelib.core import config
from infra.swatlib import metrics

DEFAULT_TIMEOUT = 20

TIMEOUT_HEADER = 'X-Backend-Timeout'

ENVIRON_TIMEOUT_HEADER = 'HTTP_X_BACKEND_TIMEOUT'

TIMEOUT_RESPECTFUL_HTTP_METHODS = ['GET']

TIMEOUT_RESPECTFUL_RPC_HTTP_METHODS = ['GET', 'OPTIONS', 'POST', 'PUT', 'PATCH']  # all excepting DELETE

TIMEOUT_ERROR_MESSAGE_TPL = '{}: handler execution aborted - it took more than {} seconds'

TIMEOUT_LOG_MESSAGE_TPL = '%s: handler execution aborted - it took more than %d seconds'

METRICS_REGISTRY = metrics.ROOT_REGISTRY.path('handler-timeout')

timeout_exceeded_handlers = METRICS_REGISTRY.get_counter('timeout-exceeded-handlers')

log = getLogger(__name__)


class HandlerTimeout(Exception):
    """
    Handler executed too long time.
    """
    pass


def _is_method_timeout_respectful(method, respectful_methods=None):
    if respectful_methods is not None:
        return method in respectful_methods
    return method in TIMEOUT_RESPECTFUL_HTTP_METHODS


def _get_timeout(header_timeout, timeout=None):
    if header_timeout is not None:
        try:
            return float(header_timeout)
        except Exception:
            # suppress exception to prevent 500 Internal Error
            pass
    if timeout is not None:
        return timeout
    return config.get_value('web.handler_timeout', DEFAULT_TIMEOUT)


def abort_on_timeout(timeout=None, methods=None):
    """
    :type timeout: int
    :type methods: list[str]
    """
    def decorator(handler):
        @functools.wraps(handler)
        def wrapper(*args, **kwargs):
            if not flask.has_request_context():
                return handler(*args, **kwargs)
            if not _is_method_timeout_respectful(flask.request.method, methods):
                return handler(*args, **kwargs)
            header_timeout = flask.request.headers.get(TIMEOUT_HEADER)
            t = _get_timeout(header_timeout, timeout)
            try:
                with gevent.Timeout(t):
                    return handler(*args, **kwargs)
            except gevent.Timeout:
                timeout_exceeded_handlers.inc()
                # gevent.Timeout bases from BaseException not from
                # Exception. That's why flask's errorhandler ingnores it.
                # So we raise HandlerTimeout which bases from Exception.
                log.exception(TIMEOUT_LOG_MESSAGE_TPL, flask.request.path, t)
                message = TIMEOUT_ERROR_MESSAGE_TPL.format(flask.request.path, t)
                raise HandlerTimeout(message)
        return wrapper
    return decorator


def abort_rpc_on_timeout(timeout=None):
    """
    :type timeout: int
    """
    decorator = abort_on_timeout(timeout=timeout, methods=TIMEOUT_RESPECTFUL_RPC_HTTP_METHODS)
    return decorator


class TimeoutAborter(object):
    TIMEOUT_EXCEEDED_STATUS = b'500 Internal Server Error'
    TIMEOUT_EXCEEDED_HEADERS = [
        (b'Content-Type', b'application/json')
    ]
    TIMEOUT_EXCEEDED_BODY = (
        b'{'
        b'"error": "InternalError", '
        b'"message": "Global timeout exceeded"'
        b'}',
    )

    def __init__(self, app, headers=None, timeout=None, methods=None):
        self.app = app
        self.headers = self.TIMEOUT_EXCEEDED_HEADERS[:]
        if headers:
            self.headers.extend(headers)
        self.timeout = timeout
        self.methods = methods

    def __call__(self, environ, start_response):
        method = environ.get('REQUEST_METHOD')
        if not _is_method_timeout_respectful(method):
            return self.app(environ, start_response)
        header_timeout = environ.get(ENVIRON_TIMEOUT_HEADER)
        t = _get_timeout(header_timeout, self.timeout)
        try:
            with gevent.Timeout(t):
                return self.app(environ, start_response)
        except gevent.Timeout:
            timeout_exceeded_handlers.inc()
            log.exception(TIMEOUT_LOG_MESSAGE_TPL, environ.get('PATH_INFO'), t)
            # It is possible, that flask app called "start_response" earlier when processing request
            # and already wrote request status code and headers to client. We cannot revert
            # this action, instead of it we do the following:
            # * Call start_response with exc_info call arg
            # * In this case start_response checks if headers already sent
            #     * If headers were not sent, than it will just send our status code 500
            #     * If headers sent (and we already sent 200 OK to client), than it's simple closes the socket
            # For details see: https://www.python.org/dev/peps/pep-3333/#error-handling
            start_response(self.TIMEOUT_EXCEEDED_STATUS,
                           self.headers,
                           exc_info=sys.exc_info())
            return self.TIMEOUT_EXCEEDED_BODY
